﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Sonic.Services;
using Sonic.Services.Controllers;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using Sonic.Web.Service;

namespace Sonic.Service.PL.Controllers.Echo
{
    [Route("api/[controller]/[action]")]
    [ApiController]
    public class EchoCustomerController : BaseController
    {
        private readonly CustomerService _service;
        private readonly IRequestRepository _requestRepository;
        private IHostingEnvironment _hostingEnvironment;

        public EchoCustomerController(IHostingEnvironment hostingEnvironment, IRequestRepository requestRepository) : base(requestRepository)
        {
            _requestRepository = requestRepository;
            _service = new CustomerService(_requestRepository);
            _hostingEnvironment = hostingEnvironment;
        }
        [HttpPost]
        [ActionName("EchoSignup")]
        public JsonResult EchoSignup([FromBody] CustomerModel customer)
        {
            return SaveNewCustomer(customer);
        }
        [HttpPost]
        [ActionName("ValidateNewCustomerSignup")]
        public JsonResult ValidateNewCustomerSignup([FromBody] EchoCustomerModel customer)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, customer.LanguageId));
                }
                bool isUniquePhone = false, isUniqueUsername = false, isUniqueEmail = false;
                GlobalErrors result = _service.ValidateEchoCustomerToSave(customer, ref isUniquePhone, ref isUniqueUsername);
                if(result == GlobalErrors.Success)
                {
                    if (!isUniquePhone)
                    {
                        return JSonResultResponse.GetSuccessJSon(null, SignupValidationErrors.InvalidPhone.GetHashCode(), "");
                    }
                    if (!isUniqueUsername)
                    {
                        return JSonResultResponse.GetSuccessJSon(null, SignupValidationErrors.InvalidUsername.GetHashCode(), "");
                    }
                    if (!isUniqueEmail)
                    {
                        return JSonResultResponse.GetSuccessJSon(null, SignupValidationErrors.InvalidEmail.GetHashCode(), "");
                    }
                    return JSonResultResponse.GetSuccessJSon(null, SignupValidationErrors.None.GetHashCode(), "");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon();
                }

            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("SaveNewCustomer")]
        public JsonResult SaveNewCustomer([FromBody] CustomerModel customer)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                 if (customer == null)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                string errorMessage = string.Empty;

                //bool isValid = true;
                //GlobalErrors result = _customerService.ValidateCustomerToSave(customerObj, ref isValid, ref errorMessage);




                result = _service.InsertNewCustomer(customer, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_While_Saving : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("UploadCustomerIdentification")]
        public JsonResult UploadCustomerIdentification([FromForm] Collection<IFormFile> formData)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                byte[] buff = null;
                string baseSonicWebPath = LocalUtilities.SonicWebPath;
                if (!String.IsNullOrEmpty(baseSonicWebPath) && Directory.Exists(baseSonicWebPath))
                {
                    for (int i = 0; i < Request.Form.Files.Count; i++)
                    {
                        string[] fileData = Request.Form.Files[i].FileName.Split('|');
                        string customerOutletFolderFullPath = Path.Combine(baseSonicWebPath, "CustomerAttachments", "CustomerAttachments", fileData[0], fileData[1]);
                        if (!Directory.Exists(customerOutletFolderFullPath))
                        {
                            Directory.CreateDirectory(customerOutletFolderFullPath);
                        }
                        var file = Request.Form.Files[i];
                        if (file.Length > 0)
                        {
                            string fileName = fileData[2];
                            string fullPath = Path.Combine(customerOutletFolderFullPath, fileName);
                            using (var stream = new FileStream(fullPath, FileMode.Create))
                            {
                                file.CopyTo(stream);
                            }
                            buff = System.IO.File.ReadAllBytes(fullPath);
                        }
                    }
                    return JSonResultResponse.GetSuccessJSon(string.Empty);
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon();
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetCustomerSOA")]
        public JsonResult GetCustomerSOA()
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<SOATransactionModel> list = new List<SOATransactionModel>();
                result = _service.GetCustomerSOA(ref list, true);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(list, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("SetCustomerInactive")]
        public JsonResult SetCustomerInactive([FromBody] OperatorModel operatorModel)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                string msg = string.Empty;
                CustomerOutletModel outlet = new CustomerOutletModel();
                result = _service.UpdateInactiveCustomer(operatorModel.CustomerId, false);
               
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(outlet, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}