﻿using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Mvc;
using Microsoft.IdentityModel.Tokens;
using Newtonsoft.Json;
using Sonic.Services;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.IO;
using System.Reflection;
using System.Security.Claims;
using System.Linq;
using System.Text;

namespace Sonic.Service.PL.Controllers.Echo
{
    [Route("api/[controller]/[action]")]
    [ApiController]
    public class EchoLoginController : ControllerBase
    {
        [HttpPost]
        [ActionName("Login")]
        public JsonResult Login([FromBody] EchoOperatorLoginModel operatorData, int languageId)
        {
            string errorMessage = string.Empty;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, languageId));
                }
                EchoOperatorModel operatorModel = null;
                CustomerAppSettingsModel customerAppSettingsModel = null;
                List<OrganizationModel> organizationsList = new List<OrganizationModel>();
                GlobalErrors result = LoginService.EchoLogin(operatorData, ref operatorModel, ref errorMessage, ref customerAppSettingsModel, languageId, ref organizationsList);
                if(result == GlobalErrors.Success)
                {
                    result = GlobalErrors.EchoCustomerMultiAccount;
                }
                if ((result == GlobalErrors.Success || result == GlobalErrors.EchoCustomerMultiAccount) && operatorModel != null)
                {
                    List<string> bannersImages = new List<string>();
                    string tokenString = GenerateToken(operatorModel);
                    if (string.IsNullOrEmpty(tokenString.Trim()))
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Generating_Token_Key, languageId));
                    }
                    string bannerPath = "wwwroot\\img\\echo\\banner";
                    if (System.IO.Directory.Exists(bannerPath))
                    {
                        var list = Directory.GetFiles(Path.Combine(Directory.GetCurrentDirectory(), bannerPath));
                        if (list != null && list.Length > 0)
                        {
                            list.ToList().ForEach(a =>
                            {
                                bannersImages.Add(Path.GetFileName(a));
                            });
                        }
                    }
                    var data = new
                    {
                        key = tokenString,
                        customerAppSettingsModel = EchoService.CustomerAppSettings,
                        operatorModel = operatorModel,
                        bannersImages = bannersImages,
                        organizationsList = organizationsList
                    };
                    return JSonResultResponse.GetSuccessJSon(data, result.GetHashCode(), string.Empty);
                }
                else if (result == GlobalErrors.VersionNotMatch)
                {
                    return JSonResultResponse.GetSuccessJSon(null, GlobalErrors.VersionNotMatch.GetHashCode(), string.Empty);
                }
                else
                {
                    if (string.IsNullOrEmpty(errorMessage))
                        errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Invalid_UserName_Password, languageId);
                    return JSonResultResponse.GetErrorJSon(errorMessage);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        private string GenerateToken(EchoOperatorModel operatorModel)
        {
            try
            {
                string securityKey = LocalUtilities.UsedSecurityKey;
                //symmetric security key
                var symmetricSecurityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(securityKey));

                //signing credentials
                var signingCredentials = new SigningCredentials(symmetricSecurityKey, SecurityAlgorithms.HmacSha256Signature);

                //add claims
                var claims = new List<Claim>();
                claims.Add(new Claim(ClaimTypes.Name, JsonConvert.SerializeObject(operatorModel)));
                claims.Add(new Claim(ClaimTypes.Role, LoginRoleType.ECHO.GetHashCode().ToString())); 
                claims.Add(new Claim(ClaimTypes.Sid, operatorModel.CustomerId + "-" + operatorModel.OutletId));
                //create token
                var token = new JwtSecurityToken(
                        issuer: LocalUtilities.UsedIssuer,
                        audience: LocalUtilities.UsedAudience,
                        expires: DateTime.Now.AddHours(LocalUtilities.UsedTimeout),
                        signingCredentials: signingCredentials
                        , claims: claims
                    );
                return new JwtSecurityTokenHandler().WriteToken(token);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return string.Empty;
            }
        }

        [HttpPost]
        [ActionName("GetPublishConfigValue")]
        public JsonResult GetPublishConfigValue()
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                dynamic keyValue = false;
                LoginService.GetSettingsWhenPublishingIOSConfig(ref keyValue);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(keyValue, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, 1));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, 1));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}