﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;

namespace Sonic.Services.Controllers.Order
{
    [Route("api/[controller]/[action]")]
    [ApiController]
    [Authorize]
    public class OrderController : BaseController
    {
        private readonly OrderService _orderService;
        private readonly IRequestRepository _requestRepository;
        public OrderController(IRequestRepository requestRepository) : base(requestRepository)
        {
            _requestRepository = requestRepository;
            _orderService = new OrderService(_requestRepository);
        }

        [HttpGet]
        [ActionName("GetOrderStatus")]
        public JsonResult GetOrderStatus(string OrderNumber)
        {
            try
            {
                // check Order
                if (string.IsNullOrEmpty(OrderNumber))
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
                OrderStatusAPIModel Status = new OrderStatusAPIModel();

                GlobalErrors result = _orderService.GetOrderStatus(OrderNumber, ref Status);

                if (result == GlobalErrors.Success)
                {
                    if (Status != null)
                    {
                        return JSonResultResponse.GetSuccessJSon(Status, "Success");
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon("No Data Found");
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetOrdersHistory")]
        public JsonResult GetOrdersHistory(string SiteCode, string From_Date, string To_Date)
        {
            try
            {
                //1. check customer code
                if (string.IsNullOrEmpty(SiteCode))
                {
                    return JSonResultResponse.GetErrorJSon();
                }
                List<OrderAPIModel> OrderList = new List<OrderAPIModel>();
                GlobalErrors result = _orderService.GetOrdersHistory(SiteCode, From_Date, To_Date, ref OrderList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(OrderList, "Success");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("SendOrder")]
        public JsonResult SendOrder([FromBody]SendOrderAPIModel Order)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                result = _orderService.SaveOrderFromAPI(Order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(Order, "Success");
                }
                else if (result == GlobalErrors.AllSerialsInvalid)//Customer Not Valid
                {
                    return JSonResultResponse.GetErrorJSon("Customer Not Valid");
                }
                else if(result == GlobalErrors.NotEnoughQuantity)//no details available
                {
                    return JSonResultResponse.GetErrorJSon("There is no items available for the order");
                }
                else if(result == GlobalErrors.DataAlreadyExists)//Order Id already used
                {
                    return JSonResultResponse.GetErrorJSon("Order Id Already Exists");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        
        [HttpGet]
        [ActionName("GetUpdatedProducts")]
        public JsonResult GetUpdatedProducts(string PriceListCode)
        {
            try
            {
                // check Order
                if (string.IsNullOrEmpty(PriceListCode))
                {
                    return JSonResultResponse.GetErrorJSon();
                }
                ProductsAPIModel Products = new ProductsAPIModel();

                GlobalErrors result = _orderService.GetUpdatedProducts(PriceListCode, ref Products);

                if (result == GlobalErrors.Success)
                {
                    if (Products != null && Products.TotalItems > 0)
                    {
                        return JSonResultResponse.GetSuccessJSon(Products, "Success");
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon("No Data Found");
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetSiteUpdatedProducts")]
        public JsonResult GetSiteUpdatedProducts(string SiteCode)
        {
            try
            {
                // check Order
                if (string.IsNullOrEmpty(SiteCode))
                {
                    return JSonResultResponse.GetErrorJSon();
                }
                ProductsAPIModel Products = new ProductsAPIModel();

                GlobalErrors result = _orderService.GetSiteUpdatedProducts(SiteCode, ref Products);

                if (result == GlobalErrors.Success)
                {
                    if (Products != null && Products.TotalItems > 0)
                    {
                        return JSonResultResponse.GetSuccessJSon(Products, "Success");
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon("No Data Found");
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Error");
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}
