using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using InCube.Security;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.FileProviders;
using Microsoft.Extensions.Hosting;
using Microsoft.IdentityModel.Tokens;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Service;
using static Sonic.Web.Model.CoreDataBaseConstants;

namespace Sonic.Service.PL
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            Helper.GetTokenKeys();
            services.AddControllersWithViews();
            string securityKey = LocalUtilities.UsedSecurityKey;
            var symmetricSecurityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(securityKey));

            services.AddSession(options =>
            {
                options.IdleTimeout = TimeSpan.FromHours(LocalUtilities.UsedTimeout);//You can set Time
            });
            services.AddCors(c => {
                c.AddPolicy("AllowOrigin", options => options.AllowAnyOrigin());
            });
            services.AddMvc(option => option.EnableEndpointRouting = false);
            services.AddAuthentication(JwtBearerDefaults.AuthenticationScheme)
                .AddJwtBearer(options =>
                {
                    options.TokenValidationParameters = new TokenValidationParameters
                    {
                        //what to validate
                        ValidateIssuer = true,
                        ValidateAudience = true,
                        ValidateIssuerSigningKey = true,
                        //setup validate data
                        ValidIssuer = LocalUtilities.UsedIssuer,
                        ValidAudience = LocalUtilities.UsedAudience,
                        IssuerSigningKey = symmetricSecurityKey
                    };
                });
            services.AddHttpContextAccessor();
            services.AddScoped<IRequestRepository, RequestRepository>();
            // Read Configuration
            LocalUtilities.WebApplication = WebApplication.Echo;
            SystemConfigService.ReadConfiguration(Directory.GetCurrentDirectory());
            // Read Languages
            SettingService.WriteJSONs();
            services.AddHostedService<BackgroundService>();
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
        {
            string rootPath = env.ContentRootPath;
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }
            else
            {
                app.UseExceptionHandler("/Home/Error");
            }
            app.UseCors(builder => builder
              .AllowAnyHeader()
              .AllowAnyMethod()
              .SetIsOriginAllowed((host) => true)
              .AllowCredentials()
          );
            app.UseStaticFiles();

            app.UseRouting();
            app.UseAuthentication();
            app.UseStaticFiles();
            app.UseStaticFiles(new StaticFileOptions()
            {
                FileProvider = new PhysicalFileProvider(Path.Combine(Directory.GetCurrentDirectory(), "wwwroot")),
                RequestPath = new PathString("/wwwroot")
            });
            app.UseSession();
            app.UseAuthorization();
            app.UseEndpoints(endpoints =>
            {
                endpoints.MapControllerRoute(
                    name: "default",
                    pattern: "{controller=Home}/{action=Index}/{id?}");
            });
            LoginService.LoadDevicesSerialNumber();
            //Create Banners Folder
            string echoFolderName = "wwwroot\\Echo";
            string echoPath = Path.Combine(rootPath, echoFolderName);
            if (!Directory.Exists(echoPath))
                Directory.CreateDirectory(echoPath);

            string bannerFoldername = "wwwroot\\Echo\\banners";
            string bannersPath = Path.Combine(rootPath, bannerFoldername);
            if (!Directory.Exists(bannersPath))
                Directory.CreateDirectory(bannersPath);
            EchoService.FillEchoGuestOpratorDataAndSettings();
        }
    }
}
