﻿using OfficeOpenXml;
using OfficeOpenXml.DataValidation;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Model.General;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Net;
using System.Reflection;
using System.Text;
using System.Xml;

namespace Sonic.Web.Core
{
    public class ImportManager
    {
        private readonly IRequestRepository _requestRepository;
        private readonly ItemManager _itemManager;
        private readonly SettingManager _settingManager;
        public ImportManager(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _itemManager = new ItemManager(requestRepository);
            _settingManager = new SettingManager(requestRepository);
        }

        #region [DECLARATIONS]

        #endregion [DECLARATIONS]

        public static GlobalErrors DownloadFile(ImportDataTypes dataType, ref string contentType, ref string fileName, ref MemoryStream memory)
        {
            GlobalErrors err = GlobalErrors.NotInitialized;
            try
            {
                fileName = GetFileName(dataType);
                if (string.IsNullOrEmpty(fileName))
                {
                    return GlobalErrors.Error;
                }
                var path = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", fileName);
                memory = new MemoryStream();
                using (var stream = new FileStream(path, FileMode.Open))
                {
                    stream.CopyTo(memory);
                }
                memory.Position = 0;
                contentType = GetContentType(path);
                err = GlobalErrors.Success;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                err = GlobalErrors.Error;
            }
            return err;
        }
        public static GlobalErrors GetAllLanguages(ref List<LanguageModel> languages)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<LanguageModel> dBHelper = null;
            try
            {
                dBHelper = new DBHelper<LanguageModel>();
                string query = string.Format(@"select Languages.*,Languages.Active IsChecked,Languages.Active OldStatus from  Languages  order by LanguageID");
                result = dBHelper.GetQueryList(query, ref languages);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        private static string GetContentType(string path)
        {
            var types = GetMimeTypes();
            var ext = Path.GetExtension(path).ToLowerInvariant();
            return types[ext];
        }
        private static string GetFileName(ImportDataTypes type)
        {
            string fileName = string.Empty;
            switch (type)
            {
                case ImportDataTypes.Items:
                    fileName = "ItemsTemplate.xlsx";
                    break;
                case ImportDataTypes.Employees:
                    fileName = "EmployeesTemplate.xlsx";
                    break;
                case ImportDataTypes.Customers:
                    fileName = "CustomersTemplate.xlsx";
                    break;
                case ImportDataTypes.Serials:
                    fileName = "SerialsTemplate.xlsx";
                    break;
                case ImportDataTypes.Serials_InputVoucher:
                    fileName = "SerialsTemplate-InputVoucher.xlsx";
                    break;
                case ImportDataTypes.DisplayUnit:
                    fileName = "DisplayUnitTemplate.xlsx";
                    break;
                case ImportDataTypes.BackToStoreOffload:
                    fileName = "SerialsTemplate-BackToStoreOffload.xlsx";
                    break;
                case ImportDataTypes.BackToStoreReset:
                    fileName = "SerialsTemplate.xlsx";
                    break;
                case ImportDataTypes.UnlockedSerial:
                    fileName = "SerialsTemplate.xlsx";
                    break;
                case ImportDataTypes.ItemsPricesWithQtyRange:
                    fileName = "ItemsPricesTemplateWithQtyRange-Template.xlsx";
                    break;
                case ImportDataTypes.ItemsPricesWithoutQtyRange:
                    fileName = "ItemsPricesTemplateWithoutQtyRangeTemplate.xlsx";
                    break;
                case ImportDataTypes.SalesIfFOC:
                    fileName = "SerialsTemplate-Invoice.xlsx";
                    break;
                case ImportDataTypes.KPIs:
                    fileName = "KPIsTemplate.xlsx";
                    break;
                case ImportDataTypes.CommissionCalc:
                    fileName = "CommissionCalc-Template.xlsx";
                    break;
                case ImportDataTypes.ShopContractTargets:
                    fileName = "ShopContractsTemplate.xlsx";
                    break;
                case ImportDataTypes.MustSellItems:
                    fileName = "MustSellItemsTemplate.xlsx";
                    break;
                case ImportDataTypes.KPIScorecard:
                    fileName = "KPIScorecardTemplate.xlsx";
                    break;
                case ImportDataTypes.CustomerPromotion:
                case ImportDataTypes.CustomerOutletPromotion:
                    fileName = "CustomerPromotionTemplate.xlsx";
                    break;
                case ImportDataTypes.ExcludeCustomerPromotion:
                case ImportDataTypes.ExcludeCustomerOutletPromotion:
                    fileName = "ExcludeCustomerPromotionTemplate.xlsx";
                    break;
                case ImportDataTypes.TeleRepPlan:
                    fileName = "TeleRepPlanTemplate.xlsx";
                    break;
                case ImportDataTypes.CommissionSkipPlan:
                    fileName = "CommissionSkipPlanTemplate.xlsx";
                    break;
                case ImportDataTypes.ContactCustomers:
                    fileName = "ContactCustomersTemplate.xlsx";
                    break;
                default:
                    fileName = string.Empty;
                    break;
            }
            return fileName;
        }
        private static Dictionary<string, string> GetMimeTypes()
        {
            return new Dictionary<string, string>
            {
                {".txt", "text/plain"},
                {".pdf", "application/pdf"},
                {".doc", "application/vnd.ms-word"},
                {".docx", "application/vnd.ms-word"},
                {".xls", "application/vnd.ms-excel"},
                {".xlsx", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"},
                {".png", "image/png"},
                {".jpg", "image/jpeg"},
                {".jpeg", "image/jpeg"},
                {".gif", "image/gif"},
                {".csv", "text/csv"}
            };

        }
        public static GlobalErrors GetTableNameAndProcedureName(ImportDataTypes dataType, ref string tableName, ref string storedProcedureName, int userId, string sessionId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                switch (dataType)
                {
                    case ImportDataTypes.Items:
                        tableName = "ImportItemsTemplate";
                        storedProcedureName = "sp_Web_ImportItems";
                        break;
                    case ImportDataTypes.Employees:
                        tableName = "ImportEmployeesTemplate";
                        storedProcedureName = "sp_Web_ImportEmployees";
                        break;
                    case ImportDataTypes.Customers:
                        tableName = "ImportCustomersTemplate";
                        storedProcedureName = "sp_Web_ImportCustomers";
                        break;
                    case ImportDataTypes.DisplayUnit:
                        tableName = "ImportDisplayUnitTemplate";
                        storedProcedureName = "sp_Web_ImportDisplayUnit";
                        break;
                    case ImportDataTypes.UnlockedSerial:
                        tableName = "ImportUnlockSerials";
                        storedProcedureName = "SP_Web_ImportUnlockSerials";
                        break;
                    case ImportDataTypes.ItemsPricesWithoutQtyRange:
                        tableName = "ImportItemsPricesTemplate";
                        break;
                    case ImportDataTypes.ItemsPricesWithQtyRange:
                        tableName = "ImportItemsPricesTemplate";
                        break;
                    case ImportDataTypes.KPIs:
                        tableName = "ImportKPIsTemplate";
                        storedProcedureName = "SP_Web_ImportKPIs";
                        break;
                    case ImportDataTypes.CommissionCalc:
                    case ImportDataTypes.CommissionSkipPlan:
                        tableName = "ImportCommissionCalcTemplate";
                        break;
                    case ImportDataTypes.MustSellItems:
                        tableName = "ImportMustSellItemsTemplate";
                        break;
                    case ImportDataTypes.KPIScorecard:
                        tableName = "ImportScorecardTemplate";
                        storedProcedureName = "SP_Web_ImportKPIScorecard";
                        break;
                    case ImportDataTypes.CustomerPromotion:
                    case ImportDataTypes.CustomerOutletPromotion:
                        tableName = "ImportCustomerPromotionTemplate";
                        break;
                    case ImportDataTypes.ExcludeCustomerPromotion:
                    case ImportDataTypes.ExcludeCustomerOutletPromotion:
                        tableName = "ImportExcludeCustomerPromotionTemplate";
                        break;
                    case ImportDataTypes.TeleRepPlan:
                        tableName = "ImportTeleRepPlanTemplate";
                        storedProcedureName = "SP_CRM_ImportTeleRepPlan";
                        break;
                    case ImportDataTypes.AppData:
                        tableName = $"ImportAppDataTemplate_{userId}_{sessionId.Replace("-","_")}";
                        storedProcedureName = "SP_Web_ImportAppData";
                        break;
                    case ImportDataTypes.ContactCustomers:
                        tableName = "ImportContactCustomersTemplate";
                        storedProcedureName = "sp_Web_ImportContactCustomers";
                        break;
                    default:
                        tableName = "";
                        storedProcedureName = "";
                        break;
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public static GlobalErrors GetExcelSheetDataAsDataTable(FileInfo file, int sheetNo, ref DataTable dt)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dt = new DataTable();
                string columnName = string.Empty;
                string rowValue = string.Empty;
                DataRow dr = null;
                int nullValuesCount = 0;
                using (ExcelPackage package = new ExcelPackage(file))
                {
                    if (package != null && package.Workbook != null && package.Workbook.Worksheets != null && package.Workbook.Worksheets.Count > 0)
                    {
                        ExcelWorksheet worksheet = package.Workbook.Worksheets[sheetNo];
                        if (worksheet != null && worksheet.Dimension != null)
                        {
                            int rowCount = worksheet.Dimension.Rows;
                            int colCount = worksheet.Dimension.Columns;
                            object[,] valueArray = worksheet.Cells.GetValue<object[,]>();
                            // columns
                            for (int col = 0; col < colCount; col++)
                            {
                                columnName = string.Empty;
                                if (valueArray[0, col] != null && !string.IsNullOrEmpty(valueArray[0, col].ToString()))
                                    columnName = valueArray[0, col].ToString();
                                dt.Columns.Add(columnName);
                            }
                            // rows
                            for (int row = 1; row < rowCount; row++)
                            {
                                nullValuesCount = 0;
                                dr = dt.NewRow();
                                for (int col = 0; col < colCount; col++)
                                {
                                    rowValue = string.Empty;
                                    if (valueArray[row, col] != null && !string.IsNullOrEmpty(valueArray[row, col].ToString()))
                                    {
                                        rowValue = valueArray[row, col].ToString();
                                    }
                                    else
                                    {
                                        nullValuesCount += 1;
                                    }
                                    dr[col] = rowValue;
                                }
                                if (nullValuesCount != colCount)
                                    dt.Rows.Add(dr);
                            }
                        }
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetXMLTextAsDataTable(string path, ref DataTable expectedExcelSheetDataTable)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                XmlReaderSettings settings = new XmlReaderSettings();
                settings.IgnoreWhitespace = true;
                ImportXMLModel importModel = new ImportXMLModel();
                List<ImportXMLModel> importModelList = new List<ImportXMLModel>();
                string serialElement = string.Empty;
                List<string> serialList = new List<string>();
                string batchNoElement = string.Empty;
                string expiryDateElement = string.Empty;
                string expiryDate = string.Empty;
                string itemCode = string.Empty;
                string uom = string.Empty;
                string gtin = string.Empty;
                string currentGtin = string.Empty;
                string serPart = string.Empty;
                string fullSerial = string.Empty;
                List<string> gtinList = new List<string>();
                string gtinString = string.Empty;
                string templateIDs = string.Empty;
                List<int> templateIDsList = new List<int>();
                List<ImportItemDetailsModel> itemsDetails = new List<ImportItemDetailsModel>();
                List<TemplateFieldModel> templateFields = new List<TemplateFieldModel>();
                DataRow Row = null;
                expectedExcelSheetDataTable = new DataTable();
                List<TemplateFieldModel> currentItemtemplateField = new List<TemplateFieldModel>();
                ImportItemDetailsModel importItemDetailsModel = new ImportItemDetailsModel();
                XmlDocument doc = new XmlDocument();
                doc.Load(path);
                XmlNodeList aNodes = doc.GetElementsByTagName("childEPCs");
                XmlNodeList childNode = aNodes[0].ChildNodes;
                string sgtin = string.Empty;
                List<string> serialPartList = new List<string>();
                string [] gtinSerial = null;
                string gtinPart = string.Empty;
                string serialPart = string.Empty;
                string lotNumber = string.Empty;
                int zeroCount = 0;
                lotNumber = doc.GetElementsByTagName("cbvmda:lotNumber")[0].InnerText;
                string expirationDate = string.Empty;
                expirationDate= doc.GetElementsByTagName("cbvmda:itemExpirationDate")[0].InnerText;
                string materialCode = string.Empty;
                materialCode = doc.GetElementsByTagName("tl:internalMaterialCode")[0].InnerText;
                string sscc = string.Empty;
                sscc = doc.GetElementsByTagName("parentID")[0].InnerText;
                sscc = sscc.Replace("urn:epc:id:sscc:", "");
                sscc = sscc.Replace(".", "");
                zeroCount = 18 - sscc.Length;
                for (int i = 0; i < zeroCount; i++)
                {
                    sscc = "0" + sscc;
                }
                sscc = "00" + sscc;
                foreach (XmlNode serial in childNode)
                {
                    sgtin = serial.InnerText.Replace("urn:epc:id:sgtin:", "");
                    gtinSerial = sgtin.Split('.');
                    if(gtinSerial != null && gtinSerial.Length > 0)
                    {
                        zeroCount = 0;
                        gtinPart = gtinSerial[0] + gtinSerial[1];
                        zeroCount = 14 - gtinPart.Length;
                        for(int i = 0; i < zeroCount; i++)
                        {
                            gtinPart = "0" + gtinPart;
                        }
                         serialPart = gtinSerial[2];
                        if (!serialPartList.Contains(serialPart))
                            serialPartList.Add(serialPart);
                    }
                }
                expiryDateElement = expirationDate;
                importModel = new ImportXMLModel();
                importModel.ItemCode = materialCode;
                importModel.BatchNo = lotNumber;
                importModel.ExpiryDate = expirationDate;
                importModel.SerialList = serialPartList;
                importModel.GTIN = gtinPart;
                if (!importModelList.Contains(importModel))
                    importModelList.Add(importModel);

                expectedExcelSheetDataTable.Columns.Add("ItemCode", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("UOM", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("FromSerial", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("ToSerial", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("ParentSerial", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("BatchNo", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("ExpiryDate", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("SSCCBarcode", typeof(string));
                expectedExcelSheetDataTable.Columns.Add("ParentSSCCBarcode", typeof(string));
                foreach (ImportXMLModel importXML in importModelList)
                {
                    gtin = importXML.GTIN;
                    if (!gtinList.Contains(gtin))
                    {
                        gtinList.Add(gtin);
                        gtinString += "'" + gtin + "'" + ',';
                    }
                }
                if (!string.IsNullOrEmpty(gtinString))
                {
                    gtinString = gtinString.TrimEnd(',');
                    templateIDs = string.Empty;
                    templateIDsList = new List<int>();
                    foreach (ImportItemDetailsModel itemModel in itemsDetails)
                    {
                        if (!templateIDsList.Contains(itemModel.TemplateId))
                        {
                            templateIDsList.Add(itemModel.TemplateId);
                            templateIDs += itemModel.TemplateId.ToString() + ',';
                        }
                    }
                    if (!string.IsNullOrEmpty(templateIDs))
                    {
                        templateIDs = templateIDs.TrimEnd(',');
                        result = MasterDataManager.GetTemplatesFields(templateIDs, ref templateFields);
                    }
                }
                foreach (ImportXMLModel importXML in importModelList)
                {
                    itemCode = importXML.ItemCode;
                    batchNoElement = importXML.BatchNo;
                    expiryDateElement = importXML.ExpiryDate;
                    currentGtin = importXML.GTIN;
                    DateTime excelSheetDateValue;
                    if (!string.IsNullOrEmpty(expiryDateElement))
                    {
                        if (DateTime.TryParseExact(expiryDateElement, _requestRepository.Configurations.SerialTemplateDateFormat, System.Globalization.CultureInfo.InvariantCulture, System.Globalization.DateTimeStyles.None, out excelSheetDateValue))
                            expiryDate = excelSheetDateValue.ToString();
                    }
                    excelSheetDateValue = Convert.ToDateTime(expiryDateElement);
                    expiryDate = excelSheetDateValue.ToString(_requestRepository.Configurations.SerialTemplateDateFormat);
                    importItemDetailsModel = itemsDetails.Where(available => available.GTIN == currentGtin).FirstOrDefault();
                    if (importItemDetailsModel == null)
                        importItemDetailsModel = new ImportItemDetailsModel();
                    currentItemtemplateField = templateFields.Where(available => available.TemplateId == importItemDetailsModel.TemplateId).ToList();
                    itemCode = importItemDetailsModel.ItemCode;
                    uom = importItemDetailsModel.UOM;
                    gtin = importModel.GTIN;
                    string ser = string.Empty;
                    TemplateFieldModel temp = new TemplateFieldModel();
                    for (int j=0; j < importXML.SerialList.Count; j++)
                    {
                        ser = importXML.SerialList[j];
                        Row = expectedExcelSheetDataTable.NewRow();
                        int seq = 1;
                        fullSerial = string.Empty;
                        for (int k = 0; k< currentItemtemplateField.Count; k++)
                        { 
                            temp = new TemplateFieldModel();
                            temp = currentItemtemplateField[k];
                            if (seq == temp.FieldSequence)
                            {
                                if (temp.FieldId == BarcodeTemplateFields.GTIN.GetHashCode())
                                    fullSerial += "01" + gtin;
                                else if (temp.FieldId == BarcodeTemplateFields.Serial.GetHashCode() && !string.IsNullOrEmpty(ser))
                                    fullSerial += "21" + ser;
                                else if (temp.FieldId == BarcodeTemplateFields.BatchNumber.GetHashCode() && !string.IsNullOrEmpty(batchNoElement))
                                    fullSerial += "10" + batchNoElement;
                                else if (temp.FieldId == BarcodeTemplateFields.ExpiryDate.GetHashCode() && !string.IsNullOrEmpty(expiryDate))
                                    fullSerial += "17" + expiryDate;
                            }
                            seq++;
                        }
                        expectedExcelSheetDataTable.Rows.Add(new object[] { itemCode, uom, fullSerial, fullSerial, "", batchNoElement, expiryDate, sscc, "" });
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public static GlobalErrors GetExcelSheetColumns(FileInfo file, int sheetNo, ref DataTable dt)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dt = new DataTable();
                string columnName = string.Empty;
                DataRow dr = null;
                using (ExcelPackage package = new ExcelPackage(file))
                {
                    if (package != null && package.Workbook != null && package.Workbook.Worksheets != null && package.Workbook.Worksheets.Count > 0)
                    {
                        ExcelWorksheet worksheet = package.Workbook.Worksheets[sheetNo];
                        if (worksheet != null && worksheet.Dimension != null)
                        {
                            int colCount = worksheet.Dimension.Columns;

                            object[,] valueArray = worksheet.Cells.GetValue<object[,]>();
                            // columns
                            for (int col = 0; col < colCount; col++)
                            {
                                columnName = string.Empty;
                                if (valueArray[0, col] != null && !string.IsNullOrEmpty(valueArray[0, col].ToString()))
                                    columnName = valueArray[0, col].ToString();
                                dt.Columns.Add(columnName);
                            }
                        }
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public static GlobalErrors ValidateUploadedExcelSheetColumns(FileInfo file, ImportDataTypes dataType, int vendorId,int sheetNo, ref bool isValid, DataTable correctExcelSheetColumns,DBHelper<int> dBHelper)
        {
            List<ExcelSheetDynamicTemplateModel> excelSheetlist = new List<ExcelSheetDynamicTemplateModel>();
            return ValidateUploadedExcelSheetColumns(file, dataType, vendorId, sheetNo, ref isValid, ref excelSheetlist, correctExcelSheetColumns, dBHelper);
        }
        public static GlobalErrors ValidateUploadedExcelSheetColumns(FileInfo file, ImportDataTypes dataType, int vendorId, int sheetNo, ref bool isValid, ref List<ExcelSheetDynamicTemplateModel> excelSheetlistSharedTable, DataTable correctExcelSheetColumns, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string fileName = string.Empty;
            isValid = true;
        
            try
            {
                fileName = GetFileName(dataType);
                if (file != null && file.Length > 0 && (!string.IsNullOrEmpty(fileName) || dataType == ImportDataTypes.AppData))
                {
                    DataTable tempFileDataTable = new DataTable();
                   
                    if (dataType != ImportDataTypes.AppData)
                    {
                        var path = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", fileName);
                        FileInfo tempFile = new FileInfo(path);
                        result = GetExcelSheetColumns(tempFile, sheetNo, ref tempFileDataTable);
                    }
                    else 
                    {
                        tempFileDataTable = correctExcelSheetColumns;
                        result = GlobalErrors.Success;
                    }

                    if (result != GlobalErrors.Success)
                        return GlobalErrors.Error;
                    DataTable uploadedExcelSheetColumns = new DataTable();
                    result = GetExcelSheetColumns(file, sheetNo, ref uploadedExcelSheetColumns);
                    if (vendorId == -1)
                    {
                        if (result == GlobalErrors.Success && tempFileDataTable != null && tempFileDataTable.Columns != null
                            && uploadedExcelSheetColumns != null && uploadedExcelSheetColumns.Columns != null)
                        {
                            if (tempFileDataTable.Columns.Count != uploadedExcelSheetColumns.Columns.Count)
                            {
                                isValid = false;
                            }
                            else
                            {
                                foreach (DataColumn dataColumn in tempFileDataTable.Columns)
                                {
                                    if (!uploadedExcelSheetColumns.Columns.Contains(dataColumn.ColumnName))
                                    {
                                        isValid = false;
                                        break;
                                    }
                                    if (!isValid)
                                        break;
                                }
                            }
                        }
                        else
                        {
                            isValid = false;
                        }
                    }
                    else
                    {
                        excelSheetlistSharedTable = new List<ExcelSheetDynamicTemplateModel>();
                        ExcelSheetDynamicTemplateModel templateObject = new ExcelSheetDynamicTemplateModel();
                        templateObject.ExcelSheetTypeId = 1;
                        templateObject.VendorId = vendorId;
                        templateObject.IsEditMode = true;

                        DBHelper<ExcelSheetDynamicTemplateModel> dbHelper = null;
                        if (dBHelper != null) dbHelper = new DBHelper<ExcelSheetDynamicTemplateModel>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                        result = SettingManager.GetExcelSheetSharedTable(templateObject, true, ref excelSheetlistSharedTable, dbHelper);
                        if (result == GlobalErrors.Success)
                        {
                            foreach (ExcelSheetDynamicTemplateModel excelSheetDynamicTemplate in excelSheetlistSharedTable)
                            {
                                if (excelSheetDynamicTemplate.DefaultValue != "ExpiryDateFormat")
                                {
                                    if (!string.IsNullOrEmpty(excelSheetDynamicTemplate.DynamicValue) && !uploadedExcelSheetColumns.Columns.Contains(excelSheetDynamicTemplate.DynamicValue))
                                    {
                                        isValid = false;
                                        break;
                                    }
                                    if (!isValid)
                                        break;
                                }
                                if (!isValid)
                                    break;
                            }
                        }
                    }
                }
                else
                {
                    isValid = false;
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors AddStoredProcedureNeededColumnsToDataTable(ref DataTable dt, ImportDataTypes dataType, ref string triggerID)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                //Random random = new Random();
                triggerID = dataType == ImportDataTypes.TeleRepPlan ? Guid.NewGuid().ToString() : _requestRepository.SessionId;
                dt.Columns.Add("ID", typeof(int));
                dt.Columns.Add("UserID", typeof(int));
                dt.Columns.Add("TriggerID", typeof(string));
                if (dataType != ImportDataTypes.ItemsPricesWithoutQtyRange && dataType != ImportDataTypes.ItemsPricesWithQtyRange && dataType != ImportDataTypes.MustSellItems && dataType != ImportDataTypes.CustomerPromotion && dataType != ImportDataTypes.ExcludeCustomerPromotion && dataType != ImportDataTypes.CustomerOutletPromotion && dataType != ImportDataTypes.ExcludeCustomerOutletPromotion)
                {
                    dt.Columns.Add("ResultID", typeof(int));
                    dt.Columns.Add("Message", typeof(string));
                    dt.Columns.Add("Inserted", typeof(bool));
                    dt.Columns.Add("Updated", typeof(bool));
                    dt.Columns.Add("Skipped", typeof(bool));
                    if(dataType != ImportDataTypes.ItemsPricesWithQtyRange)
                    dt.Columns.Add("OrganizationID", typeof(string));
                }
                if(dataType == ImportDataTypes.KPIs){
                    dt.Columns.Add("RowNumber", typeof(int));
                }
                for (int i = 0; i < dt.Rows.Count; i++)
                {
                    dt.Rows[i]["ID"] = (i + 1);
                    dt.Rows[i]["TriggerID"] = triggerID;
                    dt.Rows[i]["UserID"] = _requestRepository.CurrentOperator.EmployeeId;
                    if (dataType != ImportDataTypes.ItemsPricesWithQtyRange)
                        dt.Rows[i]["OrganizationID"] = _requestRepository.CurrentOperator.OrganizationId;
                    if (dataType == ImportDataTypes.KPIs)
                    {
                        dt.Rows[i]["RowNumber"] = i + 2;
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors CreateTemplateTableInSQL(DataTable dt, string tableName, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string dataTableCreationQuery = string.Empty;
            string columns = "";
            try
            {
                if(dBHelper == null) dBHelper = new DBHelper<int>();
                foreach (DataColumn col in dt.Columns)
                {
                    columns += "\r\n        [" + col.Caption + "] ";
                    switch (col.DataType.Name)
                    {
                        case "Int32":
                            columns += "INT";
                            break;
                        case "Boolean":
                            columns += "BIT";
                            break;
                        default:
                            columns += "NVARCHAR(200)";
                            break;
                    }
                    columns += " NULL,";
                }
                if (!string.IsNullOrEmpty(columns))
                {
                    if (tableName == "ImportScorecardTemplate") 
                    {
                        columns = columns.Replace("[OrganizationCodes] NVARCHAR(200) NULL,", "[OrganizationCodes] NVARCHAR(MAX) NULL,");
                        columns = columns.Replace("[EmployeeCodes] NVARCHAR(200) NULL,", "[EmployeeCodes]  NVARCHAR(MAX) NULL,");
                        columns = columns.Replace("[SecurityGroupCodes] NVARCHAR(200) NULL,", "[SecurityGroupCodes] NVARCHAR(MAX) NULL,");
                    }

                    dataTableCreationQuery = string.Format(@"IF not EXISTS 
                        (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'{0}') AND OBJECTPROPERTY(id, N'IsUserTable') = 1)
                        
                        CREATE TABLE {0}({1}
                         );", tableName, columns.Substring(0, columns.Length - 1));
                    result = dBHelper.ExecuteSQLCommand(dataTableCreationQuery);
                }
              }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors CopyExcelSheetTableToTemplateTableInSQL(DataTable dt, string tableName, DBHelper<int> dBHelper1)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<string> dBHelper = null;
            bool commitNow = false;
            try
            {
                if (dBHelper1 == null)
                {
                    dBHelper = new DBHelper<string>();
                    commitNow = true;
                }
                else 
                {
                    dBHelper = new DBHelper<string>(dBHelper1.GetConnection(), dBHelper1.GetDBTransaction());
                }

                result = DeleteCurrentUserDataFromTemplateTableInSQL(tableName, dBHelper1);
                if (result == GlobalErrors.Success)
                {
                    if (commitNow)
                    {
                        dBHelper.BeginTransaction();
                        using (var bulkCopy = new SqlBulkCopy(dBHelper.GetConnection().ConnectionString, SqlBulkCopyOptions.KeepIdentity))
                        {
                            // my DataTable column names match my SQL Column names, so I simply made this loop. However if your column names don't match, just pass in which datatable name matches the SQL column name in Column Mappings
                            foreach (DataColumn col in dt.Columns)
                            {
                                bulkCopy.ColumnMappings.Add(col.ColumnName, col.ColumnName);
                            }
                            bulkCopy.BatchSize = 4000;
                            // bulkCopy.BulkCopyTimeout = 300;
                            bulkCopy.DestinationTableName = tableName;
                            bulkCopy.WriteToServer(dt);
                            result = dBHelper.CommitTransaction();
                        }
                    }
                    else 
                    {
                        // https://stackoverflow.com/questions/38391801/c-sharp-convert-dbcontexttransaction-to-sqltransaction
                        using (var bulkCopy = new SqlBulkCopy((dBHelper.GetConnection()) as SqlConnection, SqlBulkCopyOptions.KeepIdentity, (dBHelper.GetDBTransaction()) as SqlTransaction))
                        {
                            // my DataTable column names match my SQL Column names, so I simply made this loop. However if your column names don't match, just pass in which datatable name matches the SQL column name in Column Mappings
                            foreach (DataColumn col in dt.Columns)
                            {
                                bulkCopy.ColumnMappings.Add(col.ColumnName, col.ColumnName);
                            }
                            bulkCopy.BatchSize = 4000;
                            // bulkCopy.BulkCopyTimeout = 300;
                            bulkCopy.DestinationTableName = tableName;
                            bulkCopy.WriteToServer(dt);
                        }

                        object field = null;
                        string query = $"Select Count(*) from {tableName}";
                        result = dBHelper1.ExecuteScalar(query, ref field);
                        if (result == GlobalErrors.Success && field != null && !string.IsNullOrEmpty(field.ToString()) && int.Parse(field.ToString()) == dt.Rows.Count)
                            return GlobalErrors.Success;
                        else
                            return GlobalErrors.Error;
                    }

                }
            }
            catch (Exception ex)
            {
                if(commitNow) dBHelper.RollBackTransaction();
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteCurrentUserDataFromTemplateTableInSQL(string tableName, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if(dBHelper == null) dBHelper = new DBHelper<int>();
                 object  obj = null;
                string query = string.Format(@" IF EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'{0}') AND OBJECTPROPERTY(id, N'IsUserTable') = 1)
                         Delete From {0} where UserID = {1}", tableName, _requestRepository.CurrentOperator.EmployeeId);
                result = dBHelper.ExecuteScalar(query, ref obj);


            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);

                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteImportDetailsStatus(string storedProcedureName, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            if(dBHelper == null) dBHelper = new DBHelper<int>();
            try
            {
                string query = string.Format(@"Delete From ImportDetailsStatus where ProcedureName = '{0}'", storedProcedureName);
                result = dBHelper.ExecuteNoneQueryWithZeroRowAffectedCheck(query);
                if (result == GlobalErrors.SuccessWithZeroRowAffected)
                    result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);

                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteImportErrors(DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            if (dBHelper == null) dBHelper = new DBHelper<int>();

            try
            {
                string query = string.Format(@"Delete From ImportDetailsErrors where UserId = {0}", _requestRepository.CurrentOperator.OperatorId);
                result = dBHelper.ExecuteNoneQueryWithZeroRowAffectedCheck(query);
                if (result == GlobalErrors.SuccessWithZeroRowAffected)
                    result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);

                return GlobalErrors.Error;
            }

            return result;
        }
        public GlobalErrors ExecuteStoredProcedure(string storedProcedureName, Dictionary<string, object> inputParamters, Dictionary<string, object> outputParamters, ref Dictionary<string, object> resultParamters, DBHelper<string> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool closeConnection = false;
            if (dBHelper == null)
            {
                dBHelper = new DBHelper<string>();
                closeConnection = true;
            }
            string query = string.Empty;
            object field = new object();
            resultParamters = new Dictionary<string, object>();
            try
            {
                
                query = string.Format(@"if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].{0}') 
                        and OBJECTPROPERTY(id, N'IsProcedure') = 1) select x='1'", storedProcedureName);
                result = dBHelper.ExecuteScalar(query, ref field);
                if (field != null && !string.IsNullOrEmpty(field.ToString()) && (field.ToString() == "1"))
                {
                    result = dBHelper.ExecuteProcedureWithParameters(storedProcedureName, inputParamters, outputParamters, ref resultParamters,0, closeConnection);
                }
                else return GlobalErrors.Error;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveImportHistory(ImportDataTypes importTypeID, string triggerID, int totalRecords, int succeedRecords, int failedRecords, DBHelper<string> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            if(dBHelper == null) dBHelper = new DBHelper<string>();
            string auditQuery = string.Empty;
            try
            {
                auditQuery = string.Format("INSERT INTO ImportHistory (TriggerID,TypeID,UserID,ImportDate,TotalRecordsCount,SucceedRecordsCount,FailedRecordsCount) VALUES('{0}',{1},{2},{3},{4},{5},{6})",
                             triggerID, //0
                             importTypeID.GetHashCode(), //1
                             _requestRepository.CurrentOperator.EmployeeId, //2
                             LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now), //3
                             totalRecords, //4
                             succeedRecords, //5
                             failedRecords //6
                            ); 
                result = dBHelper.ExecuteNonQuery(auditQuery);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetImportedExcelErrors(ImportDataTypes importTypeID, string triggerID, ref List<ImportedExcelErrorModel> importedExcelErrors, DBHelper<ImportedExcelErrorModel> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            if(dBHelper == null) dBHelper = new DBHelper<ImportedExcelErrorModel>();
            string query = string.Empty;
            string spName = string.Empty;
            try
            {
                if (importTypeID == ImportDataTypes.KPIScorecard)
                {
                    query = string.Format("SELECT ResultID, Message FROM ImportScorecardResult WHERE  (TriggerID = '{0}') AND (ResultID <> 1)", triggerID);
                }
                else 
                {
                    string tableName = "ImportKPIsTemplate";
                    string rowNumber = " , RowNumber";
                    if (importTypeID == ImportDataTypes.TeleRepPlan) 
                    { 
                        tableName = "ImportTeleRepPlanTemplate";
                        rowNumber = string.Empty;
                    }
                    if (importTypeID == ImportDataTypes.AppData)
                    {
                        GetTableNameAndProcedureName(importTypeID, ref tableName, ref spName, _requestRepository.CurrentOperator.EmployeeId, _requestRepository.SessionId);
                        rowNumber = " , ID + 1 AS RowNumber";
                    }
                    if (importTypeID == ImportDataTypes.ContactCustomers)
                    {
                        tableName = "ImportContactCustomersTemplate";
                        rowNumber = " , ID + 1 AS RowNumber";
                    }
                    query = string.Format("SELECT (ID) Id, ResultID, Message {2} FROM {1} WHERE  (TriggerID = '{0}') AND (ResultID <> 1)", triggerID, tableName, rowNumber);
                }
                result = dBHelper.GetQueryList(query, ref importedExcelErrors);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetImportDetailsList(string tableName,string triggerID, ref List<ImportModel> importModel, DBHelper<ImportModel> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                importModel = new List<ImportModel>();
                if(dbHelper == null) dbHelper = new DBHelper<ImportModel>();
                string query = string.Format(@"select {0}.TriggerID,{0}.UserID, 
                count(*) TotalRecordsCount, isnull(dtFailed.SkippedRecords,0) FailedRecordsCount,
                isnull(dtSuccess.SucceesRecords,0) SucceedRecordsCount from {0}
                left outer join 
                (select count(*) SkippedRecords, TriggerID, UserID from {0} where skipped = 1 
                group by TriggerID, UserID) dtFailed on 
                dtFailed.TriggerID = {0}.TriggerID and dtFailed.UserID = {0}.UserID

                left outer join 
                (select count(*) SucceesRecords, TriggerID, UserID from {0} where skipped = 0
                group by TriggerID, UserID) dtSuccess on 
                dtSuccess.TriggerID = {0}.TriggerID and dtSuccess.UserID = {0}.UserID

                where  {0}.triggerid = '{2}' and {0}.UserID = {1}
                group by {0}.TriggerID,{0}.UserID,dtFailed.SkippedRecords,dtSuccess.SucceesRecords",
                tableName, //0
                _requestRepository.CurrentOperator.EmployeeId, //1
                triggerID //2
                );
                result = dbHelper.GetQueryList(query, ref importModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetImportErrorsList(string tableName,ref List<ImportErrorData> ImportErrorData)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {

                DBHelper<ImportErrorData> dbHelper = new DBHelper<ImportErrorData>();
                ImportErrorData = new List<ImportErrorData>();
                if (dbHelper == null) dbHelper = new DBHelper<ImportErrorData>();
                string query = string.Format(@"select * from ImportDetailsErrors"
                );
                result = dbHelper.GetQueryList(query, ref ImportErrorData);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public static GlobalErrors ModifyExcelSheetColumns(List<LanguageModel> languages)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                int languagesCount = 0;
                if (languages != null)
                    languagesCount = languages.Count;

                var origionalPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "Items-Template.xlsx");
                var newPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "ItemsTemplate.xlsx");
                result = ModifyExcelSheetTemplate(languages, 7, languagesCount, origionalPath, newPath, "ItemDescription", string.Empty);

                origionalPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "Employees-Template.xlsx");
                newPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "EmployeesTemplate.xlsx");
                result = ModifyExcelSheetTemplate(languages, 6, languagesCount, origionalPath, newPath, "EmployeeDescription", string.Empty);

                origionalPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "Customers-Template.xlsx");
                newPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "CustomersTemplate.xlsx");
                result = ModifyExcelSheetTemplate(languages, 9, languagesCount, origionalPath, newPath, "CustomerDescription", "OutletDescription");

                origionalPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "ItemsPricesTemplateWithQtyRangeTemplate.xlsx");
                newPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "ItemsPricesTemplateWithQtyRange-Template.xlsx");
                result = ModifyExcelSheetTemplate(languages, 6, languagesCount, origionalPath, newPath, string.Empty, string.Empty);


                origionalPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "DisplayUnit-Template.xlsx");
                newPath = Path.Combine(Directory.GetCurrentDirectory(), "wwwroot\\Templates", "DisplayUnitTemplate.xlsx");
                result = ModifyExcelSheetTemplateDisplayUnit(languages, languagesCount, origionalPath, newPath );

                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public static GlobalErrors AddCustomerTypeDropDownToExcelSheet(ExcelWorksheet worksheet, string filePath, string sheetName, string customerTypeColumnName, List<string> dropDownValues)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                using (var package = new ExcelPackage(new FileInfo(filePath)))
                {
                    if (worksheet != null)
                    {
                        // Get the column index for the specified column name
                        int customerTypeColumnIndex = -1;
                        for (int col = 1; col <= worksheet.Dimension.End.Column; col++)
                        {
                            if (worksheet.Cells[1, col].Text == customerTypeColumnName)
                            {
                                customerTypeColumnIndex = col;
                                break;
                            }
                        }

                        if (customerTypeColumnIndex == -1)
                        {
                            throw new Exception($"{customerTypeColumnName} column not found.");
                        }

                        var dropDownAddress = worksheet.Cells[2, customerTypeColumnIndex, 1000, customerTypeColumnIndex];

                        var validation = worksheet.DataValidations.AddListValidation(dropDownAddress.Address);
                        validation.ShowErrorMessage = true;
                        validation.ErrorStyle = ExcelDataValidationWarningStyle.stop;
                        validation.ErrorTitle = "Invalid Selection";
                        validation.Error = "Please select a value from the drop-down list.";
                        validation.AllowBlank = true;
                        //  validation.Formula.ExcelFormula = string.Join(",", dropDownValues);

                        // Optional: Set input message
                        validation.PromptTitle = "Select Customer Type";
                        validation.Prompt = "Please select a customer type from the drop-down list.";

                        foreach (var value in dropDownValues)
                        {
                            validation.Formula.Values.Add(value);
                        }
                        

                        result = GlobalErrors.Success;
                    }
                    else
                    {
                        throw new Exception($"Worksheet '{sheetName}' not found.");
                    }
                }
            }
            catch (Exception ex)
            {
                // Handle exceptions appropriately
                Console.WriteLine("Error: " + ex.Message);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public static GlobalErrors ModifyExcelSheetTemplate(List<LanguageModel> languages,int columnIndex, int addedColumnsCount, string origionalPath, string newPath, string columnName, string columnName2)
       {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
            

                using (var stream = new FileStream(origionalPath, FileMode.Open))
                {
                    if (stream != null && stream.Length > 0)
                    {
                        using (ExcelPackage package = new ExcelPackage())
                        {
                            package.Load(stream);
                            if (package != null && package.Workbook != null && package.Workbook.Worksheets != null && package.Workbook.Worksheets.Count > 0)
                            {

                                ExcelWorksheet worksheet = package.Workbook.Worksheets[1];
                         
                                if (worksheet != null && addedColumnsCount > 0 && !(newPath.Contains("ItemsPricesTemplateWithQtyRange-Template", StringComparison.OrdinalIgnoreCase)))
                                {

                                    worksheet.InsertColumn(columnIndex, addedColumnsCount);
                                    for (int i = 0; i < addedColumnsCount; i++)
                                    {
                                        worksheet.Cells[1, columnIndex + i].Value = columnName + "-" + languages[i].Code;
                                        worksheet.Cells[1, columnIndex + i].Style.Fill.PatternType = OfficeOpenXml.Style.ExcelFillStyle.Solid;
                                        worksheet.Cells[1, columnIndex + i].Style.Fill.BackgroundColor.SetColor(System.Drawing.Color.Red);
                                    }
                                    if (!string.IsNullOrEmpty(columnName2))
                                    {
                                        columnIndex = columnIndex + addedColumnsCount;
                                        for (int i = 0; i < addedColumnsCount; i++)
                                        {
                                            worksheet.Cells[1, columnIndex + i].Value = columnName2 + "-" + languages[i].Code;
                                            worksheet.Cells[1, columnIndex + i].Style.Fill.PatternType = OfficeOpenXml.Style.ExcelFillStyle.Solid;
                                            worksheet.Cells[1, columnIndex + i].Style.Fill.BackgroundColor.SetColor(System.Drawing.Color.Red);
                                        }
                                    }
                                }
                                string tableName = string.Empty;
                                string columnTypeName = string.Empty;
                                if (newPath.Contains("ItemsPricesTemplateWithQtyRange-Template", StringComparison.OrdinalIgnoreCase))
                                    columnName = "ItemsPricesTemplateWithQtyRangeTemplate";

                                switch (columnName)
                                {
                                    case "CustomerDescription":
                                        tableName = "CustomerClassLanguage";
                                        columnTypeName = "CustomerType";
                                        break;
                                    case "ItemDescription":
                                        tableName = "ItemCategoryLanguage";
                                        columnTypeName = "CategoryCode";
                                        break;
                                    case "ItemsPricesTemplateWithQtyRangeTemplate":
                                        columnTypeName = "QuantityRange";
                                        break;
                                    default:
                                        break;
                                }

                                if (!string.IsNullOrEmpty(tableName) && !string.IsNullOrEmpty(columnTypeName) && !newPath.Contains("ItemsPricesTemplateWithQtyRangeTemplate", StringComparison.OrdinalIgnoreCase))
                                {
                                    List<string> TypeDescriptions = GetTypeDescriptions(tableName);
                                    AddCustomerTypeDropDownToExcelSheet(worksheet, newPath, worksheet.Name, columnTypeName, TypeDescriptions);
                                }
                                if (columnName == "ItemDescription")
                                {
                                    tableName = "PackTypeLanguage";
                                    columnTypeName = "UOM";
                                    List<string> TypeDescriptions = GetTypeDescriptions(tableName);

                                    AddCustomerTypeDropDownToExcelSheet(worksheet, newPath, worksheet.Name, columnTypeName, TypeDescriptions);
                                }
                                if (columnName == "ItemsPricesTemplateWithQtyRangeTemplate")
                                {
                                    List<string> TypeDescriptions = GetQuantityRangeDescriptions();
                                    AddCustomerTypeDropDownToExcelSheet(worksheet, newPath, worksheet.Name, columnTypeName, TypeDescriptions);

                                }
                                if (columnName == "ItemsPricesTemplateWithQtyRangeTemplate")
                                {
                                    tableName = "PackTypeLanguage";
                                    columnTypeName = "UOM";
                                    List<string> TypeDescriptions = GetTypeDescriptions(tableName);

                                    AddCustomerTypeDropDownToExcelSheet(worksheet, newPath, worksheet.Name, columnTypeName, TypeDescriptions);
                                }




                                // Add drop-down list to "CustomerType" column

                                if (System.IO.File.Exists(newPath))
                                {
                                    System.IO.File.Delete(newPath);
                                }
                                using (var newStream = new FileStream(newPath, FileMode.Create))
                                {
                                    package.SaveAs(newStream);
                                    package.Dispose();
                                }
                                 
                            }
                        }
                    }
                }

                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }


        public static  List<string> GetTypeDescriptions(string tableName)
        {
            List<string> descriptions = new List<string>();

            DBHelper<string> dbHelper = new DBHelper<string>();
            string query = string.Format(@"
        Select Description 
        from {0} 
        where LanguageID = {1}", tableName, 1);//_requestRepository.LanguageId);

            dbHelper.GetQueryList(query, ref descriptions);

            return descriptions;
        }
        public static List<string> GetQuantityRangeDescriptions()
        {
            List<string> descriptions = new List<string>();

            DBHelper<string> dbHelper = new DBHelper<string>();
            string query = string.Format(@"select CONCAT(RangeStart ,' - ', RangeEnd)  from PriceQuantityRange");//_requestRepository.LanguageId);

            dbHelper.GetQueryList(query, ref descriptions);

            return descriptions;
        }


        public static GlobalErrors ModifyExcelSheetTemplateDisplayUnit(List<LanguageModel> languages, int addedColumnsCount, string origionalPath, string newPath)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {

                var displayUnitColunm = new Dictionary<string, int>();

                // ... Add some keys and values.
                displayUnitColunm.Add("SerialNumber", 0);
                displayUnitColunm.Add("DisplayUnitName", 1);
                displayUnitColunm.Add("OrganizationCode", 0);
                displayUnitColunm.Add("Status", 1);
                displayUnitColunm.Add("MasterGroupCode", 0);
                displayUnitColunm.Add("MasterGroupName", 1);
                displayUnitColunm.Add("SubGroupCode", 0);
                displayUnitColunm.Add("SubGroupName", 1);
                displayUnitColunm.Add("Type", 1);
                displayUnitColunm.Add("CapacityType", 0);
                displayUnitColunm.Add("IsRentedType", 0);
                displayUnitColunm.Add("Remarks", 1);
                displayUnitColunm.Add("DisplayUnitValue", 0);
                displayUnitColunm.Add("Capacity", 0);
                displayUnitColunm.Add("Brand", 0);
                displayUnitColunm.Add("Model", 0);
                displayUnitColunm.Add("Manufacturer", 0);
                displayUnitColunm.Add("DisplayUnitTemplateCode", 0);
                displayUnitColunm.Add("CustomerCode", 0);
                displayUnitColunm.Add("OutletCode", 0);


                using (var stream = new FileStream(origionalPath, FileMode.Open))
                {
                    if (stream != null )
                    {
                        using (ExcelPackage package = new ExcelPackage())
                        {
                            package.Load(stream);
                            if (package != null && package.Workbook != null && package.Workbook.Worksheets != null )
                            {

                                ExcelWorksheet worksheet = package.Workbook.Worksheets[1];
                                if (worksheet != null)
                                {
                                    int counter = 1;
                                    foreach (var pair in displayUnitColunm)
                                    {
                                        if (pair.Value == 1)
                                        {
                                            worksheet.InsertColumn(counter, addedColumnsCount);
                                            for (int i = 0; i < addedColumnsCount; i++)
                                            {
                                                worksheet.Cells[1, counter].Value = pair.Key + "-" + languages[i].Code;
                                                counter++;
                                            }
                                        }
                                        else 
                                        {
                                            worksheet.InsertColumn(counter, 1);
                                            worksheet.Cells[1, counter].Value = pair.Key;
                                            counter++;
                                        }

                                    }
                                    if (System.IO.File.Exists(newPath))
                                    {
                                        System.IO.File.Delete(newPath);
                                    }
                                    using (var newStream = new FileStream(newPath, FileMode.Create))
                                    {
                                        package.SaveAs(newStream);
                                        package.Dispose();
                                    }
                                }
                            }
                        }
                    }
                }

                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetWebFormColumns(int webFormEntry, ref List<string> columnNames)
        {
            return GetWebFormColumns(webFormEntry,  ref columnNames, null);
        }
        public GlobalErrors GetWebFormColumns(int webFormEntry, ref List<string> columnNames, DBHelper<string> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                columnNames = new List<string>();
                if(dBHelper == null) dBHelper = new DBHelper<string>();
                string query = $@" 
                Select CASE WHEN ControlTypeID in ({(int)WebFormControlType.DropDown},{(int)WebFormControlType.MultiSelectDropDown}) AND RTRIM(LTRIM(ColumnName)) like '%id' THEN SUBSTRING(RTRIM(LTRIM(ColumnName)), 1, LEN(RTRIM(LTRIM(ColumnName)))-2) ELSE RTRIM(LTRIM(ColumnName)) END as ColumnName from WebFormControl where WebFormID = {webFormEntry} 
                AND ControlTypeID <> {(int)WebFormControlType.DisabledText} AND IsVisible = 1 ORDER BY ControlIndex ASC";
                result = dBHelper.GetQueryList(query, ref columnNames);
                
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GenerateExcelSheet(List<string> columnNames, ref MemoryStream memory)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                using (ExcelPackage package = new ExcelPackage(memory))
                {
                    ExcelWorksheet workSheet = package.Workbook.Worksheets.Add("Sheet1");
                    for (int i = 0; i < columnNames.Count(); i++)
                    {
                        workSheet.Cells[1, i + 1].Value = columnNames[i];
                        workSheet.Column(i + 1).Style.Numberformat.Format = "@";
                    }
                    package.Save();
                }
                if (memory != null)
                {
                    memory.Position = 0;
                    result = GlobalErrors.Success;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors DeleteTableInSQL(string tableName, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if(dBHelper == null)
                    dBHelper = new DBHelper<int>();
                object obj = null;
                string query = string.Format(@" IF EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'{0}') AND OBJECTPROPERTY(id, N'IsUserTable') = 1)
                BEGIN
                DROP TABLE {0}
                END  ", tableName);
                result = dBHelper.ExecuteScalar(query, ref obj);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
    }
}