﻿using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using RestSharp;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.SecureLibrary;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Sonic.Web.Core
{
    public class PaymentManager
    {
        private readonly IRequestRepository _requestRepository;
        private readonly AccountManager _accountManager;
        private readonly EmployeeManager _employeeManager;
        private readonly DocumentSequenceManager _documentSequenceManager;

        public PaymentManager(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _accountManager = new AccountManager(_requestRepository);
            _employeeManager = new EmployeeManager(_requestRepository);
            _documentSequenceManager = new DocumentSequenceManager(requestRepository);
        }

        private const string RowIDProperty = "_rowID";

        public GlobalErrors GetPaymentTypes(ref List<PaymentTypeModel> paymentTypesList, bool isDownPayment, int customerID, int outletID)
        {
            DBHelper<PaymentTypeModel> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                paymentTypesList = new List<PaymentTypeModel>();
                dbHelper = new DBHelper<PaymentTypeModel>();
                string filter = string.Format(@"And (PaymentType.PaymentTypeID Not IN({0}))",
                                          "" + PaymentTypes.CreditNote.GetHashCode().ToString() +
                                         (isDownPayment && !_requestRepository.Configurations.AllowPDCInDownPayment ?
                                         ("," + PaymentTypes.PostedDatedCheque.GetHashCode().ToString()) : string.Empty));

                string customerTypeConfiguration = string.Empty;
                if (!isDownPayment)
                {
                    var customerManager = new CustomerManager(_requestRepository);

                    var customerType = (CustomerTypes)customerManager.GetCustomerType(customerID, outletID);
                    if (customerType == CustomerTypes.CashCustomer)
                    {
                        customerTypeConfiguration = _requestRepository.Configurations.PaymentTypesForCashCustomer;
                    }
                    else if (customerType == CustomerTypes.CreditCustomer || customerType == CustomerTypes.BillToBillCustomer)
                    {
                        customerTypeConfiguration = _requestRepository.Configurations.PaymentTypesForCreditCustomer;
                    }
                }
                else
                {
                    customerTypeConfiguration = _requestRepository.Configurations.PaymentTypesForCreditCustomer;
                }
                string determinePaymentType = string.Format(@"where  PaymentType.PaymentTypeID in({0})",
                                           !string.IsNullOrEmpty(customerTypeConfiguration) ?
                                           customerTypeConfiguration.Replace("'", "") : "-1");

                string excludePaymentType = $@" AND PaymentType.PaymentTypeID NOT IN ({(int)PaymentTypes.OnlineEPayment} {(_requestRepository.Configurations.UseOnlineCreditCard? $" , {(int)PaymentTypes.CreditCard} " : string.Empty)} ) ";
                string query = string.Format(@"select PaymentType.PaymentTypeID, PaymentTypeLanguage.Description
                    from PaymentType
                    inner join PaymentTypeLanguage on PaymentTypeLanguage.PaymentTypeID = PaymentType.PaymentTypeID 
                    and PaymentTypeLanguage.LanguageID = {1} {0} 
                    {2} {3}",
                    filter,
                    _requestRepository.LanguageId, determinePaymentType, excludePaymentType);

                result = dbHelper.GetQueryList(query, ref paymentTypesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentTypesList = null;
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetALLPaymentTypes(ref List<PaymentTypeModel> paymentTypes)
        {
            DBHelper<PaymentTypeModel> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<PaymentTypeModel>();

                string query = string.Format(@"select PaymentTypeLanguage.PaymentTypeID,PaymentTypeLanguage.Description from PaymentTypeLanguage
                where PaymentTypeLanguage.LanguageID = {0}", _requestRepository.LanguageId);

                result = dbHelper.GetQueryList(query, ref paymentTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentTypes = null;
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetBankWithBranches(ref List<BankModel> banksList)
        {
            DBHelper<BankModel> dbBankHelper = null;
            DBHelper<BranchModel> dbBranchHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                banksList = new List<BankModel>();
                dbBankHelper = new DBHelper<BankModel>();

                string query = string.Format(@"select distinct  Bank.BankID, BankLanguage.Description 
					from Bank
					inner join BankLanguage on BankLanguage.BankID = Bank.BankID and BankLanguage.LanguageID = {0}
					inner join BankBranch on BankBranch.BankID = Bank.BankID
					inner join BankBranchLanguage on BankBranchLanguage.BankID = BankBranch.BankID and BankBranchLanguage.BranchID = BankBranch.BranchID 
					and BankBranchLanguage.LanguageID = {0}",
                    _requestRepository.LanguageId);

                result = dbBankHelper.GetQueryList(query, ref banksList);

                if (result == GlobalErrors.Success && banksList != null && banksList.Count > 0)
                {
                    List<BranchModel> branchesList = new List<BranchModel>();
                    dbBranchHelper = new DBHelper<BranchModel>();
                    query = string.Format(@"select BankBranch.BankID, BankBranch.BranchID, BankBranchLanguage.Description
                       from BankBranch
                       inner join BankBranchLanguage on BankBranchLanguage.BankID = BankBranch.BankID and BankBranchLanguage.BranchID = BankBranch.BranchID
                       and BankBranchLanguage.LanguageID = {0}",
                       _requestRepository.LanguageId);

                    result = dbBranchHelper.GetQueryList(query, ref branchesList);

                    if (result == GlobalErrors.Success && branchesList != null && branchesList.Count > 0)
                    {
                        foreach (BankModel bank in banksList)
                        {
                            IEnumerable<BranchModel> branchesQuery =
                            from branch in branchesList
                            where branch.BankId == bank.BankId
                            select branch;

                            bank.branchList = branchesQuery.ToList();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                banksList = null;
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors SavePayments(SavePaymentModel savePaymentModel, DBHelper<int> dbHelper, ref string customerPaymentsIDsString, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                // Check Transaction employee account before saving payments
                foreach (TransactionModel transaction in savePaymentModel.TransactionsList)
                {
                    AccountModel empAccount = _accountManager.GetEmployeeAccount(transaction.EmployeeId, dbHelper);
                    if (empAccount == null)
                    {
                        savePaymentModel.ValidationErrors = ValidationErrors.NoEmployeeAccount;
                        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Saving Payment Faild", null, "No account defined for transaction employee", 0);
                        return GlobalErrors.Error;
                    }
                }
                customerPaymentsIDsString = string.Empty;
                result = SaveTransactionPayments(savePaymentModel, ref customerPaymentsIDsString, dbHelper , ref errorMessage);

                if (result == GlobalErrors.Success)
                {
                    bool isManulCollect = false;
                    result = PostCustomerPayments(_requestRepository.CurrentOperator.EmployeeId, string.Empty, savePaymentModel.CustomerId, savePaymentModel.OutletId, isManulCollect, dbHelper, ref errorMessage);
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

            return result;
        }
        public GlobalErrors SavePayments(SavePaymentModel savePaymentModel, DBHelper<int> dbHelper, ref string errorMessage)
        {
            string customerPaymentsIDsStringRef = string.Empty;
            return SavePayments(savePaymentModel, dbHelper, ref customerPaymentsIDsStringRef, ref errorMessage);
        }

        private GlobalErrors SaveTransactionPayments(SavePaymentModel savePaymentModel, ref string customerPaymentsIDsString, DBHelper<int> dbHelper, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                Dictionary<string, int> savedPayments = new Dictionary<string, int>();
                customerPaymentsIDsString = string.Empty;

                string maxCustomerPaymentID = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, savePaymentModel.EmployeeId, DocumentTypes.Collection, savePaymentModel.DivisionId);
                string maxAppliedPaymentID = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, savePaymentModel.EmployeeId, DocumentTypes.AppliedPayment, savePaymentModel.DivisionId);

                if (string.IsNullOrEmpty(maxCustomerPaymentID) || string.IsNullOrEmpty(maxAppliedPaymentID))
                {
                    savePaymentModel.ValidationErrors = ValidationErrors.NoDocumentSequence;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Saving Payment Faild", null, "No Document Sequence Defined", 0);
                    return GlobalErrors.Error;
                }

                decimal appliedNumber = 0;
                string appliedCharsPart = string.Empty;
                string appliedNumericPart = string.Empty;
                DocumentSequenceManager.GetSequenceNumberFormat(maxAppliedPaymentID, ref appliedCharsPart, ref appliedNumericPart);
                if (!appliedNumericPart.Equals(string.Empty))
                {
                    appliedNumber = decimal.Parse(appliedNumericPart);
                }

                long customerID = -1; int outletID = -1;
                DateTime paymentDateForAllTransactions = DateTime.Now; // to save payment on all transactions with same date
                foreach (TransactionModel transaction in savePaymentModel.TransactionsList)
                {
                    // check if remainingAmount from UI same as the one in DB 
                    bool ValidRemainingAmount = CheckTransactionRemainingAmount(transaction.TransactionId , transaction.DivisionId , transaction.RemainingAmount , dbHelper);
                    if(!ValidRemainingAmount)
                    {
                        errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Refresh_Data, _requestRepository.LanguageId);
                        return result = GlobalErrors.Error;
                    }
                    if(transaction.RemainingAmount < transaction.AppliedAmount && !_requestRepository.Configurations.AllowPayingExtraAtCollection)
                    {
                        errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Another_Payments_Applied, _requestRepository.LanguageId);
                        return result = GlobalErrors.Error;
                    }
                    if (transaction.RemainingAmount > 0 && transaction.AppliedAmount > 0 && savePaymentModel.PaymentsList.Count > 0)
                    {
                        result = SaveTransactionPayments(dbHelper, transaction, savePaymentModel.PaymentsList, appliedCharsPart, appliedNumericPart, ref appliedNumber, ref savedPayments, ref customerPaymentsIDsString, ref maxCustomerPaymentID, customerID, outletID, savePaymentModel.EmployeeId, paymentDateForAllTransactions);
                        if (result != GlobalErrors.Success || savePaymentModel.PaymentsList.Count == 0)
                        {
                            break;
                        }
                        customerID = transaction.CustomerId;
                        outletID = transaction.OutletId;
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    if (!string.IsNullOrEmpty(customerPaymentsIDsString))
                    {
                        customerPaymentsIDsString = customerPaymentsIDsString.TrimEnd(",".ToCharArray());
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

            return result;
        }
        public bool CheckTransactionRemainingAmount(string transactionId , int divisionId , decimal remainingAmount , DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool isValid = false;
            string query = string.Empty;
            try
            {
                query = string.Format(@"SELECT COUNT(*) FROM [Transaction] WHERE [Transaction].TransactionID = '{0}' AND DivisionID = {1} AND RemainingAmount = {2}", transactionId , divisionId, remainingAmount );
                object objField = null;
                result = dbHelper.ExecuteScalar(query, ref objField);
                if (objField != null && !string.IsNullOrEmpty(objField.ToString()))
                {
                    isValid = Convert.ToBoolean(objField);
                }
                else
                {
                    isValid = false;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return isValid = false;
            }
            return isValid;
        }
        private GlobalErrors SaveTransactionPayments(DBHelper<int> dbHelper, TransactionModel transaction, List<PaymentModel> paymentsList, string charsPart, string numericPart, ref decimal number, ref Dictionary<string, int> savedPayments, ref string customerPaymentsIDsString, ref string maxCustomerPaymentID, long previousCustID, int previousOutID, int employeeId, DateTime paymentDateForAllTransactions)
        {
            GlobalErrors result = GlobalErrors.Error;
            decimal paidAmount = 0;
            int i = 0;
            //long employeeID;
            try
            {
                if(_requestRepository.Configurations.ApplyDivisionsOnDocumentSequence && _requestRepository.Configurations.ControlTransactionOnDivisionLevel)
                {
                    paymentsList.Select(x => { x.DivisionId = transaction.DivisionId; return x; }).ToList();
                }
                // employeeID = _requestRepository.Instance.CurrentOperator.EmployeeId;
                for (i = 0; i < paymentsList.Count; i++)
                {
                    if (transaction.RemainingAmount > 0 && (transaction.AppliedAmount > 0 || _requestRepository.Configurations.AllowPayingExtraAtCollection) && paymentsList[i].RemainingAmount > 0)
                    {
                        if (paymentsList[i].AppliedAmount == paymentsList[i].RemainingAmount)
                        {
                            paymentsList[i].CustomerPaymentId = maxCustomerPaymentID;
                            maxCustomerPaymentID = _documentSequenceManager.GetNextDocumentSequence(paymentsList[i].CustomerPaymentId, 1);
                        }
                        else if (transaction.CustomerId != previousCustID || transaction.OutletId != previousOutID)
                        {
                            paymentsList[i].CustomerPaymentId = _documentSequenceManager.GetNextDocumentSequence(paymentsList[i].CustomerPaymentId, 1);
                            maxCustomerPaymentID = _documentSequenceManager.GetNextDocumentSequence(paymentsList[i].CustomerPaymentId, 1);
                        }
                        paidAmount = 0;
                        paymentsList[i].AppliedPaymentId = string.Format("{0}{1}", charsPart, number.ToString().PadLeft(numericPart.Length, '0'));
                        number++;
                        if (transaction.AppliedAmount > paymentsList[i].RemainingAmount)
                        {
                            paidAmount = paymentsList[i].RemainingAmount;
                            transaction.RemainingAmount -= paidAmount;
                            transaction.AppliedAmount -= paidAmount;
                            paymentsList[i].PaymentStatusId = PaymentStatusTypes.Pending;
                            paymentsList[i].RemainingAmount = 0;
                        }
                        else
                        {
                            if(transaction.AppliedAmount == 0 && _requestRepository.Configurations.AllowPayingExtraAtCollection)
                            {
                                paidAmount = paymentsList[i].RemainingAmount;
                            }
                            else
                            {
                                paidAmount = transaction.AppliedAmount;
                            }
                            paymentsList[i].RemainingAmount -= paidAmount;
                            transaction.RemainingAmount -= paidAmount;
                            transaction.AppliedAmount = 0;

                            if (transaction.RemainingAmount == 0)
                            {
                                transaction.TransactionStatusId = TransactionStatues.Paid.GetHashCode();
                                paymentsList[i].PaymentStatusId = PaymentStatusTypes.FullyCollected;
                            }
                            else
                            {
                                paymentsList[i].PaymentStatusId = PaymentStatusTypes.Pending;
                            }
                        }
                        if (_requestRepository.Configurations.AutoApprovePayment)
                        {
                            paymentsList[i].PaymentStatusId = PaymentStatusTypes.Approved;
                        }
                        if(!_requestRepository.Configurations.AllowPayingExtraAtCollection && transaction.RemainingAmount < 0)
                        {
                            return GlobalErrors.Error;
                        }
                        if (InsertPayment(dbHelper, transaction, paymentsList[i], paidAmount, transaction.AccountId, int.Parse(employeeId.ToString()), paymentDateForAllTransactions) != GlobalErrors.Success)
                        {
                            return GlobalErrors.Error;
                        }
                        if (!savedPayments.ContainsKey(paymentsList[i].CustomerPaymentId))
                        {
                            if (_documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.Collection, paymentsList[i].CustomerPaymentId, employeeId, transaction.DivisionId) != GlobalErrors.Success)
                            {
                                return GlobalErrors.Error;
                            }
                            savedPayments[paymentsList[i].CustomerPaymentId] = paymentsList[i].PaymentTypeId.GetHashCode();
                            customerPaymentsIDsString += string.Format("'{0}',", paymentsList[i].CustomerPaymentId);
                        }
                        if (_documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.AppliedPayment, paymentsList[i].AppliedPaymentId, employeeId, transaction.DivisionId) != GlobalErrors.Success)
                        {
                            return GlobalErrors.Error;
                        }
                        if((_requestRepository.Configurations.AllowPayingExtraAtCollection && (transaction.RemainingAmount <= 0 || i == (paymentsList.Count - 1))) 
                            || (!_requestRepository.Configurations.AllowPayingExtraAtCollection && (transaction.RemainingAmount <= 0 || transaction.AppliedAmount <= 0)))
                        {
                            break;
                        }
                    }
                }
                if (transaction.AppliedAmount > 0)
                {
                    paymentsList.Clear();
                    result = GlobalErrors.Success;
                }
                else
                {
                    result = RemovePaidPayments(ref paymentsList, i);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors InsertPayment(DBHelper<int> dbHelper, TransactionModel transaction, PaymentModel payment, decimal paidAmount, int accountId, int employeeId, DateTime paymentDateForAllTransactions)
        {
            GlobalErrors result = GlobalErrors.Error;
            string query = string.Empty;
            int supervisorId = -1;
            int helperId = -1;
            int salesManagerId = -1;
            int organizationId = -1;
            int driverId = -1;
            int salesRepId = -1;

            try
            {
                string voucherDate = string.Empty;
                if (payment.PaymentTypeId == PaymentTypes.PostedDatedCheque || payment.PaymentTypeId == PaymentTypes.CurrentDatedCheque || payment.PaymentTypeId == PaymentTypes.CertifiedCheque)
                {
                    payment.VoucherDate = new DateTime(payment.VoucherDateModel.Year, payment.VoucherDateModel.Month, payment.VoucherDateModel.Day, payment.VoucherDateModel.Hour, payment.VoucherDateModel.Minute, payment.VoucherDateModel.Second);
                    voucherDate = LocalUtilities.ParseDateToSQLString(payment.VoucherDate);
                }
                else
                {
                    voucherDate = "NULL";
                }
                payment.PaymentDate = paymentDateForAllTransactions;

                result = _employeeManager.GetEmployeeRelatedStaff(employeeId, dbHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepId, ref salesManagerId);
                organizationId = _requestRepository.Configurations.RestrictDataToCustomerOrganization ? transaction.CustomerOrganizationId :  _employeeManager.GetEmployeeOrganization(employeeId, dbHelper);

                query = string.Format(@"INSERT INTO CustomerPayment (CustomerID,OutletID,PaymentDate,CustomerPaymentID,TransactionID,PaymentTypeID,EmployeeID,VoucherNumber,VoucherDate ,
                    BankID,BranchID,Notes,CurrencyID,AppliedAmount,Synchronized,GPSLatitude,GPSLongitude,PaymentStatusID,Posted,RemainingAmount,SourceTransactionID,RouteHistoryID,VisitNo
                    ,AppliedPaymentID,VoucherOwner,AccountID,DivisionID,supervisorid,helperid,SalesManagerID,OrganizationID,DriverID,SalesRepID,SecondCurrencyID,SecondCurrencyAmount,
                     ExchangeRate,IsPrinted,IsTransRemainingAffected,AutoGeneratedByExchange,CreatedBy,CreatedDate,UpdatedBy,UpdatedDate)
                    VALUES ({0},{1},{2},'{3}','{4}',{5},{6},'{7}',{8},{9},{10},'{11}',{12},{13},{14},{15},{16},{17},{18},{19},'{20}',{21},{22},'{23}','{24}',{25},{26},{27},{28},{29},{30},{31},{32},{33},{34},{35},'{36}','{37}','{38}',{39},{40},{39},{40})",
                     transaction.CustomerId,
                     transaction.OutletId,
                     LocalUtilities.ParseDateAndTimeToSQL(payment.PaymentDate),
                     payment.CustomerPaymentId,
                     transaction.TransactionId,
                     payment.PaymentTypeId.GetHashCode(),
                     employeeId,
                     payment.VoucherNumber == string.Empty ? "NULL" : payment.VoucherNumber.Replace("'", "''"),
                     voucherDate,
                     (payment.BankId == -1 ? "NULL" : payment.BankId.ToString()),
                     (payment.BranchId == -1 ? "NULL" : payment.BranchId.ToString()),
                     (!string.IsNullOrEmpty(payment.Notes) ? payment.Notes.Replace("'", "''") : ""),
                     transaction.CurrencyId,
                     paidAmount,
                     0,
                     "NULL",
                     "NULL",
                     (int)payment.PaymentStatusId,
                     0,
                     transaction.RemainingAmount,
                     payment.SourceTransactionId = payment.PaymentTypeId.GetHashCode() == PaymentTypes.CreditNote.GetHashCode() ? payment.CreditNoteId.Trim() : payment.SourceTransactionId,
                     payment.RouteHistoryId,
                     "NULL",
                     payment.AppliedPaymentId,
                     payment.VoucherOwner,
                     accountId,
                     transaction.DivisionId,
                     (supervisorId == -1 ? "NULL" : supervisorId.ToString()),
                     (helperId == -1 ? "NULL" : helperId.ToString()),
                     (salesManagerId == -1 ? "NULL" : salesManagerId.ToString()),
                     (organizationId == -1 ? "NULL" : organizationId.ToString()),
                     (driverId == -1 ? "NULL" : driverId.ToString()),
                      (salesRepId == -1 ? "NULL" : salesRepId.ToString()),
                      payment.SecondCurrencyId,
                      payment.SecondCurrencyAmount,
                      payment.ExchangeRate,
                      payment.IsPrinted,
                      payment.IsTransRemainingAffected,
                     payment.AutoGeneratedByExchange,
                     _requestRepository.CurrentOperator.EmployeeId,
                     LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now));

                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors RemovePaidPayments(ref List<PaymentModel> paymentsList, int index)
        {
            GlobalErrors result = GlobalErrors.Error;
            try
            {
                PaymentModel[] remainingPayments;
                int startingIndex = 0;
                if (paymentsList[index].RemainingAmount > 0)
                    startingIndex = index;
                else
                    startingIndex = index + 1;
                if (paymentsList.Count - startingIndex > 0)
                {
                    remainingPayments = new PaymentModel[paymentsList.Count - startingIndex];
                    paymentsList.CopyTo(startingIndex, remainingPayments, 0, paymentsList.Count - startingIndex);
                    paymentsList.Clear();
                    paymentsList.AddRange(remainingPayments);
                }
                else
                {
                    paymentsList.Clear();
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors VoidOnlineEPayment(PaymentModel payment)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            string query = "";
            try
            {
                dbHelper = new DBHelper<int>();
                //IsTransRemainingAffected == 1 update transaction remaining amount else dont affect it
                if (payment.IsTransRemainingAffected)
                {
                    query = String.Format(@"Update [Transaction] Set RemainingAmount = [Transaction].RemainingAmount + CustomerPayment.AppliedAmount From [Transaction]
                                Inner Join CustomerPayment On CustomerPayment.CustomerID = [Transaction].CustomerID And CustomerPayment.OutletID = [Transaction].OutletID  And CustomerPayment.TransactionID = [Transaction].TransactionID and CustomerPayment.PaymentTypeID = 12
                                Inner Join PaymentsGroupIDs On PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID And PaymentsGroupIDs.OutletID = CustomerPayment.OutletID And PaymentsGroupIDs.RouteHistoryID = CustomerPayment.RouteHistoryID
                                And PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                                Where PaymentsGroupIDs.UUID = '{0}'   and CustomerPaymentID = '{1}' and CustomerPayment.IsTransRemainingAffected = 1", payment.UUID, payment.CustomerPaymentId);
                    result = dbHelper.ExecuteNonQuery(query);
                }
                if (result != GlobalErrors.Error)
                {
                    query = String.Format(@" Update CustomerPayment Set PaymentStatusID = 5 From CustomerPayment
                                Inner Join PaymentsGroupIDs On PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID And PaymentsGroupIDs.OutletID = CustomerPayment.OutletID And PaymentsGroupIDs.RouteHistoryID = CustomerPayment.RouteHistoryID And PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                                Where PaymentsGroupIDs.UUID = '{0}' and CustomerPaymentID = '{1}'", payment.UUID, payment.CustomerPaymentId);
                    result = dbHelper.ExecuteNonQuery(query);
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdateAccountForOnlineEpayment(decimal appliedAmount, int accountID, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string query = ""; 
            try
            {

                query = $@"Update Account Set Balance=Balance - {appliedAmount} Where AccountID = {accountID}";
                result = dbHelper.ExecuteNonQuery(query);
                if(result == GlobalErrors.Success)
                {
                    query = $@"Select ParentAccountID From Account Where AccountID = {accountID} And ParentAccountID Is Not Null And ParentAccountID > 0 And ParentAccountID <> {accountID}";
                    DataTable dt = new DataTable();
                    result = dbHelper.GetQueryDataTable(query, ref dt);
                    if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                    {
                        result = UpdateAccountForOnlineEpayment(appliedAmount, int.Parse(dt.Rows[0]["ParentAccountID"].ToString()), dbHelper);
                    }
                }  
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveEPaymentNoAuthResponse(string invoiceId, string chgId)
        {
            DBHelper<int> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<int>();
                string query = $@"Insert into EPaymentNoAuthResponse values ('{invoiceId}', '{chgId}')";
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors PostOnlineEpayment(PaymentModel payment, string ApprovalCode, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string query = ""; 
            try
            {
                query = $@"Select CustomerPayment.AccountID,Sum(CustomerPayment.AppliedAmount) AppliedAmount From CustomerPayment
                        Inner Join PaymentsGroupIDs On PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID And PaymentsGroupIDs.OutletID = CustomerPayment.OutletID And PaymentsGroupIDs.RouteHistoryID = CustomerPayment.RouteHistoryID and CustomerPayment.PaymentTypeID = 12
                        Where PaymentsGroupIDs.UUID = '{payment.UUID}' and PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                        Group By CustomerPayment.AccountID
                        ";
                DataTable dt = new DataTable();
                result = dbHelper.GetQueryDataTable(query, ref dt);
                if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                {
                     result = UpdateAccountForOnlineEpayment(decimal.Parse(dt.Rows[0]["AppliedAmount"].ToString()), int.Parse(dt.Rows[0]["AccountID"].ToString()), dbHelper);
                    if (result == GlobalErrors.Success && !payment.IsTransRemainingAffected) // and IsTransRemainingAffected == 0
                    { 
                        query = $@"Update [transaction] Set [transaction].RemainingAmount = [transaction].RemainingAmount - CustomerPayment.AppliedAmount  From [transaction]
                        Inner join CustomerPayment on CustomerPayment.TransactionID = [transaction].TransactionID and CustomerPayment.PaymentTypeID = 12 and CustomerPayment.CustomerPaymentID = '{payment.CustomerPaymentId}'
                        and CustomerPayment.CustomerID = [transaction].CustomerID and CustomerPayment.OutletID = [transaction].OutletID
                        Inner Join PaymentsGroupIDs On PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID And PaymentsGroupIDs.OutletID = CustomerPayment.OutletID And PaymentsGroupIDs.RouteHistoryID = CustomerPayment.RouteHistoryID And PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                        Where PaymentsGroupIDs.UUID = '{payment.UUID}' ";
                        result = dbHelper.ExecuteNonQuery(query);
                    }
                    if (result == GlobalErrors.Success)
                    { 
                        query = $@"Update CustomerPayment Set CustomerPayment.Posted = 1, CustomerPayment.IsTransRemainingAffected = 1, CustomerPayment.PaymentStatusID = 3 , CustomerPayment.AuthenticationCode = '{ApprovalCode}', CustomerPayment.ReceiptId = '{payment.ReceiptId}'  From CustomerPayment 
                        Inner Join PaymentsGroupIDs On PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID And PaymentsGroupIDs.OutletID = CustomerPayment.OutletID And PaymentsGroupIDs.RouteHistoryID = CustomerPayment.RouteHistoryID And PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                        Where PaymentsGroupIDs.UUID = '{payment.UUID}'  and CustomerPaymentID = '{payment.CustomerPaymentId}'";
                        result = dbHelper.ExecuteNonQuery(query);
                    }
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveOnlineEPaymentResponseForUpdateStatus(string ResponseDescription, string UniqueID, string ResponseCode, PaymentModel payment, string responseContent, string chg_id)
        {
            DBHelper<int> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string infoType = string.IsNullOrEmpty(chg_id) ? CoreDataBaseConstants.EPaymentLinkType.UpdateEPaymentLink : CoreDataBaseConstants.EPaymentLinkType.UpdateEPaymentChargeLink;
                dbHelper = new DBHelper<int>();
                string str = string.Format("Update OnlineEPaymentResponse set ResponeUniqueID = '{0}', ResponseCode = '{1}', ResponseContent = '{2}', ResponseDescription = '{3}', UUID = '{6}', Chg_Id = {7} where InfoType = '{4}' and InvoiceNumber = '{5}'",
                UniqueID, //0
                ResponseCode, //1
                responseContent, //2
                ResponseDescription, //3
                infoType, //4
                payment.ThirdPartyTransactionId, //5
                payment.UUID, //6
                !string.IsNullOrEmpty(chg_id) ? "'" + chg_id + "'" : "NULL" //7
                );

                result = dbHelper.ExecuteNoneQueryWithZeroRowAffectedCheck(str.ToString());
                if (result == GlobalErrors.Success)
                {
                    return result;
                }
                else if (result == GlobalErrors.SuccessWithZeroRowAffected)
                {
                    str = string.Format(@"insert into OnlineEPaymentResponse(ResponeUniqueID, ResponseCode, InvoiceNumber, ResponseContent, ResponseDescription, InfoType, UUID, Chg_Id) values ('{0}', '{1}', '{2}', '{3}', '{4}','{5}', '{6}', {7})",
                    UniqueID, ResponseCode, payment.ThirdPartyTransactionId, responseContent, ResponseDescription, infoType, payment.UUID, !string.IsNullOrEmpty(chg_id) ? "'" + chg_id + "'" : "NULL");
                    result = dbHelper.ExecuteNonQuery(str.ToString());
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetOnlineEPaymentsList(List<PaymentModel> paymentsList)
        {

            DBHelper<int> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            PaymentModel payment;
            try
            {
                dbHelper = new DBHelper<int>();

                ConfigurationManager _configurationManager = new ConfigurationManager(null);

                dynamic configValue = 0;
                int EPaymentCheckStatusLimit = 0;
                result = _configurationManager.GetConfigurationValueFromDB("EPaymentCheckStatusLimit", ref configValue);
                if (configValue != null && !string.IsNullOrEmpty(configValue.ToString()))
                {
                    EPaymentCheckStatusLimit = Int32.Parse(configValue.ToString());
                }

                string mainQuery = string.Format(@"select c.OrganizationID, c.PaymentTypeID, c.IsTransRemainingAffected, c.PaymentDate, c.CustomerID, c.OutletID, c.RouteHistoryID RouteHistoryId, c.CustomerPaymentID CustomerPaymentId,sum(c.AppliedAmount) AppliedAmount, pg.UUID, c.ThirdPartyTransactionId from CustomerPayment c
                inner join PaymentsGroupIDs pg on pg.PaymentID = c.CustomerPaymentID and isnull(pg.UUID, '') <> ''
                where isnull(c.ThirdPartyTransactionId, '') <> '' and c.posted <> 1 and C.PaymentStatusId not in (5)
                and pg.CustomerID = c.CustomerID and pg.OutletID = c.OutletID and c.PaymentDate >= {0}
                group by c.PaymentTypeID, c.CustomerPaymentID, pg.uuid, c.ThirdPartyTransactionId, c.RouteHistoryID , c.CustomerID, c.PaymentDate, c.OutletID, c.IsTransRemainingAffected, c.OrganizationID ",
                LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now.AddHours(EPaymentCheckStatusLimit * -1)));
                DataTable dt = new DataTable();
                result = dbHelper.GetQueryDataTable(mainQuery, ref dt);
                if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                {
                    for (int i = 0; i < dt.Rows.Count; i++)
                    {

                        payment = new PaymentModel();
                        payment.OrganizationId = int.Parse(dt.Rows[i]["OrganizationID"].ToString());
                        payment.RouteHistoryId = int.Parse(dt.Rows[i]["RouteHistoryId"].ToString());
                        payment.CustomerPaymentId = dt.Rows[i]["CustomerPaymentId"].ToString();
                        payment.UUID = dt.Rows[i]["UUID"].ToString();
                        payment.ThirdPartyTransactionId = dt.Rows[i]["ThirdPartyTransactionId"].ToString();
                        payment.AppliedAmount = decimal.Parse(dt.Rows[i]["AppliedAmount"].ToString());
                        payment.CustomerId = int.Parse(dt.Rows[i]["CustomerID"].ToString());
                        payment.OutletId = int.Parse(dt.Rows[i]["OutletID"].ToString());
                        payment.PaymentDate = DateTime.Parse(dt.Rows[i]["PaymentDate"].ToString());
                        payment.IsTransRemainingAffected = Boolean.Parse(dt.Rows[i]["IsTransRemainingAffected"].ToString());
                        paymentsList.Add(payment);
                    }
                }
                result = GlobalErrors.Success;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentsList = null;
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetPaymentsList(GeneralFilter filter, ref SharedTableResult<PaymentModel> paymentsList)
        {
            paymentsList = new SharedTableResult<PaymentModel>();
            List<PaymentModel> payments = new List<PaymentModel>();
            DBHelper<PaymentModel> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<PaymentModel>();
                string searchFilter = string.Empty;
                string orderByFilter = string.Empty;
                StringBuilder orgAccessJoin = new StringBuilder();
                if (_requestRepository.Configurations.ForceCustomerOrganizationSelection)
                {
                    orgAccessJoin.AppendLine(string.Format(CoreDataBaseConstants.CustOutAccountOrgAccessFilterQuery, "CustomerOutlet", _requestRepository.CurrentOperator.OrganizationAccess));
                }
                if (filter != null)
                {
                    if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                    {
                        searchFilter = string.Format(@"and ( 
                        ( CustomerPayment.CustomerPaymentID LIKE '%{0}%')
                        OR ( CustomerPayment.PaymentDate LIKE '%{0}%')
                        OR ( CustomerLanguage.Description LIKE '%{0}%' )
                        OR ( CustomerOutletLanguage.Description LIKE '%{0}%' )
                        OR ( Customer.CustomerCode LIKE '%{0}%' )
                        OR ( CustomerOutlet.CustomerCode LIKE '%{0}%' )
                        OR ( EmployeeLanguage.Description LIKE '%{0}%' )
                        OR ( Employee.EmployeeCode LIKE '%{0}%' ))", filter.CustomListFilter.SearchFilter.Trim());
                    }
                    if (filter.EmployeeId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.EmployeeID = {0})", filter.EmployeeId);
                    }
                    if (filter.CustomerId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.CustomerID= {0}) ", filter.CustomerId);
                    }
                    if (filter.OutletId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.OutletID= {0}) ", filter.OutletId);
                    }
                    if (filter.FromToDate != null && filter.FromToDate.Count == 2)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.PaymentDate >= {0} and CustomerPayment.PaymentDate <= {1}) "
                           ,LocalUtilities.ParseDateToSQLString(new DateTime(filter.FromToDate[0].Year, filter.FromToDate[0].Month, filter.FromToDate[0].Day)),
                            LocalUtilities.ParseEndDateToSQLString(new DateTime(filter.FromToDate[1].Year, filter.FromToDate[1].Month, filter.FromToDate[1].Day)));
                    }
                    if (filter.DivisionId > 0)
                    {
                        searchFilter += string.Format(@" And CustomerPayment.DivisionID = {0}", filter.DivisionId);
                    }
                    if (!filter.IncludeCashPayment)
                    {
                        searchFilter += string.Format(@"And ([Transaction].SalesMode = {0} Or [Transaction].TransactionTypeID = {1} Or [Transaction].SalesMode = {2})", SalesModes.CreditSales.GetHashCode(), TransactionType.DebitNote.GetHashCode(), SalesModes.TemporaryCreditSales.GetHashCode());
                    }
                    if (string.IsNullOrEmpty(filter.CustomListFilter.SortBy))
                    {
                        orderByFilter = string.Format(@"Order By CustomerPayment.PaymentDate desc");
                    }
                    else
                    {
                        orderByFilter = string.Format(@"Order By {0} {1} "
                        , filter.CustomListFilter.SortBy, filter.CustomListFilter.IsSortAscending ? "ASC" : "DESC");
                    }

                }
                string supervisionJoin = string.Empty;
                if (_requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Supervisor.GetHashCode() || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.SalesManager.GetHashCode()
                    || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Director.GetHashCode())
                {
                    supervisionJoin = string.Format(@"INNER JOIN CustOutTerritory on CustomerOutlet.CustomerID = CustOutTerritory.CustomerID and CustomerOutlet.OutletID = CustOutTerritory.OutletID
                    INNER JOIN EmployeeTerritory on EmployeeTerritory.TerritoryID = CustOutTerritory.TerritoryID and EmployeeTerritory.EmployeeId {0}
                    and CustomerPayment.EmployeeID {0}",
                    _employeeManager.GetSupervisorAccess());
                }
                //// Where ((CustomerPayment.PaymentTypeID = {6} And CustomerPayment.Posted <> 1 And VoucherDate > {7}) Or CustomerPayment.Posted = 1) And CustomerPayment.PaymentStatusID in ({2},{3},{4}) And ({8}.SalesMode = {5} Or {8}.TransactionTypeID = {9}) And (CustomerPayment.OrganizationID in ({10})) {1}
                //PaymentTypes.PostedDatedCheque.GetHashCode(), //6
                //LocalUtilities.ParseEndDateToSQLString(DateTime.Now), //7
                string mainQuery = string.Format(@"Select Distinct CustomerPayment.PaymentLink, CustomerPayment.Posted, CustomerPayment.IsTransRemainingAffected, PaymentsGroupIDs.UUId, CustomerPayment.ThirdPartyTransactionID, CustomerPayment.CustomerID, CustomerPayment.OutletID, CustomerPayment.EmployeeID,
                IsNull( CustomerLanguage.Description  , '--')       + ' ' + '-'+ ' ' + IsNull( Customer.CustomerCode , '--')as CustomerNameCode,
                IsNull( CustomerOutletLanguage.Description  , '--') + ' ' + '-'+ ' ' + IsNull( CustomerOutlet.CustomerCode , '--') as OutletNameCode,
                IsNull( EmployeeLanguage.Description  , '--') + ' ' + '-'+ ' ' + IsNull( Employee.EmployeeCode , '--') as EmployeeNameCode,
                IsNull( DivisionLanguage.Description  , '--') DivisionName, PaymentStatusLanguage.Description PaymentStatusName,
                CustomerPaymentID,CustomerPayment.PaymentDate,
                Sum(AppliedAmount)AppliedAmount,CustomerPayment.PaymentTypeID, PaymentTypeLanguage.Description PaymentTypeName,
                VoucherNumber,VoucherDate,
                CustomerPayment.BankID, CustomerPayment.BranchID, CustomerPayment.OrganizationID, CustomerPayment.CurrencyID,CustomerPayment.AccountID,
                BankLanguage.Description BankName,BankBranchLanguage.Description BranchName,
				CustomerPayment.PaymentStatusID, CustomerPayment.DivisionID,
                case when CustomerPayment.PaymentTypeID in (2,3,6) then 1 else 0 end as IsCheque,
                case when CustomerPayment.PaymentTypeID in (12) and CustomerPayment.Posted = 0 and CustomerPayment.PaymentStatusId <> 5 and IsNull(ThirdPartyTransactionID,'') != '' then 1 else 0 end as AllowCancelEPayment,
                case when CustomerPayment.PaymentTypeID in (12) and CustomerPayment.Posted = 0 and CustomerPayment.PaymentStatusId <> 5 and IsNull(ThirdPartyTransactionID,'') != '' then 1 else 0 end as AllowCopyEPaymentLink,
                VoidBtn.ShowVoidBtn
                From CustomerPayment 
                Inner Join [Transaction] On [Transaction].TransactionID = CustomerPayment.TransactionID And [Transaction].CustomerID = CustomerPayment.CustomerID And [Transaction].OutletID = CustomerPayment.OutletID And [Transaction].DivisionID = CustomerPayment.DivisionID
                Inner Join Customer On Customer.CustomerID = CustomerPayment.CustomerID
                Inner Join CustomerOutlet On CustomerOutlet.CustomerID = CustomerPayment.CustomerID And CustomerOutlet.OutletID = CustomerPayment.OutletID
                Left Join CustomerLanguage On CustomerLanguage.CustomerID = Customer.CustomerID And CustomerLanguage.LanguageID = {0}
                Left Join CustomerOutletLanguage On CustomerOutletLanguage.CustomerID = CustomerPayment.CustomerID And CustomerOutletLanguage.OutletID = CustomerPayment.OutletID And CustomerOutletLanguage.LanguageID = {0}
                Inner Join Employee on Employee.EmployeeID = CustomerPayment.EmployeeID
                Left Join EmployeeLanguage on EmployeeLanguage.EmployeeID = CustomerPayment.EmployeeID AND EmployeeLanguage.LanguageID = {0}
                left outer join RouteHistory on RouteHistory.RouteHistoryID = CustomerPayment.RouteHistoryID
                Left Join Division On Division.DivisionID = CustomerPayment.DivisionID
                Left Join DivisionLanguage On DivisionLanguage.DivisionID = Division.DivisionID And DivisionLanguage.LanguageID = {0}
                Left Join PaymentTypeLanguage On PaymentTypeLanguage.PaymentTypeID = CustomerPayment.PaymentTypeID And PaymentTypeLanguage.LanguageID = {0}
                Left Join PaymentStatusLanguage On PaymentStatusLanguage.PaymentStatusID = CustomerPayment.PaymentStatusID And PaymentStatusLanguage.LanguageID = {0}
                Left Join BankLanguage On BankLanguage.BankID = CustomerPayment.BankID And BankLanguage.LanguageID = {0}
                Left Join BankBranchLanguage On BankBranchLanguage.BankID = CustomerPayment.BankID And BankBranchLanguage.BranchID = CustomerPayment.BranchID And BankBranchLanguage.LanguageID = {0}
                left join PaymentsGroupIDs on PaymentsGroupIDs.CustomerID = CustomerPayment.CustomerID and PaymentsGroupIDs.OutletID = CustomerPayment.OutletID and PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID
                cross apply  (
                select top (1) rr.ShowVoidBtn from (
                select case when ([Transaction].TransactionTypeID <> 6 and [Transaction].CreationReason <> 13 and ([Transaction].SalesMode = 1 or CustomerPayment.PaymentTypeID = 12)) OR (CustomerPayment.PaymentTypeID = {10} AND {11} = 1)  THEN 0 ELSE 1 END AS ShowVoidBtn
                from [transaction] 
                where CustomerPayment.TransactionID = [Transaction].TransactionID and CustomerPayment.CustomerID = [Transaction].CustomerID and CustomerPayment.OutletID = [Transaction].OutletID and CustomerPayment.DivisionID = [Transaction].DivisionID
                )rr
                order by rr.ShowVoidBtn asc 
                ) VoidBtn 
                {7}
                {6}
                Where 1=1 {9} and ([Transaction].CreationReason <> {8}) {1} AND (RouteHistory.RouteHistoryID Is Null OR RouteHistory.Uploaded <> 1) AND CustomerPayment.PaymentStatusID in ({2},{3},{4}) And (CustomerPayment.OrganizationID in ({5}))
                Group By PaymentStatusLanguage.Description, CustomerPayment.PaymentLink, CustomerPayment.posted, CustomerPayment.IsTransRemainingAffected, PaymentsGroupIDs.UUID, CustomerPayment.ThirdPartyTransactionID, CustomerPayment.DivisionId, Division.DivisionCode, DivisionLanguage.Description, CustomerPayment.EmployeeID, CustomerPayment.OrganizationID, CustomerPayment.CurrencyID, CustomerPayment.AccountID, CustomerPayment.BankID, CustomerPayment.BranchID, EmployeeLanguage.Description,Employee.EmployeeCode,CustomerPayment.CustomerID,CustomerPayment.OutletID,CustomerLanguage.Description,Customer.CustomerCode,CustomerOutletLanguage.Description ,CustomerOutlet.CustomerCode,CustomerPaymentID,CustomerPayment.PaymentDate,CustomerPayment.PaymentTypeID,PaymentTypeLanguage.Description ,VoucherNumber,VoucherDate,BankLanguage.Description,
                BankBranchLanguage.Description, CustomerPayment.PaymentStatusID, CustomerPayment.DivisionID,VoidBtn.ShowVoidBtn",
                _requestRepository.LanguageId, //0
                searchFilter, //1
                PaymentStatusTypes.Pending.GetHashCode(), //2
                PaymentStatusTypes.FullyCollected.GetHashCode(), //3
                PaymentStatusTypes.Approved.GetHashCode(), //4
                _requestRepository.CurrentOperator.OrganizationAccess, //5
                orgAccessJoin, //6
                supervisionJoin, //7
                TransactionCreationReason.DiscrepancyInvoice.GetHashCode(), //8
                _requestRepository.Configurations.ControlTransactionOnDivisionLevel ? string.Format(" And Division.DivisionId in ({0})", _requestRepository.CurrentOperator.DivisionAccess) : string.Empty, //9
                (int)PaymentTypes.CreditCard, //10
                _requestRepository.Configurations.UseOnlineCreditCard ? 1 : 0 //11
                );

                string paymentsListQuery = string.Format(@" {0} {1} OFFSET {2} ROWS FETCH NEXT {3} ROWS ONLY",
                mainQuery, //0
                orderByFilter, //1
                filter.CustomListFilter.Page * filter.CustomListFilter.PageSize, //2
                filter.CustomListFilter.PageSize //3
                );

                string countQuery = string.Format(@"Select IsNull(Count(*),0) from ({0})tt", mainQuery);

                // get count
                object objField = 0;
                result = dbHelper.ExecuteScalar(countQuery, ref objField);
                if (result == GlobalErrors.Success && objField != null && !string.IsNullOrEmpty(objField.ToString()))
                {
                    paymentsList.TotalItems = int.Parse(objField.ToString().Trim());
                    if (paymentsList.TotalItems <= 0)
                    {
                        paymentsList.Data = payments;
                        return GlobalErrors.Success;
                    }
                }
                else
                {
                    return GlobalErrors.Error;
                }

                result = dbHelper.GetQueryList(paymentsListQuery, ref payments);
                paymentsList.Data = payments;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentsList = null;
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetPaymentsListPayerLevel(GeneralFilter filter, string payerCodeName, ref SharedTableResult<PaymentModel> paymentsList)
        {
            paymentsList = new SharedTableResult<PaymentModel>();
            List<PaymentModel> payments = new List<PaymentModel>();
            DBHelper<PaymentModel> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<PaymentModel>();
                string searchFilter = string.Empty;
                string orderByFilter = string.Empty;
                StringBuilder orgAccessJoin = new StringBuilder();
                
                if (filter != null)
                {
                    if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                    {
                        searchFilter = string.Format(@"and ( 
                        ( CustomerPayment.CustomerPaymentID LIKE '%{0}%')
                        OR ( CustomerPayment.PaymentDate LIKE '%{0}%')
                        OR ( EmployeeLanguage.Description LIKE '%{0}%' )
                        OR ( Employee.EmployeeCode LIKE '%{0}%' ))", filter.CustomListFilter.SearchFilter.Trim());
                    }
                    if (filter.EmployeeId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.EmployeeID = {0})", filter.EmployeeId);
                    }
                    if (filter.FromToDate != null && filter.FromToDate.Count == 2)
                    {
                        searchFilter += string.Format(" And (CustomerPayment.PaymentDate >= {0} and CustomerPayment.PaymentDate <= {1}) "
                           , LocalUtilities.ParseDateToSQLString(new DateTime(filter.FromToDate[0].Year, filter.FromToDate[0].Month, filter.FromToDate[0].Day)),
                            LocalUtilities.ParseEndDateToSQLString(new DateTime(filter.FromToDate[1].Year, filter.FromToDate[1].Month, filter.FromToDate[1].Day)));
                    }
                    if (filter.DivisionId > 0)
                    {
                        searchFilter += string.Format(@" And CustomerPayment.DivisionID = {0}", filter.DivisionId);
                    }
                    if (string.IsNullOrEmpty(filter.CustomListFilter.SortBy))
                    {
                        orderByFilter = string.Format(@"Order By CustomerPayment.PaymentDate desc");
                    }
                    else
                    {
                        orderByFilter = string.Format(@"Order By {0} {1} " , filter.CustomListFilter.SortBy, filter.CustomListFilter.IsSortAscending ? "ASC" : "DESC");
                    }
                }
                string supervisionJoin = string.Empty;
                if (_requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Supervisor.GetHashCode() || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.SalesManager.GetHashCode()
                    || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Director.GetHashCode())
                {
                    supervisionJoin = string.Format(@" and CustomerPayment.EmployeeID {0} ", _employeeManager.GetSupervisorAccess());
                }

                string mainQuery = string.Format(@"Select CustomerPayment.PaymentLink, CustomerPayment.IsTransRemainingAffected, PayGroups.UUId, CustomerPayment.ThirdPartyTransactionID, CustomerPayment.EmployeeID,
                IsNull( EmployeeLanguage.Description  , '--') + ' ' + '-'+ ' ' + IsNull( Employee.EmployeeCode , '--') as EmployeeNameCode,
                IsNull( DivisionLanguage.Description  , '--') DivisionName, CustomerPayment.CustomerPaymentID, CustomerPayment.PaymentDate,
                CustomerPayment.PaymentTypeID, PaymentTypeLanguage.Description PaymentTypeName, VoucherNumber,VoucherDate,
                CustomerPayment.BankID, CustomerPayment.BranchID, CustomerPayment.OrganizationID, CustomerPayment.CurrencyID,CustomerPayment.AccountID,
				CustomerPayment.PaymentStatusID, CustomerPayment.DivisionID, PaymentStatusLanguage.Description PaymentStatusName,
                case when CustomerPayment.PaymentTypeID in (2,3,6) then 1 else 0 end as IsCheque,
                case when CustomerPayment.PaymentTypeID in (12) and CustomerPayment.Posted = 0 and CustomerPayment.PaymentStatusId <> 5 and IsNull(ThirdPartyTransactionID,'') != '' then 1 else 0 end as AllowCancelEPayment,
                case when CustomerPayment.PaymentTypeID in (12) and CustomerPayment.Posted = 0 and CustomerPayment.PaymentStatusId <> 5 and IsNull(ThirdPartyTransactionID,'') != '' then 1 else 0 end as AllowCopyEPaymentLink,
                '{13}' PayerCodeName, Sum(AppliedAmount)AppliedAmount
                From CustomerPayment 
                Cross Apply (
                    Select top (1) CustomerPaymentId, CustomerPayment.DivisionID From CustomerPayment cp
                    Inner Join [Transaction] On [Transaction].TransactionID = cp.TransactionID And [Transaction].CustomerID = cp.CustomerID And [Transaction].OutletID = cp.OutletID And [Transaction].DivisionID = cp.DivisionID
                    And ([Transaction].SalesMode in ({14},{15}) Or [Transaction].TransactionTypeID = {16}) {17}
                    where cp.CustomerPaymentId = CustomerPayment.CustomerPaymentId and cp.DivisionId = CustomerPayment.DivisionId
                )trans
                Inner Join Employee on Employee.EmployeeID = CustomerPayment.EmployeeID
                Left Join EmployeeLanguage on EmployeeLanguage.EmployeeID = CustomerPayment.EmployeeID AND EmployeeLanguage.LanguageID = {0}
                left outer join RouteHistory on RouteHistory.RouteHistoryID = CustomerPayment.RouteHistoryID
                Left Join PaymentTypeLanguage On PaymentTypeLanguage.PaymentTypeID = CustomerPayment.PaymentTypeID And PaymentTypeLanguage.LanguageID = {0}
                Left Join PaymentStatusLanguage On PaymentStatusLanguage.PaymentStatusID = CustomerPayment.PaymentStatusID And PaymentStatusLanguage.LanguageID = {0}
                Left Join Division On Division.DivisionID = CustomerPayment.DivisionID
                Left Join DivisionLanguage On DivisionLanguage.DivisionID = Division.DivisionID And DivisionLanguage.LanguageID = {0}
                outer apply (Select top (1) UUID From PaymentsGroupIDs Where PaymentsGroupIDs.PaymentID = CustomerPayment.CustomerPaymentID)PayGroups
                Where CustomerPayment.AccountId in ({12}) {1} {6} {7} {9} AND (RouteHistory.RouteHistoryID Is Null OR RouteHistory.Uploaded <> 1) AND CustomerPayment.PaymentStatusID in ({2},{3},{4}) And (CustomerPayment.OrganizationID in ({5}))
                Group By PaymentStatusLanguage.Description, CustomerPayment.PaymentLink, CustomerPayment.IsTransRemainingAffected, PayGroups.UUID, CustomerPayment.ThirdPartyTransactionID, CustomerPayment.Posted, Division.DivisionCode, DivisionLanguage.Description, CustomerPayment.EmployeeID, CustomerPayment.OrganizationID, CustomerPayment.CurrencyID, CustomerPayment.AccountID, CustomerPayment.BankID, CustomerPayment.BranchID, EmployeeLanguage.Description,Employee.EmployeeCode,CustomerPayment.CustomerPaymentID,CustomerPayment.PaymentDate,CustomerPayment.PaymentTypeID,PaymentTypeLanguage.Description ,VoucherNumber,VoucherDate,
                CustomerPayment.PaymentStatusID, CustomerPayment.DivisionID",
                _requestRepository.LanguageId, //0
                searchFilter, //1
                PaymentStatusTypes.Pending.GetHashCode(), //2
                PaymentStatusTypes.FullyCollected.GetHashCode(), //3
                PaymentStatusTypes.Approved.GetHashCode(), //4
                _requestRepository.CurrentOperator.OrganizationAccess, //5
                orgAccessJoin, //6
                supervisionJoin, //7
                TransactionCreationReason.DiscrepancyInvoice.GetHashCode(), //8
                _requestRepository.Configurations.ControlTransactionOnDivisionLevel ? string.Format(" And Division.DivisionId in ({0})", _requestRepository.CurrentOperator.DivisionAccess) : string.Empty, //9
                (int)PaymentTypes.CreditCard, //10
                _requestRepository.Configurations.UseOnlineCreditCard ? 1 : 0, //11
                filter.AccountId, //12
                payerCodeName, //13
                SalesModes.CreditSales.GetHashCode(), //14
                SalesModes.TemporaryCreditSales.GetHashCode(),  //15
                TransactionType.DebitNote.GetHashCode(), //16
                filter.ShowForOutletOnly ? $" And [Transaction].CustomerId = {filter.CustomerId} and [Transaction].OutletId = {filter.OutletId} " : "" //17
                );

                string paymentsListQuery = string.Format(@" {0} {1} OFFSET {2} ROWS FETCH NEXT {3} ROWS ONLY",
                mainQuery, //0
                orderByFilter, //1
                filter.CustomListFilter.Page * filter.CustomListFilter.PageSize, //2
                filter.CustomListFilter.PageSize //3
                );

                string countQuery = string.Format(@"Select IsNull(Count(*),0) from ({0})tt", mainQuery);

                // get count
                object objField = 0;
                result = dbHelper.ExecuteScalar(countQuery, ref objField , true);
                if (result == GlobalErrors.Success && objField != null && !string.IsNullOrEmpty(objField.ToString()))
                {
                    paymentsList.TotalItems = int.Parse(objField.ToString().Trim());
                    if (paymentsList.TotalItems <= 0)
                    {
                        paymentsList.Data = payments;
                        return GlobalErrors.Success;
                    }
                }
                else
                {
                    return GlobalErrors.Error;
                }

                result = dbHelper.GetQueryList(paymentsListQuery, ref payments , true);
                paymentsList.Data = payments;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentsList = null;
                return GlobalErrors.Error;
            }
            return result;
        }
        public bool CheckIfPaymentVoided(PaymentModel payment, DBHelper<int> dBHelper)
        {
            try
            {
                string query = string.Format(@"Select Top 1 PaymentStatusID From CustomerPayment Where CustomerID = {0} And OutletID = {1} and DivisionId = {3} And CustomerPaymentID = '{2}'", payment.CustomerId, payment.OutletId, payment.CustomerPaymentId, payment.DivisionId);
                object statusID = new object();
                dBHelper.ExecuteScalar(query, ref statusID);
                if (statusID != null && statusID != DBNull.Value && statusID.ToString() != string.Empty)
                {
                    if (int.Parse(statusID.ToString()) == PaymentStatusTypes.Voided.GetHashCode() || int.Parse(statusID.ToString()) == PaymentStatusTypes.Bounced.GetHashCode())
                        return true;
                    else
                        return false;
                }
                else
                {
                    return true;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return true;
            }
        }
        public bool CheckIfMainPaymentVoided(PaymentModel payment, DBHelper<int> dBHelper)
        {
            try
            {
                string query = string.Format(@"Select Top 1 PaymentStatusID From CustomerPayment Where DivisionId = {1} And CustomerPaymentID = '{0}'",  payment.CustomerPaymentId, payment.DivisionId);
                object statusID = new object();
                dBHelper.ExecuteScalar(query, ref statusID);
                if (statusID != null && statusID != DBNull.Value && statusID.ToString() != string.Empty)
                {
                    if (int.Parse(statusID.ToString()) == PaymentStatusTypes.Voided.GetHashCode() || int.Parse(statusID.ToString()) == PaymentStatusTypes.Bounced.GetHashCode())
                        return true;
                    else
                        return false;
                }
                else
                {
                    return true;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return true;
            }
        }
        public bool CheckIfPaymentBounced(PaymentModel payment, DBHelper<int> dBHelper)
        {
            try
            {
                string query = string.Format(@"Select Top 1 PaymentStatusID From CustomerPayment Where CustomerID = {0} And OutletID = {1} And CustomerPaymentID = '{2}'", payment.CustomerId, payment.OutletId, payment.CustomerPaymentId);
                object statusID = new object();
                dBHelper.ExecuteScalar(query, ref statusID);
                if (statusID != null && statusID != DBNull.Value && statusID.ToString() != string.Empty)
                {
                    if (int.Parse(statusID.ToString()) == PaymentStatusTypes.Bounced.GetHashCode() || int.Parse(statusID.ToString()) == PaymentStatusTypes.Bounced.GetHashCode())
                        return true;
                    else
                        return false;
                }
                else
                {
                    return true;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return true;
            }
        }
        public GlobalErrors GetPaymentTransactions(PaymentModel payment, DBHelper<PaymentTransaction> dbHelperTransactions, ref List<PaymentTransaction> paymentTransactions, bool excludeCustomerCheck = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelperTransactions = new DBHelper<PaymentTransaction>();
                string query = string.Format(@"select CustomerPayment.CustomerPaymentID, CustomerPayment.TransactionID,CustomerPayment.CurrencyID,
                CustomerPayment.SourceTransactionID, CustomerPayment.AppliedAmount,
                CustomerPayment.PaymentTypeID, CustomerPayment.Posted,
                [Transaction].EmployeeID TransactionEmployeeId, [Transaction].SalesMode, [Transaction].TransactionDate
				,CustomerPayment.TransactionID InvoiceNo, [Transaction].{5} TransactionIdInUI, [Transaction].DueDate,
                CustomerPayment.PaymentTypeID,CustomerPayment.VoucherNumber,CustomerPayment.VoucherDate,CustomerPayment.Notes,    
                CustomerPayment.AppliedAmount PaidAmount,CustomerPayment.PaymentStatusID,     
                CustomerPayment.RemainingAmount RemainingAmount,CustomerPayment.AppliedPaymentID,CustomerPayment.DivisionID,CustomerPayment.OrganizationID,  
                CustomerPayment.CollectionDiscount,Customer.CustomerID ,Customer.CustomerCode,CustomerOutlet.OutletID,    
                CustomerOutlet.CustomerCode AS CustomerOutletCode,CustomerLanguage.Description AS CustomerName,CustomerOutletLanguage.Description AS OutletName, 
                Employee.EmployeeID,Employee.EmployeeCode,EmployeeLanguage.Description AS EmployeeName,Division.DivisionID , 
                Division.DivisionCode,DivisionLanguage.Description AS DivisionName,PaymentTypeLanguage.PaymentTypeID,
                PaymentTypeLanguage.Description PaymentType,BankLanguage.Description BankName,BankBranchLanguage.Description BranchName,   
                CustomerPayment.VoucherNumber ChequeNo,CustomerPayment.VoucherDate ChequeDate ,  IsNull(CustomerLanguage.Description, '--')     + ' ' + '-'+ ' ' +  IsNull( Customer.CustomerCode, '--')as CustomerNameCode,
                IsNull(CustomerOutletLanguage.Description, '--')+ ' ' + '-'+ ' ' +    IsNull( CustomerOutlet.CustomerCode , '--')as OutletNameCode,
				IsNull(EmployeeLanguage.Description,'--') as EmployeeName,
                CustomerPayment.PaymentDate
                From CustomerPayment
                Inner join [Transaction] on [Transaction].TransactionID = CustomerPayment.TransactionID And [Transaction].CustomerID = CustomerPayment.CustomerID And [Transaction].OutletID = CustomerPayment.OutletID And [Transaction].DivisionID = CustomerPayment.DivisionID
                INNER JOIN Customer ON Customer.CustomerID = CustomerPayment.CustomerID 
                INNER JOIN CustomerOutlet ON CustomerOutlet.CustomerID = CustomerPayment.CustomerID AND CustomerOutlet.OutletID = CustomerPayment.OutletID 
                LEFT OUTER JOIN CustomerLanguage ON CustomerLanguage.CustomerID = Customer.CustomerID AND CustomerLanguage.LanguageID = {4}
                LEFT OUTER JOIN CustomerOutletLanguage ON CustomerOutletLanguage.CustomerID = CustomerPayment.CustomerID AND CustomerOutletLanguage.OutletID = CustomerPayment.OutletID AND CustomerOutletLanguage.LanguageID = {4} 
                INNER JOIN Employee ON Employee.EmployeeID = CustomerPayment.EmployeeID 
                LEFT OUTER JOIN EmployeeLanguage ON EmployeeLanguage.EmployeeID = CustomerPayment.EmployeeID AND EmployeeLanguage.LanguageID = {4}
                LEFT OUTER JOIN Division ON Division.DivisionID = CustomerPayment.DivisionID 
                LEFT OUTER JOIN DivisionLanguage ON DivisionLanguage.DivisionID = Division.DivisionID AND DivisionLanguage.LanguageID = {4} 
                LEFT OUTER JOIN PaymentTypeLanguage ON PaymentTypeLanguage.PaymentTypeID = CustomerPayment.PaymentTypeID AND PaymentTypeLanguage.LanguageID = {4}
                left join BankLanguage on BankLanguage.BankID = CustomerPayment.BankID and  BankLanguage.LanguageID = {4} 
                left join BankBranchLanguage  on BankBranchLanguage.BankID = CustomerPayment.BankID and BankBranchLanguage.BranchID = CustomerPayment.BranchID and  BankBranchLanguage.LanguageID = {4} 
                Where CustomerPayment.CustomerPaymentID = '{2}' And CustomerPayment.DivisionID = {3} {6}",
                payment.CustomerId,//0
                payment.OutletId,//1
                payment.CustomerPaymentId,//2
                payment.DivisionId,//3
                _requestRepository.LanguageId, //4
                _requestRepository.Configurations.UseSourceTransactionIDAsInvoiceName ? "SourceTransactionID" : "TransactionID", //5
                excludeCustomerCheck ? "" : $" and CustomerPayment.CustomerID = {payment.CustomerId} And CustomerPayment.OutletID = {payment.OutletId} "//6
                );
                result = dbHelperTransactions.GetQueryList(query, ref paymentTransactions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveOnlineEPayments(SavePaymentModel savePaymentModel, ref string errorMessage, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<string> addedPayments = new List<string>();
            bool isSameCustomer = false;
            try
            {
                Dictionary<string, int> savedPayments = new Dictionary<string, int>();

                string maxCustomerPaymentID = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, savePaymentModel.EmployeeId, DocumentTypes.Collection, savePaymentModel.DivisionId);
                string maxAppliedPaymentID = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, savePaymentModel.EmployeeId, DocumentTypes.AppliedPayment, savePaymentModel.DivisionId);

                if (string.IsNullOrEmpty(maxCustomerPaymentID))
                {
                    savePaymentModel.ValidationErrors = ValidationErrors.NoDocumentSequence;
                    errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Document_Sequence_Payment, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (string.IsNullOrEmpty(maxAppliedPaymentID))
                {
                    savePaymentModel.ValidationErrors = ValidationErrors.NoDocumentSequence;
                    errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Document_Sequence_Applied_Payment, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                int visitedCustomerId = savePaymentModel.CustomerId;
                int visitedOutledId = savePaymentModel.OutletId;
                DateTime paymentDateForAllTransactions = DateTime.Now; // to save payment on all transactions with same date
                string UUID = Guid.NewGuid().ToString();
                int sequence = GetMaxSequenceID("Sequence", "PaymentsGroupIDs", dbHelper);
                if(sequence <= 0)
                {
                    return GlobalErrors.Error;
                }
                savePaymentModel.PaymentsList.ForEach(x =>
                {
                    x.CustomerPaymentId = maxCustomerPaymentID;
                    x.AppliedPaymentId = maxAppliedPaymentID;
                    x.UUID = UUID;
                });
                foreach (TransactionModel transaction in savePaymentModel.TransactionsList)
                {
                    if (transaction.RemainingAmount > 0 && transaction.AppliedAmount > 0 && savePaymentModel.PaymentsList.Count > 0)
                    {
                        transaction.ValuesUsed = transaction.AppliedAmount; // we need this to send e-payment link
                        string key = maxCustomerPaymentID + "_" + transaction.CustomerId + "_" + transaction.OutletId;
                        if (!addedPayments.Contains(key))
                        {
                            addedPayments.Add(key);
                            isSameCustomer = false;
                            if (!isSameCustomer)
                            {
                                result = savePaymentsGroup(maxCustomerPaymentID, transaction.CustomerId, transaction.OutletId, transaction.DivisionId, UUID, sequence, visitedCustomerId, visitedOutledId, dbHelper);
                                if (result != GlobalErrors.Success)
                                {
                                    return result;
                                }
                            }
                        }
                        else
                        {
                            isSameCustomer = true;
                        }
                        result = SaveEPayments(dbHelper, transaction, savePaymentModel.PaymentsList, savePaymentModel.EmployeeId, UUID, sequence, paymentDateForAllTransactions);

                        if (result != GlobalErrors.Success)
                        {
                            break;
                        }
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    if (_documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.Collection, maxCustomerPaymentID, savePaymentModel.EmployeeId, savePaymentModel.DivisionId) != GlobalErrors.Success)
                    {
                        return GlobalErrors.Error;
                    }
                    if (_documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.AppliedPayment, maxAppliedPaymentID, savePaymentModel.EmployeeId, savePaymentModel.DivisionId) != GlobalErrors.Success)
                    {
                        return GlobalErrors.Error;
                    }
                }
              
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

            return result;
        }
        public static int GetMaxSequenceID(string columnName, string tableName, DBHelper<int> dBHelper)
        {
            try
            {
                if (dBHelper == null)
                    dBHelper = new DBHelper<int>();
                object max = null;
                string query = string.Format(@"select isnull(max({0}),0)+1 from {1} where RouteHistoryID = -1  ", columnName, tableName);
                GlobalErrors result = dBHelper.ExecuteScalar(query, ref max);
                if (result == GlobalErrors.Success && max != null && !string.IsNullOrEmpty(max.ToString()))
                {
                    return int.Parse(max.ToString());
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return -1;
            }
            return 1;
        }
        private GlobalErrors SaveEPayments(DBHelper<int> dbHelper, TransactionModel transaction, List<PaymentModel> paymentsList, int employeeId, string UUID, int sequence, DateTime paymentDateForAllTransactions)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                for (int i = 0; i < paymentsList.Count; i++)
                {
                    if (transaction.RemainingAmount > 0 && (transaction.AppliedAmount > 0 || _requestRepository.Configurations.AllowPayingExtraAtCollection))
                    {
                        decimal paidAmount = transaction.AppliedAmount;
                        transaction.RemainingAmount -= paidAmount;
                        transaction.AppliedAmount = 0;
                        paymentsList[i].RouteHistoryId = -1;
                        paymentsList[i].OrganizationId = _requestRepository.Configurations.RestrictDataToCustomerOrganization ? transaction.CustomerOrganizationId : _requestRepository.CurrentOperator.OrganizationId;
                        paymentsList[i].CurrencyId = transaction.CurrencyId;
                        paymentsList[i].PaymentStatusId = PaymentStatusTypes.Pending;
                        paymentsList[i].CustomerId = transaction.CustomerId;
                        paymentsList[i].OutletId = transaction.OutletId;

                        result = InsertPayment(dbHelper, transaction, paymentsList[i], paidAmount, transaction.AccountId, int.Parse(employeeId.ToString()), paymentDateForAllTransactions);

                        if (result == GlobalErrors.Success)
                        {
                            result = _accountManager.UpdateRemainingAmountAndIsTransactionAffected(dbHelper, paymentsList[i].CustomerPaymentId, transaction.TransactionId, transaction.CustomerId, transaction.OutletId, transaction.DivisionId, paidAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                            if(result == GlobalErrors.Success)
                            {
                                paymentsList[i].IsTransRemainingAffected = true;
                            }
                        }
                        else
                        {
                            break;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors savePaymentsGroup(string customerPaymentId, int customerId, int outletId, int divisionId, string UUID, int sequence, int visitedCustomerId, int visitedOutletId, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format(@" insert into PaymentsGroupIDs (sequence,PaymentID,CustomerID,OutletID,DivisionID,RouteHistoryID, UUID, VisitedCustomerID, VisitedOutletID) values ({0},'{1}',{2},{3},{4},{5}, '{6}', {7}, {8})",
                        sequence,
                        customerPaymentId,
                        customerId,
                        outletId,
                        divisionId,
                        -1,
                        UUID,
                        visitedCustomerId,
                        visitedOutletId);
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
//        public GlobalErrors SendPaymentLink(PaymentModel payment, List<OnlineEPaymentResponseModel> responseModels, DBHelper<int> dbHelper)
//        {
//            GlobalErrors result = GlobalErrors.NotInitialized;
//            dynamic paymentGateWay = "";
//            try
//            {
//                CustomerModel visitedCustomer = new CustomerModel();
//                Dictionary<int, string> currencies = new Dictionary<int, string>();
//                result = FillCurrencies(currencies);
//                if (result == GlobalErrors.Success)
//                {
//                    result = FillPaymentVisitedCustomer(visitedCustomer, payment.UUID, dbHelper);
//                }
//                if (result == GlobalErrors.Success)
//                {
//                    paymentGateWay = _requestRepository.Configurations.PaymentGateway;
//                }
//                else
//                {
//                    return result;
//                }
//                var client = new RestClient(paymentGateWay);
//                var request = new RestRequest();
//                request.AddHeader("Accept", " application/json");
//                request.AddHeader("Content-Type", "application/json");
//                DateTime paymentDate = payment.PaymentDate;
//                DateTime paymentExpiryDate = payment.PaymentDate.AddHours(_requestRepository.Configurations.OnlineEPaymentExpiryDateLimit);
//                var body = @"{
//" + "\n" +
//                @"    ""GenerateEInvoice"": {
//" + "\n" +
//                $@"        ""Customer"": ""{_requestRepository.Configurations.CompanyName}"",
//" + "\n" +
//                $@"        ""Store"": ""{_requestRepository.Configurations.EPaymentStore}"",
//" + "\n" +
//                $@"        ""Terminal"": ""{_requestRepository.Configurations.EPaymentTerminal}"",
//" + "\n" +
//                $@"        ""OrderID"": ""{payment.CustomerPaymentId}"",
//" + "\n" +
//                $@"        ""OrderName"": ""BRF {payment.CustomerPaymentId + visitedCustomer.CustomerId}"",
//" + "\n" +
//                $@"        ""OrderInfo"": ""BRF {payment.UUID}"",
//" + "\n" +
//                $@"        ""Amount"": ""{payment.AppliedAmount.ToString()}"",
//" + "\n" +
//                $@"        ""Currency"": ""{ currencies[payment.CurrencyId] }"",
//" + "\n" +
//                @"        ""OverrideCapture"": ""Auto"",
//" + "\n" +
//                @"        ""RegisterForRecurrence"": ""False"",
//" + "\n" +
//                $@"        ""EffectiveStartDate"": ""{paymentDate.Year + "-" + paymentDate.Month + "-" + paymentDate.Day + " " + paymentDate.Hour + ":" + paymentDate.Minute + ":" + paymentDate.Second}"",
//" + "\n" +
//                $@"        ""ExpiryDate"": ""{paymentExpiryDate.Year + "-" + paymentExpiryDate.Month + "-" + paymentExpiryDate.Day + " " + paymentExpiryDate.Hour + ":" + paymentExpiryDate.Minute + ":" + paymentExpiryDate.Second}"",
//" + "\n" +
//                @"        ""InvoiceType"": ""Once"",
//" + "\n" +
//                $@"        ""CardHolderName"": ""{visitedCustomer.CustomerName}"",
//" + "\n" +
//                $@"        ""CardHolderEmail"": ""{visitedCustomer.Email}"",
//" + "\n" +
//                $@"        ""CardHolderMobile"": ""{visitedCustomer.Phone}"",
//" + "\n" +
//                $@"        ""MerchantMessage"": ""{_requestRepository.Configurations.MerchantMessage}"",
//" + "\n" +
//                $@"        ""UserName"": ""{_requestRepository.Configurations.PaymentGateWayUserName}"",
//" + "\n" +
//                $@"        ""Password"": ""{_requestRepository.Configurations.PaymentGateWayPassword}""
//" + "\n" +
//                @"    }
//" + "\n" +
//                @"}";
//                //var body = JsonConvert.SerializeObject(new test2(payment, customer, _requestRepository.Configurations.OnlineEPaymentExpiryDateLimit));
//                request.AddParameter("application/json", body, ParameterType.RequestBody);
//                request.Timeout = 50000; //Time in milliseconds 
//                var response = client.ExecutePost(request);
//                OnlineEPaymentResponseModel responseModel = new OnlineEPaymentResponseModel();
//                JObject jObject = JObject.Parse(response.Content);
//                JToken jTransaction = jObject["Transaction"];
//                responseModel.UniqueID = jTransaction["UniqueID"] == null ? "" : (string)jTransaction["UniqueID"];
//                responseModel.ResponseCode = jTransaction["ResponseCode"] == null ? -1 : (int)jTransaction["ResponseCode"];
//                responseModel.ResponseDescription = jTransaction["ResponseDescription"] == null ? "" : (string)jTransaction["ResponseDescription"];
//                responseModel.ActivityLogingSequenceNumber = jTransaction["ActivityLogingSequenceNumber"] == null ? -1 : (int)jTransaction["ActivityLogingSequenceNumber"];
//                responseModel.InvoiceNumber = jTransaction["InvoiceNumber"] == null ? "" : (string)jTransaction["InvoiceNumber"];
//                responseModel.InvoiceURL = jTransaction["InvoiceURL"] == null ? "" : (string)jTransaction["InvoiceURL"];
//                responseModel.NotificationResponse = jTransaction["NotificationResponse"] == null ? "" : (string)jTransaction["NotificationResponse"];
//                responseModel.Language = jTransaction["Language"] == null ? "" : (string)jTransaction["Language"];
//                responseModel.ResponseClass = jTransaction["ResponseClass"] == null ? -1 : (int)jTransaction["ResponseClass"];
//                responseModel.ResponseClassDescription = jTransaction["ResponseClassDescription"] == null ? "" : (string)jTransaction["ResponseClassDescription"];
//                responseModel.UUID = payment.UUID;
//                responseModel.ResponseContent = response.Content;
//                payment.ThirdPartyTransactionId = responseModel.InvoiceNumber;
//                if (SaveOnlineEPaymentResponse(responseModel, dbHelper, payment) != GlobalErrors.Success)
//                {
//                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "E-Payment Request :  ", null, body, 0);
//                    ResourcesManager.TranslateKey("Failed to save Online e-payment respone" + "PaymentID => [ " + payment.CustomerPaymentId + " ] UUID => [ " /*+ payment.UUID */+ " ]" + "Response content => [ " + response.Content + " ]", _requestRepository.LanguageId);
//                }
//                //Console.WriteLine(response.Content);
//                responseModels.Add(responseModel);
//                if (!string.IsNullOrEmpty(responseModel.InvoiceNumber))
//                    result = AddPaymentsThirdPartTransactionIDs(payment, dbHelper);
//                //you should update customerPayment set ThirdPartTransactionId incase link sent successfully

//            }
//            catch (Exception ex)
//            {
//                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
//                return GlobalErrors.Error;
//            }
//            return result;
//        }
        public GlobalErrors SendPaymentLink(SavePaymentModel savePaymentModel)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                CustomerModel visitedCustomer = new CustomerModel();
                result = FillPaymentVisitedCustomer(visitedCustomer, savePaymentModel.PaymentsList[0].UUID);
                if (result != GlobalErrors.Success)
                {
                    return result;
                }
                
                if (_requestRepository.Configurations.OnlineEPaymentExpiryDateLimit == 0)
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Missing Configuration Value : OnlineEPaymentExpiryDateLimit, For Organization " + savePaymentModel.PaymentsList[0].OrganizationId, null, "", 0);
                    return GlobalErrors.Error;
                }
                string PaymentGateWayPassword = string.Empty;
                string EPaymentCountryNote = string.Empty;
                string EPaymentCurrency = string.Empty;

                var config = ConfigurationManager.GetConfigurationOnOrganization("EPaymentCurrency", savePaymentModel.PaymentsList[0].OrganizationId);
                if (config != null && !string.IsNullOrEmpty(config.KeyValue))
                {
                    EPaymentCurrency = config.KeyValue.Trim().ToString();
                }
                if (string.IsNullOrEmpty(EPaymentCurrency) )
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Missing Configuration Value : EPaymentCurrency, For Organization " + savePaymentModel.PaymentsList[0].OrganizationId, null, "", 0);
                    return GlobalErrors.Error;
                }

                config = ConfigurationManager.GetConfigurationOnOrganization("EPaymentCountryNote", savePaymentModel.PaymentsList[0].OrganizationId);
                if (config != null && !string.IsNullOrEmpty(config.KeyValue))
                {
                    EPaymentCountryNote = config.KeyValue.Trim().ToString();
                }
                if (string.IsNullOrEmpty(EPaymentCountryNote))
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Missing Configuration Value : EPaymentCountryNote, For Organization " + savePaymentModel.PaymentsList[0].OrganizationId, null, "", 0);
                    return GlobalErrors.Error;
                }

                config = ConfigurationManager.GetConfigurationOnOrganization("PaymentGateWayPassword", savePaymentModel.PaymentsList[0].OrganizationId);
                if (config != null && !string.IsNullOrEmpty(config.KeyValue))
                {
                    PaymentGateWayPassword = config.KeyValue.Trim().ToString();
                }
                if (string.IsNullOrEmpty(PaymentGateWayPassword))
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Missing Configuration Value : PaymentGateWayPassword, For Organization " + savePaymentModel.PaymentsList[0].OrganizationId, null, "", 0);
                    return GlobalErrors.Error;
                }

                if (string.IsNullOrEmpty("PaymentGateway"))
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Missing Configuration Value : PaymentGateway", null, "", 0);
                    return GlobalErrors.Error;
                }
                string currencyCode = EPaymentCurrency;
                DateTime startDate = new DateTime(1970, 1, 1);
                DateTime dueDate = DateTime.UtcNow.AddHours(_requestRepository.Configurations.OnlineEPaymentExpiryDateLimit);
                double dueDateEpoc = (dueDate - startDate).TotalSeconds * 1000;
                StringBuilder invoicesString = new StringBuilder();
                int index = 0;
                string caption = string.Empty;
                foreach (var trans in savePaymentModel.TransactionsList)
                {
                    caption = (_requestRepository.Configurations.UseSourceTransactionIDAsInvoiceName ? trans.SourceTransactionId : trans.TransactionId) + " (Due on " + trans.DueDate.ToString("yyyy/MM/dd") + ")";
                    if (index > 0)
                    {
                        invoicesString.Append(",");
                    }
                    invoicesString.AppendLine(@"{
                    " + "\n" + $@"""amount"": {trans.ValuesUsed},
                    " + "\n" + $@"""currency"": ""{currencyCode}"",
                    " + "\n" + $@"""name"": ""{caption}"",
                    " + "\n" + $@"""quantity"": 1" + "}");
                    index++;
                }
                List<string> notificationArray = new List<string>();
                bool shareLink = false;
                string phoneString = string.Empty;
                if (savePaymentModel.NotifyBySMS && savePaymentModel.CustomerContact != null && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.Number) && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.PrefixCode))
                {
                    shareLink = true;
                    notificationArray.Add("\"" + "SMS" + "\"");
                   
                    phoneString = @"        ,""phone"": {
                    " + "\n" +
                                            $@"            ""country_code"": ""{savePaymentModel.CustomerContact.PrefixCode}"",
                    " + "\n" +
                                            $@"            ""number"": ""{savePaymentModel.CustomerContact.Number}""
                    " + "\n" +
                                        @"        }";
                }
                if (savePaymentModel.NotifyByEmail && savePaymentModel.CustomerContact != null && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.ContactEmail))
                {
                    shareLink = true;
                    notificationArray.Add("\"" +"EMAIL" + "\"");
                }
                else
                {
                    savePaymentModel.NotifyByEmail = true;
                    shareLink = true;
                    notificationArray.Add("\"" + "EMAIL" + "\"");
                    savePaymentModel.CustomerContact.ContactEmail = "info@brf.com";
                }
                string notifictionString = string.Join(",", notificationArray);

                var options = new RestClientOptions(_requestRepository.Configurations.PaymentGateway)
                {
                    MaxTimeout = 30000,
                };
                var client = new RestClient(options);
                var request = new RestRequest("/v2/invoices", Method.Post);
                request.AddHeader("Content-Type", "application/json");
                request.AddHeader("Authorization", PaymentGateWayPassword);
                var body = @"{
                " + "\n" +
                                @"    ""draft"": false,
                 " + "\n" +
                                $@"    ""due"": {dueDateEpoc},
                " + "\n" +
                                $@"    ""expiry"": {dueDateEpoc},
                " + "\n" +
                                @"    ""mode"": ""INVOICE"",
                " + "\n" +
                                @"    ""savecard"": false,
                " + "\n" +
                                $@"    ""note"": "" {EPaymentCountryNote} "",
                " + "\n" +
                                @"    ""notifications"": {
                " + "\n" +
                                $@"        ""channels"": [{notifictionString}],
                " + "\n" +
                                $@"        ""dispatch"": {(shareLink ? "true" : "false")}
                " + "\n" +
                                @"    },
                " + "\n" +
                                @"    ""currencies"": [
                " + "\n" +
                                $@"        ""{currencyCode}""
                " + "\n" +
                                @"    ],
                " + "\n" +
                                @"    ""metadata"": {
                " + "\n" +
                                $@"        ""udf1"": ""{savePaymentModel.PaymentsList[0].UUID}"",
                " + "\n" +
                                @"        ""udf2"": """",
                " + "\n" +
                                @"        ""udf3"": """"
                " + "\n" +
                                @"    },
                " + "\n" +
                                @"    ""charge"": {
                " + "\n" +
                                @"        ""receipt"": {
                " + "\n" +
                                $@"            ""email"": {(savePaymentModel.NotifyByEmail && savePaymentModel.CustomerContact != null && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.ContactEmail) ? "true" : "false")},
                " + "\n" +
                                $@"            ""sms"":   {(savePaymentModel.NotifyBySMS && savePaymentModel.CustomerContact != null && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.Number) && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.PrefixCode) ? "true" : "false")}
                " + "\n" +
                                @"        }
                " + "\n" +
                                @"    },
                " + "\n" +
                                @"    ""customer"": {
                " + "\n" +
                                $@"        ""email"": ""{(savePaymentModel.NotifyByEmail && savePaymentModel.CustomerContact != null && !string.IsNullOrEmpty(savePaymentModel.CustomerContact.ContactEmail) ? savePaymentModel.CustomerContact.ContactEmail : "")}"",
                " + "\n" +
                                $@"        ""first_name"": ""{visitedCustomer.CustomerName}""
                " + "\n" +
                                $@"       {phoneString}
                " + "\n" +
                                @"    },
                " + "\n" +
                                @"    ""order"": {
                " + "\n" +
                                $@"        ""amount"": {savePaymentModel.PaymentsList[0].AppliedAmount},
                " + "\n" +
                                $@"        ""currency"": ""{currencyCode}"",
                " + "\n" +
                                @"        ""items"": [
                " + "\n" +
                                $@"   {invoicesString}      
                " + "\n" +
                                @"        ]
                " + "\n" +
                                @"    }
                " + "\n" +
                                @"}";
                request.AddStringBody(body, DataFormat.Json);
                RestResponse response = client.ExecutePost(request);
                OnlineEPaymentResponseModel responseModel = new OnlineEPaymentResponseModel();
                responseModel.ResponseCode = response.StatusCode.GetHashCode();
                responseModel.ResponseDescription = response.StatusDescription == null ? "" : (string)response.StatusDescription;
                responseModel.ResponseContent = response.Content;
                responseModel.UUID = savePaymentModel.PaymentsList[0].UUID;

                if (response.StatusCode == System.Net.HttpStatusCode.OK)
                {
                    JObject jObject = JObject.Parse(response.Content);
                    responseModel.InvoiceURL = jObject["url"] == null ? "" : (string)jObject["url"];
                    responseModel.InvoiceNumber = jObject["id"] == null ? "" : (string)jObject["id"];
                    responseModel.Language = jObject["lang_code"] == null ? "" : (string)jObject["lang_code"];
                    responseModel.UniqueID = responseModel.InvoiceNumber;
                    savePaymentModel.PaymentsList[0].ThirdPartyTransactionId = responseModel.InvoiceNumber;
                    savePaymentModel.PaymentsList[0].PaymentLink = responseModel.InvoiceURL;
                }
                else
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Send E-Payment Link Failed , UUID : " + savePaymentModel.PaymentsList[0].UUID, null, body, 0);
                }

                result = ReflectOnlineEPayment(responseModel, savePaymentModel, body);
                if (result != GlobalErrors.Success || response.StatusCode != System.Net.HttpStatusCode.OK)
                {
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ReflectOnlineEPayment(OnlineEPaymentResponseModel response, SavePaymentModel savePaymentModel, string body)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {

                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                if (!string.IsNullOrEmpty(response.InvoiceNumber) && response.ResponseCode == HttpStatusCode.OK.GetHashCode())
                {
                    result = AddPaymentsThirdPartTransactionIDs(savePaymentModel.PaymentsList[0], dbHelper);
                    if (result != GlobalErrors.Success)
                    {
                        return result;
                    }
                }
                result = SaveOnlineEPaymentResponse(response, savePaymentModel.PaymentsList[0], dbHelper); 
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    if (dbHelper != null)
                    {
                        dbHelper.Dispose();
                        dbHelper = null;
                    }
                }
            }
            return result;
        }
        public GlobalErrors FillCurrencies(Dictionary<int, string> currencies)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DataTable dt = null;
            string query = "";
            DBHelper<int> dbHelper = new DBHelper<int>();
            try
            {
                query = $" select * from CurrencyLanguage where LanguageID = {_requestRepository.LanguageId}";
                result = dbHelper.GetQueryDataTable(query, ref dt);
                if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                {
                    for (int i = 0; i < dt.Rows.Count; i++)
                    {
                        currencies.Add(Int16.Parse(dt.Rows[i]["CurrencyID"].ToString()), dt.Rows[i]["Description"].ToString());
                    }
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors FillPaymentsSum(List<PaymentModel> payments, Dictionary<string, decimal> FillPaymentsSum)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            StringBuilder query = new StringBuilder();
            int appliedAmount = 0;
            try
            {

                for (int i = payments.Count - 1; i >= 0; i--)
                {
                    if (FillPaymentsSum.Keys.Contains(payments.ElementAt(i).UUID))
                    {
                        FillPaymentsSum[payments.ElementAt(i).UUID] = FillPaymentsSum[payments.ElementAt(i).UUID] + payments.ElementAt(i).AppliedAmount;
                        if (payments.Where(p => p.UUID == payments.ElementAt(i).UUID).ToList().Count > 1)
                            payments.RemoveAt(i);
                        else
                            payments.ElementAt(i).AppliedAmount = FillPaymentsSum[payments.ElementAt(i).UUID];
                    }
                    else
                    {
                        FillPaymentsSum.Add(payments.ElementAt(i).UUID, payments.ElementAt(i).AppliedAmount);
                        if (payments.Where(p => p.UUID == payments.ElementAt(i).UUID).ToList().Count > 1)
                        {

                            payments.RemoveAt(i);

                        }
                    }

                }
                result = GlobalErrors.Success;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors FillPaymentVisitedCustomer(CustomerModel customer, string UUID)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string Value = string.Empty;
            DataTable dt = null;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                string query = string.Format(@" select distinct p.UUID, col.Description CustomerName, co.CustomerCode CustomerCode, co.Email CustomerEmail, co.Phone CustomerPhone,co.CustomerID from CustomerOutlet co
                    inner join PaymentsGroupIDs p on p.VisitedCustomerID = co.CustomerID and p.VisitedOutletID = co.OutletID and p.UUID in ('{0}')
                    inner join CustomerOutletLanguage col on col.CustomerID = p.VisitedCustomerID and col.OutletID = p.VisitedOutletID and col.LanguageID = {1} ", UUID, _requestRepository.LanguageId);
                result = dbHelper.GetQueryDataTable(query, ref dt);
                if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                {
                    Value = String.Empty;

                    if (!string.IsNullOrEmpty(dt.Rows[0]["UUID"].ToString()))
                    {

                        UUID = dt.Rows[0]["UUID"].ToString();
                    }
                    if (!string.IsNullOrEmpty(dt.Rows[0]["CustomerName"].ToString()))
                    {

                        customer.CustomerName = dt.Rows[0]["CustomerName"].ToString();
                    }
                    if (!string.IsNullOrEmpty(dt.Rows[0]["CustomerCode"].ToString()))
                    {

                        customer.CustomerCode = dt.Rows[0]["CustomerCode"].ToString();
                    }
                    if (!string.IsNullOrEmpty(dt.Rows[0]["CustomerEmail"].ToString()))
                    {

                        customer.Email = dt.Rows[0]["CustomerEmail"].ToString();
                    }
                    if (!string.IsNullOrEmpty(dt.Rows[0]["CustomerPhone"].ToString()))
                    {

                        customer.Phone = dt.Rows[0]["CustomerPhone"].ToString();
                    }
                    if (!string.IsNullOrEmpty(dt.Rows[0]["CustomerID"].ToString()))
                    {

                        customer.CustomerId = int.Parse(dt.Rows[0]["CustomerID"].ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveOnlineEPaymentResponse(OnlineEPaymentResponseModel responseModel, PaymentModel payment, DBHelper<int> dbHelper, bool isCancel = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            StringBuilder str = new StringBuilder();
            try
            {
                str = str.AppendFormat(@"insert into OnlineEPaymentResponse(UUID, ResponeUniqueID, ResponseCode, ResponseDescription, ActivityLogingSequenceNumber, 
                    InvoiceNumber, InvoiceURL, NotificationResponse, [Language], ResponseClass, ResponseClassDescription, CreateddDate, RouteHistoryID, ResponseContent, InfoType) values ('{0}', '{1}', '{2}', '{3}', '{4}', '{5}', '{6}', '{7}', '{8}', '{9}', '{10}', {11}, '{12}', '{13}','{14}')",
                    responseModel.UUID, responseModel.UniqueID, responseModel.ResponseCode.ToString(), responseModel.ResponseDescription, responseModel.ActivityLogingSequenceNumber.ToString(),
                    responseModel.InvoiceNumber, responseModel.InvoiceURL, responseModel.NotificationResponse, responseModel.Language, responseModel.ResponseClass.ToString(), responseModel.ResponseClassDescription,
                    LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now), payment.RouteHistoryId == 0 ? "-1" : payment.RouteHistoryId.ToString(), responseModel.ResponseContent, isCancel ? CoreDataBaseConstants.EPaymentLinkType.CancelEPaymentLink : CoreDataBaseConstants.EPaymentLinkType.SendEPaymentLink);
                result = dbHelper.ExecuteNonQuery(str.ToString());
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors AddPaymentsThirdPartTransactionIDs(PaymentModel payment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            StringBuilder query = new StringBuilder();
            try
            {

                query = query.AppendFormat(@" update customerpayment set thirdpartytransactionid = '{0}', PaymentLink = '{2}' where customerpaymentid = '{1}'",
                                           payment.ThirdPartyTransactionId.ToString(), payment.CustomerPaymentId, payment.PaymentLink);
                result = dbHelper.ExecuteNonQuery(query.ToString());
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {

            }
            return result;
        }
        #region[Down Payments]
        public GlobalErrors GetDownPaymentsList(GeneralFilter filter, ref SharedTableResult<PaymentModel> downPaymentsSharedList)
        {
            downPaymentsSharedList = new SharedTableResult<PaymentModel>();
            List<PaymentModel> downPaymentsList = new List<PaymentModel>();
            DBHelper<PaymentModel> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<PaymentModel>();
                string searchFilter = string.Empty;
                string orderByFilter = string.Empty;
                StringBuilder orgAccessJoin = new StringBuilder();
                if (_requestRepository.Configurations.ForceCustomerOrganizationSelection)
                {
                    orgAccessJoin.AppendLine(string.Format(CoreDataBaseConstants.CustOutAccountOrgAccessFilterQuery, "CustomerOutlet", _requestRepository.CurrentOperator.OrganizationAccess));
                }
                if (filter != null)
                {
                    if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                    {
                        searchFilter = string.Format(@"and ( 
                        ( CustomerUnallocatedPayment.CustomerPaymentID LIKE '%{0}%')
                        OR ( CustomerUnallocatedPayment.PaymentDate LIKE '%{0}%')
                        OR ( CustomerLanguage.Description LIKE '%{0}%' )
                        OR ( CustomerOutletLanguage.Description LIKE '%{0}%' )
                        OR ( Customer.CustomerCode LIKE '%{0}%' )
                        OR ( CustomerOutlet.CustomerCode LIKE '%{0}%' )
                        OR ( EmployeeLanguage.Description LIKE '%{0}%' )
                        OR ( Employee.EmployeeCode LIKE '%{0}%' ))", filter.CustomListFilter.SearchFilter.Trim());
                    }
                    if (filter.EmployeeId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerUnallocatedPayment.EmployeeID = {0})", filter.EmployeeId);
                    }
                    if (filter.CustomerId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerUnallocatedPayment.CustomerID= {0}) ", filter.CustomerId);
                    }
                    if (filter.OutletId != -1)
                    {
                        searchFilter += string.Format(" And (CustomerUnallocatedPayment.OutletID= {0}) ", filter.OutletId);
                    }
                    if (!string.IsNullOrEmpty(filter.TransactionId))
                    {
                        searchFilter += string.Format(" And (CustomerUnallocatedPayment.CustomerPaymentID like('%{0}%')) ", filter.TransactionId);
                    }
                    if (filter.DateModel != null)
                    {
                        searchFilter += string.Format(" And (CustomerUnallocatedPayment.PaymentDate >= {0} and CustomerUnallocatedPayment.PaymentDate <= {1}) "
                            , LocalUtilities.ParseDateToSQLString(new DateTime(filter.DateModel.Year, filter.DateModel.Month, filter.DateModel.Day)),
                            LocalUtilities.ParseEndDateToSQLString(new DateTime(filter.DateModel.Year, filter.DateModel.Month, filter.DateModel.Day)));
                    }
                    if (filter.DivisionId > 0)
                    {
                        searchFilter += string.Format(@" And CustomerUnallocatedPayment.DevisionID = {0}", filter.DivisionId);
                    }
                    if (string.IsNullOrEmpty(filter.CustomListFilter.SortBy))
                    {
                        orderByFilter = string.Format(@"Order By CustomerUnallocatedPayment.PaymentDate desc");
                    }
                    else
                    {
                        orderByFilter = string.Format(@"Order By {0} {1} ",
                        filter.CustomListFilter.SortBy,
                        filter.CustomListFilter.IsSortAscending ? "ASC" : "DESC");
                    }
                }
                string supervisionJoin = string.Empty;
                if (_requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Supervisor.GetHashCode() || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.SalesManager.GetHashCode()
                    || _requestRepository.CurrentOperator.EmployeeTypeId == EmployeeTypes.Director.GetHashCode())
                {
                    supervisionJoin = string.Format(@"INNER JOIN CustOutTerritory on CustomerOutlet.CustomerID = CustOutTerritory.CustomerID and CustomerOutlet.OutletID = CustOutTerritory.OutletID
                    INNER JOIN EmployeeTerritory on EmployeeTerritory.TerritoryID = CustOutTerritory.TerritoryID and EmployeeTerritory.EmployeeId {0}
                    and CustomerUnallocatedPayment.EmployeeID {0}",
                    _employeeManager.GetSupervisorAccess());
                }
                string mainQuery = string.Format(@"Select Distinct CustomerUnallocatedPayment.DevisionID As DivisionID ,  CustomerUnallocatedPayment.PaidAmount AppliedAmount,CustomerUnallocatedPayment.Voided,
                CustomerUnallocatedPayment.CustomerID,CustomerUnallocatedPayment.OutletID,CustomerUnallocatedPayment.EmployeeID,
                IsNull( CustomerLanguage.Description  , '--')       + ' ' + '-'+ ' ' + IsNull( Customer.CustomerCode , '--')as CustomerNameCode,
                IsNull( CustomerOutletLanguage.Description  , '--') + ' ' + '-'+ ' ' + IsNull( CustomerOutlet.CustomerCode , '--') as OutletNameCode,
                IsNull( EmployeeLanguage.Description  , '--') + ' ' + '-'+ ' ' + IsNull( Employee.EmployeeCode , '--') as EmployeeNameCode,
                IsNull( DivisionLanguage.Description  , '--') DivisionName,
                CustomerUnallocatedPayment.CustomerPaymentID,CustomerUnallocatedPayment.PaymentDate,
                CustomerUnallocatedPayment.PaidAmount,CustomerUnallocatedPayment.PaymentTypeID, PaymentTypeLanguage.Description PaymentTypeName,
                VoucherNumber,VoucherDate,
                CustomerUnallocatedPayment.BankID, CustomerUnallocatedPayment.BranchID, CustomerUnallocatedPayment.OrganizationID, CustomerUnallocatedPayment.CurrencyID,
                BankLanguage.Description BankName,BankBranchLanguage.Description BranchName,
                Case CustomerUnallocatedPayment.Voided when 1 then '{5}' else '---' end as PaymentStatusName
                From CustomerUnallocatedPayment
                Inner Join Customer On Customer.CustomerID = CustomerUnallocatedPayment.CustomerID
                Left Join CustomerLanguage On CustomerLanguage.CustomerID = Customer.CustomerID And CustomerLanguage.LanguageID = {0}
                Inner Join CustomerOutlet On CustomerOutlet.CustomerID = CustomerUnallocatedPayment.CustomerID And CustomerOutlet.OutletID = CustomerUnallocatedPayment.OutletID
                Left Join CustomerOutletLanguage On CustomerOutletLanguage.CustomerID = CustomerUnallocatedPayment.CustomerID And CustomerOutletLanguage.OutletID = CustomerUnallocatedPayment.OutletID And CustomerOutletLanguage.LanguageID = {0}
                Left Join PaymentTypeLanguage On PaymentTypeLanguage.PaymentTypeID = CustomerUnallocatedPayment.PaymentTypeID And PaymentTypeLanguage.LanguageID = {0}
                Left Join BankLanguage On BankLanguage.BankID = CustomerUnallocatedPayment.BankID And BankLanguage.LanguageID = {0}
                Left Join BankBranchLanguage On BankBranchLanguage.BankID = CustomerUnallocatedPayment.BankID And BankBranchLanguage.BranchID = CustomerUnallocatedPayment.BranchID And BankBranchLanguage.LanguageID = {0}
                Inner Join Employee On Employee.EmployeeID = CustomerUnallocatedPayment.EmployeeID
                Left Join EmployeeLanguage On EmployeeLanguage.EmployeeID = CustomerUnallocatedPayment.EmployeeID And EmployeeLanguage.LanguageID = {0}
                Left Join Division On Division.DivisionID = CustomerUnallocatedPayment.DevisionID
                Left Join DivisionLanguage On DivisionLanguage.DivisionID = Division.DivisionID and DivisionLanguage.LanguageID  = {0}
                {6}
                {4}
                Where (CustomerUnallocatedPayment.OrganizationID in ({2})) {7}  {3}",
                _requestRepository.LanguageId, //0
                _requestRepository.CurrentOperator.EmployeeId, //1,
                _requestRepository.CurrentOperator.OrganizationAccess, //2
                searchFilter, //3
                orgAccessJoin, //4
                ResourcesManager.TranslateKey(MessagesConstants.Desc_Voided_Caption, _requestRepository.LanguageId), //5
                supervisionJoin, //6
                _requestRepository.Configurations.ControlTransactionOnDivisionLevel ? string.Format(" And Division.DivisionId in ({0})", _requestRepository.CurrentOperator.DivisionAccess) : string.Empty //7
                );

                string paymentsSelectionQuery = string.Format(@" {0} {1} OFFSET {2} ROWS FETCH NEXT {3} ROWS ONLY",
                mainQuery, //0
                orderByFilter, //1
                (filter.CustomListFilter.Page) * filter.CustomListFilter.PageSize, //2
                filter.CustomListFilter.PageSize //3
                );

                string countQuery = string.Format(@"Select IsNull(Count(*),0) From ({0}) tt", mainQuery);

                object objField = 0;
                result = dbHelper.ExecuteScalar(countQuery, ref objField , true);
                if (result == GlobalErrors.Success && objField != null && !string.IsNullOrEmpty(objField.ToString()))
                {
                    downPaymentsSharedList.TotalItems = int.Parse(objField.ToString().Trim());
                    if (downPaymentsSharedList.TotalItems <= 0)
                    {
                        downPaymentsSharedList.Data = downPaymentsList;
                        return GlobalErrors.Success;
                    }
                }
                else
                {
                    return GlobalErrors.Error;
                }

                result = dbHelper.GetQueryList(paymentsSelectionQuery, ref downPaymentsList , true);
                downPaymentsSharedList.Data = downPaymentsList;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                downPaymentsSharedList = null;
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveUnallocatedPayment(PaymentModel payment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string voucherDate = string.Empty;
                if (payment.PaymentTypeId == PaymentTypes.PostedDatedCheque || payment.PaymentTypeId == PaymentTypes.CurrentDatedCheque || payment.PaymentTypeId == PaymentTypes.CertifiedCheque)
                {
                    payment.VoucherDate = new DateTime(payment.VoucherDateModel.Year, payment.VoucherDateModel.Month, payment.VoucherDateModel.Day, payment.VoucherDateModel.Hour, payment.VoucherDateModel.Minute, payment.VoucherDateModel.Second);
                    voucherDate = LocalUtilities.ParseDateToSQLString(payment.VoucherDate);
                }
                else
                {
                    voucherDate = "NULL";
                }
                payment.PaymentDate = DateTime.Now;

                string query = string.Format(@"INSERT INTO CustomerUnallocatedPayment (CustomerID,OutletID,CustomerPaymentID,DevisionID,PaymentDate,PaymentTypeID
                ,EmployeeID,VoucherDate,VoucherNumber,VoucherOwner,BankID,BranchID,Notes,CurrencyID,Synchronised,GPSLatitude
                ,GPSLangitude,PaidAmount,Downloaded,VisitNo,RouteHistoryID,HelperID,SupervisorID,SalesManagerID,OrganizationID,DriverID,SalesRepID,Voided)
                VALUES ({0},{1},'{2}',{3},{4},{5},{6},{7},'{8}','{9}',{10},{11},'{12}',{13},{14},{15},{16},{17},{18},{19},{20},{21},{22},{23},{24},{25},{26},{27})",
                payment.CustomerId, //0
                payment.OutletId, //1
                payment.CustomerPaymentId, //2
                payment.DivisionId, //3
                LocalUtilities.ParseDateAndTimeToSQL(payment.PaymentDate), //4
                (int)payment.PaymentTypeId, //5
                payment.EmployeeId, //6
                voucherDate, //7
                string.IsNullOrEmpty(payment.VoucherNumber) ? "NULL" : payment.VoucherNumber.Replace("'", "''"), //8
                string.IsNullOrEmpty(payment.VoucherOwner) ? "NULL" : payment.VoucherOwner.Replace("'", "''"), //9
                (payment.BankId == -1 ? "NULL" : payment.BankId.ToString()), //10
                (payment.BranchId == -1 ? "NULL" : payment.BranchId.ToString()), //11
                string.IsNullOrEmpty(payment.Notes) ? "NULL" : payment.Notes.Replace("'", "''"), //12
                payment.CurrencyId, //13
                0, //14 Synchronised
                "NULL", //15 GPSLatitude
                "NULL", //16 GPSLangitude
                payment.AppliedAmount, //17 
                0, //18 Downloaded
                "NULL", //19 VisitNo
                -1, //20 RouteHistoryID
                (payment.HelperId == -1 ? "NULL" : payment.HelperId.ToString()), //21
                (payment.SupervisorId == -1 ? "NULL" : payment.SupervisorId.ToString()), //22
                (payment.SalesManagerId == -1 ? "NULL" : payment.SalesManagerId.ToString()), //23
                (payment.OrganizationId == -1 ? "NULL" : payment.OrganizationId.ToString()), //24
                (payment.DriverId == -1 ? "NULL" : payment.DriverId.ToString()), //25
                (payment.SalesRepId == -1 ? "NULL" : payment.SalesRepId.ToString()), //26
                0 //27 voided
                );
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors VoidDownPayment(PaymentModel payment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format(@" Update CustomerUnallocatedPayment Set Voided = 1
                Where CustomerPaymentID = '{0}' And CustomerID = {1} And OutletID = {2} And EmployeeID = {3} and DevisionID = {4}",
                payment.CustomerPaymentId,//0
                payment.CustomerId,//1
                payment.OutletId, //2
                payment.EmployeeId, //3
                payment.DivisionId //4
                );
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public bool CheckUploadedCreditNote(int customerID, int outletID, DateTime creditNoteDate, DBHelper<int> dBHelper)
        {
            bool isCNUploaded = false;

            try
            {
                string filter = string.Empty;
                object field = new object();

                if (_requestRepository.Configurations.AllowTerritoryCustomerVisit)
                {
                    filter = " inner join CustOutTerritory on RouteHistory.TerritoryID = CustOutTerritory.TerritoryID ";
                }
                else
                {
                    filter = " inner join RouteCustomer on RouteHistory.RouteID = RouteCustomer.RouteID ";
                }

                string query = string.Format(@"select Top(1) Uploaded from RouteHistory
                {0}
                where CustomerID = {1} and OutletID = {2} and  SyncDate >= {3}  order by RouteHistoryID desc ",
                filter, customerID, outletID, LocalUtilities.ParseDateAndTimeToSQL(creditNoteDate));
                GlobalErrors result = dBHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != null && !string.IsNullOrEmpty(field.ToString()))
                {
                    isCNUploaded = Convert.ToBoolean(field.ToString());
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return isCNUploaded;
        }
        public bool CheckIfCreditNoteUsedInPayment(string creditNoteTransID, int divisionID, DBHelper<int> dbHelper)
        {
            bool isUsed = false;
            try
            {
                object field = new object();
                string query = string.Format("SELECT count(*) from CustomerPayment Where SourceTransactionID ='{0}' and DivisionID = {2} and PaymentStatusID not in ({1})",
                creditNoteTransID,
                PaymentStatusTypes.Voided.GetHashCode(),
                divisionID
                );
                GlobalErrors result = dbHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != null && !string.IsNullOrEmpty(field.ToString()) && int.Parse(field.ToString()) > 0)
                {
                    isUsed = true;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return isUsed;
        }
        public bool IsDownPaymentSynchronized(PaymentModel downPayment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            object field = null;
            bool isSync = false;
            try
            {
                string query = string.Format(@"Select Synchronised From CustomerUnallocatedPayment 
                Where CustomerPaymentID = '{0}' and CustomerID = {1} and outletID = {2} and DevisionID = {3} ",
                downPayment.CustomerPaymentId,
                downPayment.CustomerId,
                downPayment.OutletId,
                downPayment.DivisionId);

                result = dbHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != DBNull.Value && !string.IsNullOrEmpty(field.ToString()))
                {
                    isSync = Convert.ToBoolean(field);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return isSync;
        }
        public bool IsDownPaymentVoided(PaymentModel downPayment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            object field = null;
            bool isVoided = false;
            try
            {
                string query = string.Format(@"Select Voided From CustomerUnallocatedPayment 
                Where CustomerPaymentID = '{0}' and CustomerID = {1} and outletID = {2} and DevisionID = {3}",
                downPayment.CustomerPaymentId,
                downPayment.CustomerId,
                downPayment.OutletId,
                downPayment.DivisionId);
                result = dbHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != DBNull.Value && !string.IsNullOrEmpty(field.ToString()))
                {
                    isVoided = Convert.ToBoolean(field);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return isVoided;
        }
        #endregion

        #region [Posting Payments]

        public GlobalErrors PostCustomerPayments(int employeeID, string customerPaymentID, int customerID, int outletID, bool isManualPosting, DBHelper<int> dbHelper, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            SalesModes _salesMode = SalesModes.None;
            try
            {
                StringBuilder sqlStatement = new StringBuilder();
                sqlStatement.AppendFormat(@"SELECT CustomerPayment.CustomerID,CustomerPayment.OutletID,CustomerPayment.TransactionID,CustomerPayment.EmployeeID PaymentEmployeeID,{0}.RouteHistoryID TransactionRouteHistoryID ,{0}.EmployeeID,{0}.RemainingAmount,{0}.TransactionTypeID,{0}.SalesMode,
                CustomerPaymentID,CustomerPayment.RouteHistoryID PaymentRouteHistoryID,CustomerPayment.AppliedAmount,CustomerPayment.AccountID,CustomerOutlet.CustomerTypeID ,CustomerPayment.SourceTransactionID,{0}.CreationReason,
                PaymentTypeID,VoucherDate,IsTransRemainingAffected,CustomerPayment.CurrencyID PaymentCurrencyID ,Account.CurrencyID CustomerAccountCurrnecyID ,PaymentStatusID, CustomerPayment.DivisionID,CNTransaction.DivisionID CNTransDivisionID,CNTransaction.RouteHistoryID CNTransRouteHistoryID FROM CustomerPayment
                INNER JOIN {0} ON {0}.TransactionID = customerPayment.TransactionID AND {0}.CustomerID = customerPayment.CustomerID  AND {0}.DivisionID = CustomerPayment.DivisionID 
                AND {0}.OutletID = CustomerPayment.OutletID  
                INNER JOIN CustomerOutlet ON CustomerPayment.CustomerID = CustomerOutlet.CustomerID AND CustomerPayment.OutletID = CustomerOutlet.OutletID 
                INNER JOIN Customer ON Customer.CustomerID = CustomerPayment.CustomerID
                LEFT OUTER JOIN Account ON CustomerPayment.AccountID = Account.AccountID
                LEFT OUTER JOIN {0} CNTransaction ON CNTransaction.TransactionID = customerPayment.SourceTransactionID AND CNTransaction.CustomerID = customerPayment.CustomerID  
                AND CNTransaction.OutletID = CustomerPayment.OutletID and CNTransaction.DivisionID = CustomerPayment.DivisionID and CNTransaction.TransactionTypeID in (5,11)                 
                WHERE CustomerPayment.Posted = 0 And (PaymentStatusID NOT IN ({1},{2}) ) ", CoreDataBaseConstants.TableNames.Transaction, PaymentStatusTypes.Bounced.GetHashCode(), PaymentStatusTypes.Voided.GetHashCode());
                if (!string.IsNullOrEmpty(customerPaymentID))
                {
                    if (customerPaymentID.Contains("'"))
                        sqlStatement.AppendFormat(" AND CustomerPayment.CustomerPaymentID in ({0}) ", customerPaymentID);
                    else
                        sqlStatement.AppendFormat(" AND CustomerPayment.CustomerPaymentID in ('{0}') ", customerPaymentID);
                }
                if (customerID != -1)
                {
                    sqlStatement.AppendFormat(" AND CustomerPayment.CustomerID = {0} ", customerID);
                }
                if (outletID != -1)
                {
                    sqlStatement.AppendFormat(" AND CustomerPayment.OutletID = {0} ", outletID);
                }
                if (employeeID != -1)
                {
                    sqlStatement.AppendFormat(" AND CustomerPayment.EmployeeID = {0} ", employeeID);
                }
                sqlStatement.Append(" ORDER BY CustomerPaymentID DESC ");

                DataTable dt = new DataTable();
                result = dbHelper.GetQueryDataTable(sqlStatement.ToString(), ref dt);

                if (result == GlobalErrors.Success && dt != null && dt.Rows.Count > 0)
                {
                    Dictionary<long, AccountModel> employeesAccounts = new Dictionary<long, AccountModel>();
                    Dictionary<int, Dictionary<int, CustomerOutletModel>> _customerOutlets = new Dictionary<int, Dictionary<int, CustomerOutletModel>>();
                    while (dt.Rows.Count > 0)
                    {
                        DataTable dtModified = dt.Copy();
                        string custPaymentID = string.Empty;
                        int currnetCustomerID = -1, currnetOutletID = -1, customerTypeID = -1, transEmployeeID = -1, divisionID = -1;
                        int lastRowIndex = dtModified.Rows.Count - 1;
                        // Read general data one time only .
                        if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerID].ToString())) { currnetCustomerID = int.Parse(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerID].ToString()); }
                        if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.OutletID].ToString())) { currnetOutletID = Convert.ToInt32(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.OutletID]); }
                        if (!_customerOutlets.ContainsKey(currnetCustomerID))
                        {
                            _customerOutlets[currnetCustomerID] = new Dictionary<int, CustomerOutletModel>();
                        }
                        if (!_customerOutlets[currnetCustomerID].ContainsKey(currnetOutletID))
                        {
                            _customerOutlets[currnetCustomerID][currnetOutletID] = new CustomerOutletModel();
                            _customerOutlets[currnetCustomerID][currnetOutletID].CustomerId = currnetCustomerID;
                            _customerOutlets[currnetCustomerID][currnetOutletID].OutletId = currnetOutletID;
                            if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerTypeID].ToString())) { customerTypeID = int.Parse(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerTypeID].ToString()); }
                            if (customerTypeID == CustomerTypes.CashCustomer.GetHashCode())
                            {
                                _customerOutlets[currnetCustomerID][currnetOutletID].IsCredit = false;
                            }
                            else
                            {
                                _customerOutlets[currnetCustomerID][currnetOutletID].IsCredit = true;
                            }
                        }
                        if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.DivisionID].ToString())) { divisionID = int.Parse(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.DivisionID].ToString()); }
                        if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerPaymentID].ToString())) { custPaymentID = dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.CustomerPaymentID].ToString(); }
                        string filter = string.Format(" CustomerID = {0} AND OutletID = {1} AND CustomerPaymentID = '{2}' AND DivisionID = {3}", currnetCustomerID, currnetOutletID, custPaymentID, divisionID);
                        if (!string.IsNullOrEmpty(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.SalesMode].ToString())) { _salesMode = (SalesModes)int.Parse(dtModified.Rows[lastRowIndex][CoreDataBaseConstants.QueryColumnsNames.SalesMode].ToString()); }

                        DataRow[] dr = dtModified.Select(filter);
                        CustomerPaymentForPosting payment = null;
                        int custAccountCurrancyID = -1;
                        if (dr != null && dr.Length > 0)
                        {
                            for (int i = dr.Length - 1; i >= 0; i--)
                            {
                                if (!string.IsNullOrEmpty(dr[i][CoreDataBaseConstants.QueryColumnsNames.EmployeeID].ToString()))
                                {
                                    transEmployeeID = int.Parse(dr[i][CoreDataBaseConstants.QueryColumnsNames.EmployeeID].ToString());
                                }
                                if (!employeesAccounts.ContainsKey(transEmployeeID))
                                {
                                    // we need to update the account of the saleperson who did the transaction, not the one collected the payment
                                    AccountModel empAccount;
                                    empAccount = _accountManager.GetEmployeeAccount(transEmployeeID, dbHelper);
                                    if (empAccount == null)
                                    {
                                        return GlobalErrors.Error;
                                    }
                                    employeesAccounts.Add(transEmployeeID, empAccount);
                                }
                                result = FillAppliedPayment(dr[i], transEmployeeID, currnetCustomerID, currnetOutletID, ref payment, ref custAccountCurrancyID, _customerOutlets);
                                if (result == GlobalErrors.Success)
                                {
                                    // Remove visited rows so we will not go over them again
                                    System.Reflection.FieldInfo fieldInfo = dr[i].GetType().GetField(RowIDProperty, System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.Instance);
                                    if (fieldInfo == null) return GlobalErrors.Error;
                                    int rowID = (int)Convert.ToInt64(fieldInfo.GetValue(dr[i]));
                                    dt.Rows.RemoveAt(rowID - 1);
                                }
                                if (result != GlobalErrors.Success) return result;
                            }
                            if (result == GlobalErrors.Success)
                            {
                                result = PostSingleCustomerPayment(dbHelper, currnetCustomerID, currnetOutletID, payment, custAccountCurrancyID, employeesAccounts, isManualPosting, _customerOutlets, _salesMode, ref errorMessage);
                            }
                            if (result != GlobalErrors.Success)
                            {
                                return GlobalErrors.Error;
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors FillAppliedPayment(DataRow dr, int transEmployeeID, int customerID, int outletID, ref CustomerPaymentForPosting payment, ref int custAccountCurrancyID, Dictionary<int, Dictionary<int, CustomerOutletModel>> _customerOutlets)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (dr != null)
                {
                    if (payment == null)
                    {
                        payment = new CustomerPaymentForPosting();
                        payment.CustomerId = customerID;
                        payment.OutletId = outletID;
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.CustomerPaymentID].ToString()))
                        {
                            payment.CustomerPaymentId = dr[CoreDataBaseConstants.QueryColumnsNames.CustomerPaymentID].ToString();
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.CustomerAccountCurrnecyID].ToString()))
                        {
                            custAccountCurrancyID = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.CustomerAccountCurrnecyID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentStatusID].ToString()))
                        {
                            payment.PaymentStatusId = (PaymentStatusTypes)int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentStatusID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.AccountID].ToString()))
                        {
                            payment.AccountId = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.AccountID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentRouteHistoryID].ToString()))
                        {
                            payment.RouteHistoryId = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentRouteHistoryID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentCurrencyID].ToString()))
                        {
                            payment.CurrencyId = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentCurrencyID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.VoucherDate].ToString()))
                        {
                            payment.VoucherDate = (DateTime)(dr[CoreDataBaseConstants.QueryColumnsNames.VoucherDate]);
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.DivisionID].ToString()))
                        {
                            payment.DivisionId = Convert.ToInt32(dr[CoreDataBaseConstants.QueryColumnsNames.DivisionID]);
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentTypeID].ToString()))
                        {
                            payment.PaymentTypeId = (PaymentTypes)int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentTypeID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.SourceTransactionID].ToString()))
                        {
                            payment.SourceTransactionId = dr[CoreDataBaseConstants.QueryColumnsNames.SourceTransactionID].ToString();
                        }
                        if (!string.IsNullOrEmpty(dr["CNTransDivisionID"].ToString()))
                        {
                            payment.CnTransDivisionId = int.Parse(dr["CNTransDivisionID"].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr["CNTransRouteHistoryID"].ToString()))
                        {
                            payment.CnTransRouteHistroyId = int.Parse(dr["CNTransRouteHistoryID"].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentEmployeeID].ToString()))
                        {
                            payment.EmployeeId = Int32.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.PaymentEmployeeID].ToString());
                        }
                    }
                    string transactionID = string.Empty;
                    if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.TransactionID].ToString()))
                    {
                        transactionID = dr[CoreDataBaseConstants.QueryColumnsNames.TransactionID].ToString();
                    }
                    if (!payment.CoveredTransactions.ContainsKey(transactionID))
                    {
                        payment.CoveredTransactions[transactionID] = new PaymentTransaction();
                        payment.CoveredTransactions[transactionID].TransactionId = transactionID;
                        payment.CoveredTransactions[transactionID].TransactionEmployeeId = transEmployeeID;
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.AppliedAmount].ToString()))
                        {
                            payment.CoveredTransactions[transactionID].CoveredAmount = decimal.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.AppliedAmount].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.TransactionRouteHistoryID].ToString()))
                        {
                            payment.CoveredTransactions[transactionID].TransactionRouteHistoryId = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.TransactionRouteHistoryID].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.CreationReason].ToString()))
                        {
                            payment.CoveredTransactions[transactionID].CreationReason = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.CreationReason].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.IsTransRemainingAffected].ToString()))
                        {
                            payment.CoveredTransactions[transactionID].IsTransRemainingAffected = Convert.ToBoolean(dr[CoreDataBaseConstants.QueryColumnsNames.IsTransRemainingAffected]);
                        }
                    }
                    else
                    {
                        payment.CoveredTransactions[transactionID].CoveredAmount += decimal.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.AppliedAmount].ToString());
                    }
                    if (!_customerOutlets[customerID][outletID].Transactions.ContainsKey(transactionID))
                    {
                        _customerOutlets[customerID][outletID].Transactions[transactionID] = new TransactionModel();
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.RemainingAmount].ToString()))
                        {
                            _customerOutlets[customerID][outletID].Transactions[transactionID].RemainingAmount = decimal.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.RemainingAmount].ToString());
                        }
                        if (!string.IsNullOrEmpty(dr[CoreDataBaseConstants.QueryColumnsNames.TransactionTypeID].ToString()))
                        {
                            _customerOutlets[customerID][outletID].Transactions[transactionID].TransactionTypeId = int.Parse(dr[CoreDataBaseConstants.QueryColumnsNames.TransactionTypeID].ToString());
                            _customerOutlets[customerID][outletID].Transactions[transactionID].TransactionType = (TransactionType)(_customerOutlets[customerID][outletID].Transactions[transactionID].TransactionTypeId);

                        }
                        _customerOutlets[customerID][outletID].Transactions[transactionID].DivisionId = payment.DivisionId;
                    }
                    result = GlobalErrors.Success;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors PostSingleCustomerPayment(DBHelper<int> dbHelper, int customerID, int outletID, CustomerPaymentForPosting payment, int customerAccountCurrencyID, Dictionary<long, AccountModel> employeesAccounts, bool isManualPosting, Dictionary<int, Dictionary<int, CustomerOutletModel>> _customerOutlets, SalesModes _salesMode, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (payment != null && payment.CoveredTransactions != null && payment.CoveredTransactions.Count > 0)
                {
                    bool isValidPayment = true;
                    if (payment.PaymentStatusId != PaymentStatusTypes.Voided)
                    {
                        foreach (string transactionID in payment.CoveredTransactions.Keys)
                        {
                            if (_customerOutlets[customerID][outletID].Transactions.ContainsKey(transactionID))
                            {
                                if (payment.RouteHistoryId != payment.CoveredTransactions[transactionID].TransactionRouteHistoryId && (_customerOutlets[customerID][outletID].Transactions[transactionID].RemainingAmount < payment.CoveredTransactions[transactionID].CoveredAmount && !payment.CoveredTransactions[transactionID].IsTransRemainingAffected))
                                {
                                    isValidPayment = false;
                                    break;
                                }
                            }
                            else
                            {
                                return GlobalErrors.Error;
                            }
                        }
                    }
                    isValidPayment = isValidPayment && !(payment.PaymentTypeId == PaymentTypes.PostedDatedCheque && !_requestRepository.Configurations.AcceptPDCPaymentOnReceipt);
                    result = GlobalErrors.Success;

                    if (result == GlobalErrors.Success)
                    {
                        bool continuePosting = true;
                        if (payment.PaymentStatusId != PaymentStatusTypes.Voided)
                        {
                            foreach (string transactionID in payment.CoveredTransactions.Keys)
                            {
                                if (isValidPayment)
                                {
                                    if (!payment.CoveredTransactions[transactionID].IsTransRemainingAffected)
                                    {
                                        if (payment.RouteHistoryId != payment.CoveredTransactions[transactionID].TransactionRouteHistoryId
                                            || (payment.RouteHistoryId == -1 && payment.CoveredTransactions[transactionID].TransactionRouteHistoryId == -1)
                                            || ((payment.RouteHistoryId == payment.CoveredTransactions[transactionID].TransactionRouteHistoryId) && (payment.CoveredTransactions[transactionID].CreationReason == (int)TransactionCreationReason.OrderDeliveryBOInvoice)))
                                        {
                                            result = _accountManager.UpdateTransactionRemainingAmount(dbHelper, transactionID, customerID, outletID, payment.DivisionId, _customerOutlets[customerID][outletID].Transactions[transactionID].TransactionType, payment.CoveredTransactions[transactionID].CoveredAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                                            if (result == GlobalErrors.Success)
                                            {
                                                result = UpdatePaymentAffectingTransaction(payment, dbHelper);
                                            }
                                        }
                                        if (result == GlobalErrors.Success && (payment.PaymentTypeId == PaymentTypes.CreditNote))
                                        {
                                            if (payment.CnTransRouteHistroyId != payment.RouteHistoryId || (payment.CnTransRouteHistroyId == -1 && payment.RouteHistoryId == -1))
                                            {
                                                result = _accountManager.UpdateTransactionRemainingAmount(dbHelper, payment.SourceTransactionId, customerID, outletID, payment.CnTransDivisionId, TransactionType.CreditNote, payment.CoveredTransactions[transactionID].CoveredAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                                            }
                                        }
                                        else if (result == GlobalErrors.Success && (payment.PaymentTypeId == PaymentTypes.RentFees))
                                        {
                                            if (payment.CnTransRouteHistroyId != payment.RouteHistoryId || (payment.CnTransRouteHistroyId == -1 && payment.RouteHistoryId == -1))
                                            {
                                                result = _accountManager.UpdateTransactionRemainingAmount(dbHelper, payment.SourceTransactionId, customerID, outletID, payment.CnTransDivisionId, TransactionType.RentFees, payment.CoveredTransactions[transactionID].CoveredAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                                            }
                                        }
                                        if (result == GlobalErrors.Success && payment.RouteHistoryId == payment.CoveredTransactions[transactionID].TransactionRouteHistoryId)
                                        {
                                            result = UpdatePaymentAffectingTransaction(payment, dbHelper);
                                        }
                                    }
                                    if (result == GlobalErrors.Success)
                                    {
                                        result = UpdateBalancesByPayment(dbHelper, _customerOutlets[customerID][outletID], payment, transactionID, employeesAccounts, ref continuePosting, isManualPosting, _salesMode, ref errorMessage);
                                    }
                                    if (result != GlobalErrors.Success) break;
                                }
                            }
                        }
                        if (result == GlobalErrors.Success && !isValidPayment)
                        {
                            payment.PaymentStatusId = PaymentStatusTypes.ExtraPaid;
                            result = UpdateCustomerPaymentStatus(payment, dbHelper);
                        }
                        if (result == GlobalErrors.Success && continuePosting)
                        {
                            result = UpdateCustomerPaymentPosted(dbHelper, payment);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors UpdatePaymentAffectingTransaction(PaymentModel payment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.Error;
            try
            {
                string query = string.Format(@"Update CustomerPayment SET IsTransRemainingAffected = 1, UpdatedBy = {4}, UpdatedDate = {5}  WHERE CustomerPaymentID = '{0}' AND  CustomerID = {1} AND  OutletID  = {2} AND DivisionID = {3}",
                    payment.CustomerPaymentId, payment.CustomerId, payment.OutletId, payment.DivisionId, _requestRepository.CurrentOperator.EmployeeId, LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now));
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdateCustomerPaymentStatus(PaymentModel payment, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format(@"UPDATE CustomerPayment SET PaymentStatusID = {0}, UpdatedBy = {6}, UpdatedDate = {7} WHERE CustomerID = {1} AND OutletID = {2}
                AND CustomerPaymentID = '{3}' AND EmployeeID = {4} AND DivisionID = {5} ", (int)payment.PaymentStatusId, payment.CustomerId, payment.OutletId, payment.CustomerPaymentId, payment.EmployeeId, payment.DivisionId, _requestRepository.CurrentOperator.EmployeeId, LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now));
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors UpdateBalancesByPayment(DBHelper<int> dbHelper, CustomerOutletModel outlet, CustomerPaymentForPosting payment, string transactionID, Dictionary<long, AccountModel> employeesAccounts, ref bool continuePosting, bool isManualPosting, SalesModes _salesMode, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (payment.PaymentTypeId != PaymentTypes.PostedDatedCheque || (payment.PaymentTypeId == PaymentTypes.PostedDatedCheque && DateTime.Compare(DateTime.Now, payment.VoucherDate) >= 0 && (_requestRepository.Configurations.PostPDCAutomatically || isManualPosting)))
                {
                    decimal paidAmount = payment.CoveredTransactions[transactionID].CoveredAmount * -1;
                    result = _accountManager.UpdateAccountBalance(payment.AccountId, paidAmount, DateTime.Now, payment.CoveredTransactions[transactionID].TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, transactionID, dbHelper, 5);
                    if (result == GlobalErrors.Success)
                    {
                        // Update Employee Account Account.
                        if (
                            (!outlet.IsCredit && _salesMode != SalesModes.CreditSales) || 
                            (outlet.IsCredit && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSale) ||
                            (_salesMode == SalesModes.CreditSales && !outlet.IsCredit && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSaleForCashCustomerOnly)
                          )
                        {
                            if (employeesAccounts.ContainsKey(payment.CoveredTransactions[transactionID].TransactionEmployeeId))
                            {
                                result = _accountManager.UpdateAccountBalance(employeesAccounts[payment.CoveredTransactions[transactionID].TransactionEmployeeId].AccountId, paidAmount, DateTime.Now, payment.CoveredTransactions[transactionID].TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, transactionID, dbHelper, 1);
                                if (result == GlobalErrors.Success && _requestRepository.Configurations.ControlTransactionOnDivisionLevel && _requestRepository.Configurations.CheckCreditLimitOnEmployeeDivisionAccount)
                                    result = _accountManager.UpdateEmployeeDivisionAccountBalance(employeesAccounts[payment.CoveredTransactions[transactionID].TransactionEmployeeId].AccountId, paidAmount, DateTime.Now, payment.CoveredTransactions[transactionID].TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, transactionID, dbHelper, 1, payment.DivisionId, ref errorMessage);
                            }
                        }
                    }
                }
                else
                {
                    result = GlobalErrors.Success;
                    continuePosting = false;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors UpdateCustomerPaymentPosted(DBHelper<int> dbHelper, PaymentModel payment)
        {
            GlobalErrors result = GlobalErrors.Error;
            try
            {
                string query = string.Format(@"Update CustomerPayment SET Posted = 1, UpdatedBy = {4}, UpdatedDate = {5}  WHERE CustomerPaymentID = '{0}' AND  CustomerID = {1} AND  OutletID  = {2}  AND DivisionID = {3}",
                    payment.CustomerPaymentId, payment.CustomerId, payment.OutletId, payment.DivisionId, _requestRepository.CurrentOperator.EmployeeId, LocalUtilities.ParseDateAndTimeToSQL(DateTime.Now));
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetBalanceOriginalValue(string transactionId, ref decimal originalValue, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                object field = new object();
                string query = string.Format(@"select sum(ActualValue) from iCashTransactionDetails where TransactionID = '{0}'", transactionId);
                result = dbHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != null && !string.IsNullOrEmpty(field.ToString()) && decimal.Parse(field.ToString()) > 0)
                {
                    originalValue = decimal.Parse(field.ToString());
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors VoidICashBalance(string transactionId, int customerId, int outletId, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format(@"UPDATE iCashBalance SET Balance = ((Select Balance from iCashBalance where CustomerID = {1} AND OutletID = {2}) - (select sum(ActualValue) from iCashTransactionDetails where TransactionID = '{0}')) WHERE CustomerID = {1} AND OutletID= {2}", transactionId, customerId, outletId);
                result = dbHelper.ExecuteNonQuery(query);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdateICashBalanceICashPayment(string transactionId, int customerId, int outletId, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format(@"UPDATE iCashBalance SET Balance = ((Select Balance from iCashBalance where CustomerID = {0} AND OutletID = {1}) + 
               (select isNull(sum(AppliedAmount),0) from CustomerPayment where TransactionID = '{2}' and PaymentTypeID= {3})) WHERE CustomerID = {0} AND OutletID= {1}", customerId, outletId, transactionId,PaymentTypes.iCash.GetHashCode());
                result = dbHelper.ExecuteNoneQueryWithZeroRowAffectedCheck(query);
                if (result == GlobalErrors.SuccessWithZeroRowAffected)
                    result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #endregion


        #region[Post PDC]
        public GlobalErrors GetPDCSharedTable(GeneralFilter filter,ref List<PaymentModel> pdcSharedTable)
        {
            GlobalErrors result = GlobalErrors.Error;
            string dateFilter = string.Empty;
            string searchFilter = string.Empty;
            string customerFilter = string.Empty;
            string employeeFilter = string.Empty;
            try
            {
                if (filter.DateModel != null)
                {
                    dateFilter = $@" AND VoucherDate  >= {LocalUtilities.ParseDateToSQLString(filter.DateModel.Date)} and VoucherDate  <= {LocalUtilities.ParseEndDateToSQLString(filter.DateModel.Date)}";
                }
                else
                {
                    dateFilter = $@" AND VoucherDate  < {LocalUtilities.ParseDateToSQLString(DateTime.Now.AddDays(1))}";
                }
                if (filter.DateModel != null)
                {
                    dateFilter = $@" AND VoucherDate  >= {LocalUtilities.ParseDateToSQLString(filter.DateModel.Date)} and VoucherDate  <= {LocalUtilities.ParseEndDateToSQLString(filter.DateModel.Date)}";
                }
                if (filter.CustomerId > -1)
                {
                    customerFilter = $@" And CustomerPayment.CustomerId = {filter.CustomerId}";
                    if (filter.OutletId > -1) customerFilter += $@" And CustomerPayment.OutletId = {filter.OutletId}";
                }
                if(filter.EmployeeId > -1)
                {
                    employeeFilter = $@" And CustomerPayment.EmployeeID = {filter.EmployeeId}";
                }
                if (filter.CustomListFilter != null && !string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    searchFilter = $@" AND (
                                            VoucherNumber LIKE '%{filter.CustomListFilter.SearchFilter}%' OR
                                            CustomerLanguage.Description LIKE '%{filter.CustomListFilter.SearchFilter}%' OR
                                            Customer.CustomerCode LIKE '%{filter.CustomListFilter.SearchFilter}%' OR
                                            employeeLanguage LIKE '%{filter.CustomListFilter.SearchFilter}%' OR
                                            VoucherDate LIKE '%{filter.CustomListFilter.SearchFilter}%' 
                                            )";
                }
                DBHelper<PaymentModel> dBHelper = new DBHelper<PaymentModel>();
                string query = $@"Select Distinct VoucherNumber,CustomerLanguage.Description CustomerName,Customer.CustomerCode,
                CustomerPayment.EmployeeID,employeeLanguage.Description
                Employee,VoucherDate,CustomerPayment.CustomerID ,
                IsNull( Customer.CustomerCode , '--') + ' - ' + IsNull( CustomerLanguage.Description , '--') as CustomerCodeName,
                IsNull( CustomerOutlet.CustomerCode , '--') + ' - ' + IsNull( CustomerOutletLanguage.Description , '--') as OutletCodeName,
                IsNull( Employee.EmployeeCode , '--') + ' - ' + IsNull( EmployeeLanguage.Description , '--') as EmployeeCodeName
                From CustomerPayment
                Inner Join Customer On Customer.CustomerID = CustomerPayment.CustomerID
                Left Join CustomerLanguage On CustomerLanguage.CustomerID = CustomerPayment.CustomerID And CustomerLanguage.LanguageID = {_requestRepository.LanguageId}
                INNER JOIN Employee On Employee.EmployeeID = CustomerPayment.EmployeeID
                LEFT JOIN EmployeeLanguage On EmployeeLanguage.EmployeeID = Employee.EmployeeID And EmployeeLanguage.LanguageID =  {_requestRepository.LanguageId}
                inner join CustomerOutlet on CustomerOutlet.CustomerID = CustomerPayment.CustomerID
					and CustomerOutlet.OutletID = CustomerPayment.OutletID
				LEFT JOIN CustomerOutletLanguage on CustomerOutletLanguage.CustomerID = CustomerPayment.CustomerID
					and  CustomerOutletLanguage.OutletID = CustomerPayment.OutletID and CustomerOutletLanguage.LanguageID =  {_requestRepository.LanguageId}
                Where 
                CustomerPayment.Posted = 0 
                AND PaymentTypeID = {PaymentTypes.PostedDatedCheque.GetHashCode()} 
                And (PaymentStatusID NOT IN ({(int)PaymentStatusTypes.Bounced},
                                            {(int)PaymentStatusTypes.Voided},
                                            {(int)PaymentStatusTypes.ExtraPaid},
                                            {(int)PaymentStatusTypes.ConvertedToCreditNote}) 
                And (RouteHistoryID IS Null Or RouteHistoryID = -1) 
                AND Employee.OrganizationID in ({_requestRepository.CurrentOperator.OrganizationAccess})) {dateFilter}
                {customerFilter}
                {employeeFilter}
                {searchFilter}
                ORDER BY VoucherDate";
                result = dBHelper.GetQueryList(query, ref pdcSharedTable , true);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors CancelOnlineEPayments(PaymentModel payment)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = new DBHelper<int>();
            try
            {
                string PaymentGateWayPassword = string.Empty;

                var config = ConfigurationManager.GetConfigurationOnOrganization("PaymentGateWayPassword", payment.OrganizationId);
                if (config != null && !string.IsNullOrEmpty(config.KeyValue))
                {
                    PaymentGateWayPassword = config.KeyValue.Trim().ToString();
                }
                if (string.IsNullOrEmpty(PaymentGateWayPassword))
                {
                    return GlobalErrors.Error;
                }
                if (string.IsNullOrEmpty(_requestRepository.Configurations.PaymentGateway))
                {
                    return GlobalErrors.Error;
                }

                var options = new RestClientOptions(_requestRepository.Configurations.PaymentGateway)
                {
                    MaxTimeout = 30000,
                };
                var client = new RestClient(options);
                var request = new RestRequest("/v2/invoices/" + payment.ThirdPartyTransactionId, Method.Delete);
                //request.AddHeader("Content-Type", "application/json");
                request.AddHeader("Authorization", PaymentGateWayPassword);

                RestResponse response = client.Execute(request);
                OnlineEPaymentResponseModel responseModel = new OnlineEPaymentResponseModel();
                responseModel.ResponseCode = response.StatusCode.GetHashCode();
                responseModel.ResponseDescription = response.StatusDescription == null ? "" : (string)response.StatusDescription;
                responseModel.ResponseContent = response.Content;
                responseModel.UUID = payment.UUID;

                result = SaveOnlineEPaymentResponse(responseModel, payment, dBHelper, true);
                if(result != GlobalErrors.Success || response.StatusCode != System.Net.HttpStatusCode.OK)
                {
                    return GlobalErrors.Error;
                }
                if (result == GlobalErrors.Success && response.StatusCode == System.Net.HttpStatusCode.OK)
                {
                    JObject jObject = JObject.Parse(response.Content);
                    string epaymentStatus = jObject["status"] == null ? "" : (string)jObject["status"];

                    if (epaymentStatus.ToLower() == "cancelled" || epaymentStatus.ToLower() == "expired")
                    {
                        result = VoidOnlineEPayment(payment);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
    }
    #endregion
}
