﻿using Dapper;
using Oracle.ManagedDataAccess.Client;
using Sonic.Web.Model;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Reflection;
using static System.Net.Mime.MediaTypeNames;

namespace Sonic.Web.DAL
{
    public class DBHelper<T>
    {
        private int _employeeId = 0;
        internal bool Opened = false;
        private IDbConnection _connection;
        private IDbTransaction _dbTransaction;
        private bool _dbTransactionIsActive = false;
        private int _connectionTimeout = 300;
        public IDbConnection GetConnection()
        {
            return _connection;
        }
        public IDbTransaction GetDBTransaction()
        {
            return _dbTransaction;
        }

        public DBHelper()
        {
            try
            {
                _employeeId = 0;
                if (LocalUtilities.SystemEnvironment == SystemEnvironment.Oracle)
                {
                    _connection = new OracleConnection(LocalUtilities.ConnectionString);
                }
                else if (LocalUtilities.SystemEnvironment == SystemEnvironment.SQL)
                {
                    _connection = new SqlConnection(LocalUtilities.ConnectionString);
                }
                _dbTransaction = null;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
            }
        }
        public DBHelper(int employeeId) : this()
        {
            try
            {
                _employeeId = employeeId;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
            }
        }
        public DBHelper(IDbConnection connection, IDbTransaction dbTransaction, int employeeId = 0)
        {
            try
            {
                _employeeId = employeeId;
                _connection = connection;
                if (_dbTransaction != null)
                {
                    _dbTransactionIsActive = true;
                }
                _dbTransaction = dbTransaction;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
            }
        }
        #region [TRANSACTIONS AREA]

        public GlobalErrors BeginTransaction()
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                SetConnectionString();
                _connection.Open();
                _dbTransaction = _connection.BeginTransaction();
                _dbTransactionIsActive = true;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (_dbTransaction == null)
                {
                    result = GlobalErrors.Error;
                }
                else
                {
                    result = GlobalErrors.Success;
                }
            }
            return result;
        }

        public GlobalErrors CommitTransaction()
        {
            try
            {
                _dbTransactionIsActive = false;

                if (_dbTransaction != null)
                {
                    _dbTransaction.Commit();
                    return GlobalErrors.Success;
                }
                else
                {
                    return GlobalErrors.NotInitialized;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (_dbTransaction != null)
                    _dbTransaction.Dispose();
                _dbTransaction = null;
                if (_connection != null)
                {
                    _connection.Close();
                    _connection.Dispose();
                    _connection = null;
                }
            }
        }

        public void Dispose()
        {
            if (_dbTransaction != null)
            {
                _dbTransaction.Dispose();
                _dbTransaction = null;
            }
            if (_connection != null)
            {
                _connection.Dispose();
                _connection = null;
            }
        }

        public GlobalErrors RollBackTransaction()
        {
            try
            {
                _dbTransactionIsActive = false;

                if (_dbTransaction != null)
                {
                    _dbTransaction.Rollback();
                    return GlobalErrors.Success;
                }
                else
                {
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (_dbTransaction != null)
                    _dbTransaction.Dispose();
                _dbTransaction = null;
                if (_connection != null)
                {
                    _connection.Close();
                    _connection.Dispose();
                    _connection = null;
                }
            }
        }

        #endregion [TRANSACTIONS AREA]

        public GlobalErrors ExecuteNoneQueryWithZeroRowAffectedCheck(string query)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                int affectedRows = -1;
                if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
                {
                    _connection.Close();
                    result = GlobalErrors.Error;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                    return result;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        affectedRows = dbConnection.Execute(query);
                    }
                }
                else
                {
                    affectedRows = _connection.Execute(query, null, _dbTransaction, 0, CommandType.Text);
                }
                if (affectedRows == 0)
                    result = GlobalErrors.SuccessWithZeroRowAffected;
                else if (affectedRows > 0)
                    result = GlobalErrors.Success;
                else
                    result = GlobalErrors.Error;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ExecuteNonQuery(string query, bool writeLog, Dictionary<string, object> paramters)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
                {
                    _connection.Close();
                    result = GlobalErrors.Error;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                    return result;
                }
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                int affectedRows = -1;
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        affectedRows = dbConnection.Execute(query, paramters != null ? new DynamicParameters(paramters) : null);
                    }
                }
                else
                {
                    affectedRows = _connection.Execute(query, paramters != null ? new DynamicParameters(paramters) : null, _dbTransaction, 0, CommandType.Text);
                }
                if (affectedRows > 0)
                    result = GlobalErrors.Success;
                else
                    result = GlobalErrors.Error;
            }
            catch (Exception ex)
            {
                if (writeLog)
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                }
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ExecuteNonQuery(string query)
        {
            return ExecuteNonQuery(query, true, null);
        }
        public GlobalErrors ExecuteNonQuery(string query, Dictionary<string, object> paramters)
        {
            return ExecuteNonQuery(query, true, paramters);
        }

        public GlobalErrors ExecuteSQLCommand(string query)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
                {
                    _connection.Close();
                    result = GlobalErrors.Error;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                    return result;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        dbConnection.Execute(query);
                    }
                }
                else
                {
                    _connection.Execute(query, null, _dbTransaction, 0, CommandType.Text);
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ExecuteNonQueryWithParameters(string query, object parameters)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                int affectedRows = -1;
                if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
                {
                    _connection.Close();
                    result = GlobalErrors.Error;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                    return result;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        affectedRows = _connection.Execute(query, parameters, null, 0, CommandType.Text);
                    }
                }
                else
                {
                    affectedRows = _connection.Execute(query, parameters, _dbTransaction, 0, CommandType.Text);
                }
                if (affectedRows > 0)
                    result = GlobalErrors.Success;
                else
                    result = GlobalErrors.Error;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors ExecuteScalar(string query, ref Object field, bool setIsolationLevel = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (setIsolationLevel)
                {
                    query = "SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED " + query + " SET TRANSACTION ISOLATION LEVEL READ COMMITTED";
                }
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
                {
                    _connection.Close();
                    result = GlobalErrors.Error;
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                    return result;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        field = dbConnection.ExecuteScalar(query, null, null, _connectionTimeout, CommandType.Text);
                        result = GlobalErrors.Success;
                    }
                }
                else
                {
                    field = _connection.ExecuteScalar(query, null, _dbTransaction, _connectionTimeout, CommandType.Text);
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetQueryDataTable(string query, ref DataTable data, bool setIsolationLevel = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (setIsolationLevel)
                {
                    query = "SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED " + query + " SET TRANSACTION ISOLATION LEVEL READ COMMITTED";
                }
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                data = new DataTable();
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        var dataReader = dbConnection.ExecuteReader(query, null, null, _connectionTimeout);
                        if (dataReader != null)
                        {
                            data.Load(dataReader);
                        }
                        else
                        {
                            data = null;
                        }
                    }
                }
                else
                {
                    var dataReader = _connection.ExecuteReader(query, null, _dbTransaction, _connectionTimeout);
                    if (dataReader != null)
                    {
                        data.Load(dataReader);
                    }
                    else
                    {
                        data = null;
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetQueryList(string query, ref List<T> data, bool setIsolationLevel = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (setIsolationLevel)
                {
                    query = "SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED " + query + " SET TRANSACTION ISOLATION LEVEL READ COMMITTED";
                }
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        var list = dbConnection.Query<T>(query, null, null, true, _connectionTimeout);
                        if (list != null)
                        {
                            data = list.AsList();
                        }
                        else
                        {
                            data = null;
                        }
                    }
                }
                else
                {
                    var list = _connection.Query<T>(query, null, _dbTransaction, true, _connectionTimeout);
                    if (list != null && list.AsList().Count > 0)
                    {
                        data = list.AsList();
                    }
                    else
                    {
                        data = null;
                    }
                }
                result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetQuerySingle(string query, ref T data)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }
                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        var list = dbConnection.Query<T>(query, null, null, true, _connectionTimeout);
                        if (list != null && list.AsList().Count > 0)
                        {
                            data = list.AsList()[0];
                        }
                        else
                        {
                            data = default(T);
                        }
                        result = GlobalErrors.Success;
                    }
                }
                else
                {
                    var list = _connection.Query<T>(query, null, _dbTransaction, true, _connectionTimeout);
                    if (list != null && list.AsList().Count > 0)
                    {
                        data = list.AsList()[0];
                    }
                    else
                    {
                        data = default(T);
                    }
                    result = GlobalErrors.Success;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetQuerySingle(string query, ref T data, Dictionary<String,Object> par)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!isValidQuery(query))
                {
                    return GlobalErrors.Error;
                }

                if (_dbTransaction == null)
                {
                    SetConnectionString();
                    using (IDbConnection dbConnection = _connection)
                    {
                        var list = dbConnection.Query<T>(query, new DynamicParameters(par), null, true, _connectionTimeout);
                        if (list != null && list.AsList().Count > 0)
                        {
                            data = list.AsList()[0];
                        }
                        else
                        {
                            data = default(T);
                        }
                        result = GlobalErrors.Success;
                    }
                }
                else
                {
                    var list = _connection.Query<T>(query, null, _dbTransaction, true, _connectionTimeout);
                    if (list != null && list.AsList().Count > 0)
                    {
                        data = list.AsList()[0];
                    }
                    else
                    {
                        data = default(T);
                    }
                    result = GlobalErrors.Success;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }


        private void SetConnectionString()
        {
            if (!LocalUtilities.ConnectionString.Contains("Connection Timeout"))
            {
                if (!LocalUtilities.ConnectionString.EndsWith(';'))
                    LocalUtilities.ConnectionString += " ; ";
                LocalUtilities.ConnectionString += $" Connection Timeout= {_connectionTimeout}; ";
            }
            else
            {
                if(_connectionTimeout == 300)
                _connectionTimeout = _connection.ConnectionTimeout;
            }
            _connection.ConnectionString = LocalUtilities.ConnectionString;
        }

        public void SetTimeOut(int timeout)
        {
            _connectionTimeout = timeout;
        }
        public GlobalErrors ExecuteProcedureWithParameters(string procedureName, Dictionary<string, object> inputParameters, Dictionary<string, object> outputParameters, ref Dictionary<string, object> resultParamters, int timeOutSeconds) {

            return ExecuteProcedureWithParameters(procedureName, inputParameters, outputParameters, ref resultParamters, timeOutSeconds, true);
        }

        public GlobalErrors ExecuteProcedureWithParameters(string procedureName, Dictionary<string, object> inputParameters, Dictionary<string, object> outputParameters, ref Dictionary<string, object> resultParamters, int timeOutSeconds , bool closeConnection)
        {
            if (Opened) return GlobalErrors.DBQueryAlreadyOpened;
            if (_dbTransactionIsActive && (_dbTransaction == null || _dbTransaction.Connection == null))
            {
                _connection.Close();
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "The dBhelper began and its value became null during the process", null, "", 0);
                return GlobalErrors.Error;
            }
            if (_dbTransaction == null)
            {
                SetConnectionString();
            }
            //IDbConnection dbConnection = _connection;
            //IDbCommand dbCommand = dbConnection.CreateCommand();
            //if (dbConnection.State == ConnectionState.Closed || dbConnection.State == ConnectionState.Broken)
            //{
            //    dbConnection.Open();
            //}
            IDbCommand dbCommand = null;
            try
            {
                dbCommand = new SqlCommand();
                if(_connection.State == ConnectionState.Closed)
                {
                    _connection.Open();
                    Opened = true;
                }
                dbCommand.Connection = _connection;
                dbCommand.Transaction = _dbTransaction;
                dbCommand.CommandType = CommandType.StoredProcedure;
                dbCommand.CommandText = procedureName;
                // Setting time out
                dbCommand.CommandTimeout = timeOutSeconds;
                SqlParameter sqlParam = null;
                // Input
                //if(_dbTransaction != null)
                //{
                //    dbCommand.Transaction = _dbTransaction;
                //}
                foreach (string param in inputParameters.Keys)
                {
                    sqlParam = new SqlParameter(param, inputParameters[param]);
                    sqlParam.Direction = ParameterDirection.Input;
                    dbCommand.Parameters.Add(sqlParam);
                }
                // Output
                foreach (string param in outputParameters.Keys)
                {
                    sqlParam = new SqlParameter(param, SqlDbType.VarChar, 50);
                    sqlParam.Direction = ParameterDirection.Output;
                    dbCommand.Parameters.Add(sqlParam);
                }
                // Execution
                dbCommand.ExecuteNonQuery();
                //Result
                resultParamters = new Dictionary<string, object>();
                if (dbCommand.Parameters != null && dbCommand.Parameters.Count > 0)
                {
                    foreach (string param in outputParameters.Keys)
                    {
                        if (dbCommand.Parameters.Contains(param))
                            resultParamters.Add(param, ((SqlParameter)dbCommand.Parameters[param]).Value);
                    }
                }
            }
            catch (Exception ex)
            {
                GlobalErrors err = GlobalErrors.Error;
                if (ex.GetType() == typeof(System.Data.SqlClient.SqlException))
                {
                    if (((System.Data.SqlClient.SqlException)ex).Number == -2)
                        err = GlobalErrors.DBTimeoutExpired;
                    if (((System.Data.SqlClient.SqlException)ex).Number == 103)
                        err = GlobalErrors.DBCannotExecuteQuery;
                }
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", dbCommand.CommandText), null, string.Empty, _employeeId);
                return err;
            }
            finally
            {
                if(_connection.State == ConnectionState.Open && closeConnection)
                {
                    _connection.Close();
                    Opened = false;
                }
            }
            return GlobalErrors.Success;
        }

        public bool isValidQuery(string query)
        {
            try
            {
                if (!string.IsNullOrEmpty(query) && query.ToLower().Contains("waitfor ") && query.ToLower().Contains("delay"))
                {
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Wrong Query : SQL Injection", null, "Waitfor and Delay", _employeeId);
                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);

                    return false;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _employeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, string.Format("Query is [{0}]", query), null, string.Empty, _employeeId);
                return false;
            }
            return true;
        }

    }
}