import { HttpErrorResponse } from '@angular/common/http';
import { Component, Input, OnInit, ViewChild } from '@angular/core';
import { TranslateService } from '@ngx-translate/core';
import { Subject } from 'rxjs';
import { CoreSession } from '../../../../core/core.session';
import { NavBarService } from '../../../../core/sidebar/sidebar.service';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { DynamicTemplateEmailTypes } from '../../../../shared/models/enums/dynamic-template-email-types.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { NavigationData } from '../../../../shared/models/navigation-data/navigation-data.model';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { DynamicTemplateService } from '../dynamic-template.service';
import { SharedDialogComponent } from '../../../../shared/components/shared-dialog/shared-dialog.component';

@Component({
  selector: 'app-dynamic-templates-list',
  templateUrl: './dynamic-templates-list.component.html',
  styleUrls: ['./dynamic-templates-list.component.css']
})
export class DynamicTemplatesListComponent implements OnInit {

  @Input() emailTypeId = DynamicTemplateEmailTypes.Customer.valueOf();
  @ViewChild('ItemEntryDialog', { static: true }) ItemEntryDialog: SharedDialogComponent;

  isCustomerDynamicTemplate = true;
  navLinks: any[];
  isEditMode: boolean = false;
  TableProperties: ITableProperties = {
    pageSize: 1000,
    showPaginator: false,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    showNewBtn: false,
    rowOperations: [
    ],
    multiSelectionOperations: [],
    columns: [
      {
        title: 'Desc_Name',
        key: 'templateName',
        isSortable: true,
        width: '40%'
      },
      {
        title: 'Desc_Email_Type',
        key: 'emailTypeDescription',
        isSortable: true,
        useTranslate: true,
        width: '35%',
      },
      {
        title: 'Desc_Status',
        key: 'status',
        isSortable: true,
        useTranslate: true,
        width: '25%',
      }
    ]
  };

  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  showEntrySlider: boolean = false;
  saveSubject: Subject<void> = new Subject<void>();
  dynamicTempObj: any;
  constructor(
    public coreSession: CoreSession,
    private translateService: TranslateService,
    private navService: NavBarService,
    private dynamicTemplateService: DynamicTemplateService
  ) { }

  ngOnInit(): void {
    this.isCustomerDynamicTemplate = this.emailTypeId == DynamicTemplateEmailTypes.Customer.valueOf();
    this.coreSession.SetTitle('Dynamic Templates');
    this.prepareNavigation();
    var index = this.navLinks.findIndex(x => x.link == this.navService.searchMenuURL);
    if (index == -1) index = 0;
    this.navLinks[index].isActive = true;
    this.setNavigation(this.navLinks[index]);
    this.addActions();
    this.populateTemplatesList();
  }

  openItemEntryDialog() {
    this.ItemEntryDialog.Show(true).then((res) => { });
  }
  addActions() {
    var url = this.isCustomerDynamicTemplate ? ConstantURLs.dynamicTemplatesURL : ConstantURLs.EmployeeGroupEmailsURL;
    if (this.coreSession.checkActivitiesAvailability(url, MenuActions.Add)) {
      this.TableProperties.showNewBtn = true;
    }
    if (this.coreSession.checkActivitiesAvailability(url, MenuActions.Edit)) {
      this.TableProperties.rowOperations.push({
        operation: RowOperation.edit,
        title: "Desc_Edit",
        icon: "fa-regular fa-pen-to-square",
        color: "#12344d"
      });
      this.TableProperties.rowOperations.push({
        operation: RowOperation.Activate,
        title: "Desc_Reactivate",
        icon: "fa fa-toggle-on",
        color: '#02b875',
        showWhenKeyValueTrue: true,
        showHideOptionPerLine: true,
        controlKeyName: 'inActive'
      });

      this.TableProperties.rowOperations.push(
        {
          operation: RowOperation.Deactivate,
          title: 'Desc_Deactivate',
          icon: 'fa-solid fa-ban',
          color: '#f1685e',
          showWhenKeyValueFalse: true,
          showHideOptionPerLine: true,
          controlKeyName: 'inActive'
        });
    }
  }
  prepareNavigation() {
    this.navLinks = [
      {
        label: this.isCustomerDynamicTemplate ? 'Desc_Dynamic_Templates' : 'Desc_Employee_Group_Emails',
        index: 0,
        isActive: false,
        hidden: false,
        disable: false
      }
    ];
  }

  setNavigation(tab) {
    let navigationList: NavigationData[] = [];
    navigationList.push(
      {
        Name: 'Desc_Settings',
        URL: ConstantURLs.settingsURL
      },
      {
        Name: tab.label,
        URL: undefined
      }
    );
    this.navService.setNavigation(navigationList);
  }

  populateTemplatesList() {
    this.coreSession.ModalLoading.Show();
    this.dynamicTemplateService.getDynamicTemplates(this.emailTypeId).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.dataSource = {
            data: response.data,
            totalItems: response.data.length
          };
        } else {
          this.coreSession.showError(
            this.translateService.instant(ConstantMessages.ErrorCaption),
            this.translateService.instant(ConstantMessages.ErrorHappened)
          );
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(
          this.translateService.instant(ConstantMessages.ErrorCaption),
          this.translateService.instant(ConstantMessages.ErrorHappened)
        );
      }
    );
  }
  onRowOperation(event) {
    var url = this.isCustomerDynamicTemplate ? ConstantURLs.dynamicTemplatesURL : ConstantURLs.EmployeeGroupEmailsURL;
    switch (event.operation) {
      case RowOperation.edit:
        if (this.coreSession.checkActivitiesAvailability(url, MenuActions.Edit)) {
          if (event.object && event.object != null) {
            this.getTemplateData(event.object.templateId);
          }
        }
        break
      case RowOperation.Deactivate:
        this.updateDyanmicTemplateActiveStatus(event.object, false);
        break;
      case RowOperation.Activate:
        this.updateDyanmicTemplateActiveStatus(event.object, true);
        break;
    }

  }
  updateDyanmicTemplateActiveStatus(dynamicTemplate: any, isActive) {
    var msg = '';
    if (isActive) {
      msg = ConstantMessages.MsgReactivateConfirmation;
    } else {
      msg = ConstantMessages.MsgDeactivateConfirmation;
    }
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(msg), DialogMode.YesNo).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          dynamicTemplate.inActive = !isActive;
          this.updateDTStatus(dynamicTemplate);
        }
      });
  }

  updateDTStatus(dynamicTemplate: any) {

    this.coreSession.ModalLoading.Show();
    this.dynamicTemplateService.updateDynamicTemplateActiveStatus(dynamicTemplate).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
        this.populateTemplatesList();
      } else {
        dynamicTemplate.inActive = !dynamicTemplate.inActive;
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }

  getTemplateData(templateId: number) {
    if (templateId && templateId > -1) {
      this.coreSession.ModalLoading.Show();
      this.dynamicTemplateService.getDynamicTemplateData(templateId).subscribe(
        (response) => {
          this.coreSession.ModalLoading.Hide();
          if (response.status !== null && response.status >= 0) {
            this.dynamicTempObj = response.data;
            this.openEditScreen();
          } else {
            this.coreSession.showError(
              this.translateService.instant(ConstantMessages.ErrorCaption),
              this.translateService.instant(ConstantMessages.ErrorHappened)
            );
          }
        },
        (error: HttpErrorResponse) => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(
            this.translateService.instant(ConstantMessages.ErrorCaption),
            this.translateService.instant(ConstantMessages.ErrorHappened)
          );
        }
      );
    }
  }
  openEditScreen() {
    this.isEditMode = true;
    this.showEntrySlider = true;
    this.openItemEntryDialog()
  }

  onNewBtnClick() {
    this.isEditMode = false;
    this.showEntrySlider = true;
    this.openItemEntryDialog()
  }
  closeSlider() {
    this.isEditMode = false;
    this.showEntrySlider = false;
    this.ItemEntryDialog.Close()
  }
  onSaveClick() {
    this.saveSubject.next();
  }

  afterSave() {
    this.populateTemplatesList();
    this.closeSlider();
  }

  ngOnDestroy() {
    this.navService.setNavigation([]);
  }
}
