import { ExcelSheetTemplatesService } from './../excelSheetTemplate.service';
import { HttpErrorResponse } from '@angular/common/http';
import { Component, OnInit, ViewChild } from '@angular/core';
import { TranslateService } from '@ngx-translate/core';
import { Subject } from 'rxjs';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { CoreSession } from '../../../../core/core.session';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { NavigationData } from '../../../../shared/models/navigation-data/navigation-data.model';
import { NavBarService } from '../../../../core/sidebar/sidebar.service';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ExcelSheetTypes } from '../../../../shared/models/enums/excel-sheet-types.enum';
import { SharedDialogComponent } from '../../../../shared/components/shared-dialog/shared-dialog.component';


@Component({
    selector: 'app-excel-sheet-templates-list',
    templateUrl: './excel-sheet-templates-list.component.html',
    styleUrls: ['./excel-sheet-templates-list.component.css']
})
export class ExcelSheetDynamicTemplatesListComponent implements OnInit {

  @ViewChild('CategoryEntryDialog', { static: true }) CategoryEntryDialog: SharedDialogComponent;

    saveSubject: Subject<void> = new Subject<void>();
    showTemplateEntrySlider = false;
    isEditMode = false;
    isViewMode = false;
    templateObject: any = null;
    excelSheetTypesList: any[] = [];

    TableData: ITableProperties = {
        pageSize: this.coreSession.pageSize,
        showPaginator: true,
        isOnline: true,
        showSearch: true,
        showSearchBtn: false,
        showNewBtn: false,
        isMultiSelection: false,
        rowOperations: [],
        multiSelectionOperations: [],
        columns: [
            { title: 'Desc_Vendor', key: 'vendorNameCode', isSortable: true, width: '50%' },
            { title: 'Desc_Template_Type', key: 'excelSheetType', isSortable: true, width: '50%' }]
    };
    dataSource: SharedTableResult = {
        totalItems: 0,
        data: []
    };
    templateFilter: any = {
        customListFilter: {
            searchFilter: '',
            page: 0,
            pageSize: this.coreSession.pageSize
        },
    };
    navLinks: any[] = [];
    constructor(
        private coreSession: CoreSession,
        private excelSheetTemplatesService: ExcelSheetTemplatesService,
        private translateService: TranslateService,
        private sessionDataProvider: SessionDataProvider,
        private navService: NavBarService) {
    }

    openCategoryEntryDialog() {
      this.CategoryEntryDialog.Show(true).then((res) => { });
    }
    ngOnInit() {
        this.coreSession.SetTitle('Excel Sheet Dynamic Templates');
        this.prepareNavigation();
        var index = this.navLinks.findIndex(x => x.link == this.navService.searchMenuURL);
        if (index == -1) index = 0;
        this.navLinks[index].isActive = true;
        this.setNavigation(this.navLinks[index]);
        this.TableData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.excelSheetDynamicTemplatesURL, MenuActions.Add);
        this.addActions();
        this.fillExcelSheetTypesList();
        this.populateTemplateList();
    }
    prepareNavigation() {
        for (let menu of this.sessionDataProvider.allMenusAvailable) {
            if (menu.menuId === ('900-001-12')) {
                let newName = {
                    label: menu.menuName,
                    isActive: false,
                    hidden: false,
                    disable: false,
                };
                this.navLinks.push(newName);
            }
        }
    }
    addActions() {
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.excelSheetDynamicTemplatesURL, MenuActions.Delete)) {
            this.TableData.rowOperations.push(
                {
                    operation: RowOperation.delete,
                    title: 'Desc_Delete',
                    icon: 'fa-solid fa-trash',
                    color: '#f1685e'
                });
        }

        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.excelSheetDynamicTemplatesURL, MenuActions.Edit)) {
            this.TableData.rowOperations.push({
                operation: RowOperation.edit,
                title: "Desc_Edit",
                icon: "fa-regular fa-pen-to-square",
                color: "#12344d"
            });

        }

        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.excelSheetDynamicTemplatesURL, MenuActions.View)) {
            this.TableData.rowOperations.push(
                {
                    operation: RowOperation.View,
                    title: "Desc_View",
                    icon: "fa-regular fa-eye",
                    color: "#12344d"
                });
        }
    }

    fillExcelSheetTypesList() {
        this.excelSheetTypesList = [
            { excelSheetTypeId: ExcelSheetTypes.Input, excelSheetTypeName: this.translateService.instant(ConstantMessages.Desc_Input_Caption) },
        ]
    }
    populateTemplateList() {
        this.coreSession.ModalLoading.Show();
        this.excelSheetTemplatesService.getExcelSheetDynamicTemplateListSharedTable(this.templateFilter).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
                this.dataSource = {
                    totalItems: response.data.totalItems,
                    data: response.data.data,
                }
            } else {
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
        }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        });
    }
    onTableFilterChanged(query: any) {
        this.templateFilter.customListFilter = query;
        this.populateTemplateList();
    }
    onRowOperation(event: any) {
        switch (event.operation) {
            case RowOperation.edit:
                this.isEditMode = true;
                this.getExcelSheetTemplateData(event.object);
                break;
            case RowOperation.delete:
                this.onDeleteClicked(event.object);
                break;
            case RowOperation.View:
                this.isViewMode = true;
                this.isEditMode = false;
                this.getExcelSheetTemplateData(event.object);
        }
    }
    getExcelSheetTemplateData(object) {
        this.coreSession.ModalLoading.Show();
        this.excelSheetTemplatesService.getExcelSheetTemplateData(object.vendorId, object.excelSheetTypeId).subscribe(
            response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                    this.templateObject = response.data;
                    this.templateObject.vendorNameCode = object.vendorNameCode;
                    this.templateObject.excelSheetType = object.excelSheetType;
                    this.showTemplateEntrySlider = true;
                    this.openCategoryEntryDialog();
                } else {
                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                }
            },
            error => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
        );
    }
    onDeleteClicked(event: any) {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
            (result: DialogResult) => {
                if (result === DialogResult.Yes) {
                    this.coreSession.ModalLoading.Show();
                    this.excelSheetTemplatesService
                        .deleteExcelSheetData(event.vendorId, event.excelSheetTypeId)
                        .subscribe(
                            response => {
                                this.coreSession.ModalLoading.Hide();
                                if (response.status != null && response.status >= 0) {
                                    this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
                                    this.populateTemplateList();
                                } else {
                                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                                }
                            },
                            error => {
                                this.coreSession.ModalLoading.Hide();
                                this.coreSession.showError("", ConstantMessages.ErrorHappened);
                            }
                        );
                }
            });
    }
    openSliderToAddNewTemplate() {
        this.templateObject = null;
        this.showTemplateEntrySlider = true;
        this.openCategoryEntryDialog();
    }
    openSliderToEditTemplate(templateObj) {
        this.templateObject = templateObj;
        this.showTemplateEntrySlider = true;
        this.openCategoryEntryDialog()
    }
    closeTemplateEntrySlider() {
        this.isEditMode = false;
        this.isViewMode = false;
        this.templateObject = null;
        this.showTemplateEntrySlider = false;
        this.CategoryEntryDialog.Close();
        this.coreSession.SetTitle('Excel Sheet Dynamic Templates');
    }
    onSaveTemplateClick() {
        this.saveSubject.next()
    }
    afterSaveTemplate() {
        this.closeTemplateEntrySlider();
        this.populateTemplateList();
    }
    ngOnDestroy() {
        this.navService.setNavigation([]);
    }
    onChangeSeletedTab(tab) {
        this.setNavigation(tab);
    }
    setNavigation(tab) {
        let navigationList: NavigationData[] = [];
        navigationList.push(
            {
                Name: 'Desc_Settings',
                URL: ConstantURLs.settingsURL
            },
            {
                Name: tab.label,
                URL: undefined
            }
        );
        this.navService.setNavigation(navigationList);
    }
}

    //#endregion

