import { HttpErrorResponse } from '@angular/common/http';
import { Component, Input, OnDestroy, OnInit, Output, EventEmitter } from '@angular/core';
import { FormControl, FormGroup, Validators } from '@angular/forms';
import { TranslateService } from '@ngx-translate/core';
import { Observable } from 'rxjs';
import { CoreSession } from '../../../../core/core.session';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { ITextInput } from '../../../../shared/models/shared-table/custom-list.interface';
import { NotificationService } from '../../../../core/notification.service';
import { NotificationTypes } from '../../../../shared/models/enums/notification-Types.enum';

@Component({
  selector: 'app-notification-template-entry',
  templateUrl: './notification-template-entry.component.html',
  styleUrls: ['./notification-template-entry.component.css']
})
export class NotificationTemplateEntryComponent implements OnInit, OnDestroy {

  @Output() onSavingTemplateEvent = new EventEmitter();
  @Input() saveEvent: Observable<void>;
  @Input() templateObject: any = null;
  @Input() isEditMode: boolean = false;
  @Input() isViewMode: boolean = false;
  showCommCalcTypeInEntry: boolean = false;
  saveBtnSubscription: any;
  templateForm: FormGroup;
  notificationTypesList: any[] = [];

  otherTemplateNameLanguages: any[] = [];
  templateNameInput: ITextInput = {
    formControlName: 'description',
    placeHolder: 'Desc_Template_Name',
    label: 'Desc_Template_Name',
    isRequierd: !this.isViewMode,
  };

  otherTemplateHeaderLanguages: any[] = [];
  templateHeaderInput: ITextInput = {
    formControlName: 'headerDescription',
    placeHolder: 'Desc_Header',
    label: 'Desc_Header',
    isRequierd: !this.isViewMode,
    isTextArea: true,
  };
  headerHint = '';

  otherTemplateBodyLanguages: any[] = [];
  templateBodyInput: ITextInput = {
    formControlName: 'bodyDescription',
    placeHolder: 'Desc_Body',
    label: 'Desc_Body',
    isRequierd: !this.isViewMode,
    isTextArea: true,
  };
  bodyHint = '';
  constructor(
    public coreSession: CoreSession,
    private translateService: TranslateService,
    private notificationService: NotificationService) {
  }
  ngOnInit() {
    this.subscribeSaveClick();
    if (this.isEditMode || this.isViewMode) {
      this.coreSession.SetTitle(this.isEditMode ? 'Edit Notification Template' : 'View Notification Template');
      this.getNotificationTemplateDataOnEdit();
    } else {
      this.coreSession.SetTitle('Add Notification Template');
      this.templateObject = {
        templateId: -1,
        description: null,
        headerDescription: null,
        bodyDescription: null,
        notificationTypeId: null,
        inactive: false,
        descriptionsList: [],
        headerDescriptionsList: [],
        bodyDescriptionsList: []
      }
    }
    if (!this.isEditMode && !this.isViewMode) {
      this.isValidToGetCCNotificationType();
    } else {
      this.showCommCalcTypeInEntry = true;
      this.fillNotificationTypesList();
    }


    this.initTemplateForm();
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }
  subscribeSaveClick() {
    this.saveBtnSubscription = this.saveEvent.subscribe(() => {
      this.onSaveNotificationTemplate();
    });
  }
  fillNotificationTypesList() {
    this.notificationTypesList = [];
    this.notificationTypesList = [
      { notificationTypeId: NotificationTypes.Recall, notificationTypeName: this.translateService.instant(ConstantMessages.Desc_Recall_Caption) },
      { notificationTypeId: NotificationTypes.QualityControl, notificationTypeName: this.translateService.instant(ConstantMessages.Desc_Quality_Control_Caption) }
    ]
    if (this.showCommCalcTypeInEntry) {
      this.notificationTypesList.push({ notificationTypeId: NotificationTypes.FinalIncentive, notificationTypeName: this.translateService.instant(ConstantMessages.Desc_Commission_Calculation) });
    }
  }

  isValidToGetCCNotificationType() {
    this.coreSession.ModalLoading.Show();
    this.notificationService.isValidToGetCCNotificationType().subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        if (response.data) {
          this.showCommCalcTypeInEntry = true;
        }
        this.fillNotificationTypesList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
     
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  initTemplateForm() {
    this.templateForm = new FormGroup({
      description: new FormControl({ value: this.templateObject.description, disabled: this.isViewMode }, Validators.required),
      headerDescription: new FormControl({ value: this.templateObject.headerDescription, disabled: this.isViewMode }, Validators.required),
      bodyDescription: new FormControl({ value: this.templateObject.bodyDescription, disabled: this.isViewMode }, Validators.required),
      notificationTypeId: new FormControl({ value: this.templateObject.notificationTypeId, disabled: this.isViewMode || this.isEditMode }, Validators.required),
      inactive: new FormControl({ value: this.templateObject.inactive, disabled: this.isViewMode }),
    });
    this.onChangeNotificationType();
  }
  getNotificationTemplateDataOnEdit() {
    this.coreSession.ModalLoading.Show();
    var templateId = this.templateObject.templateId;
    this.notificationService.getNotificationTemplateData(templateId).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.templateObject.descriptionsList = response.data.descriptionsList;
          this.templateObject.headerDescriptionsList = response.data.headerDescriptionsList;
          this.templateObject.bodyDescriptionsList = response.data.bodyDescriptionsList;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  onChangeNotificationType() {
    this.headerHint = '';
    this.bodyHint = '';
    if (this.templateForm.get('notificationTypeId').value === NotificationTypes.Recall) {
      this.headerHint = ConstantMessages.Desc_Recall_Header_Hint;
      this.bodyHint = ConstantMessages.Desc_Recall_Body_Hint;
    } else if (this.templateForm.get('notificationTypeId').value === NotificationTypes.QualityControl) {
      this.headerHint = ConstantMessages.Desc_Quality_Control_Header_Hint;
      this.bodyHint = ConstantMessages.Desc_Quality_Control_Body_Hint;
    } else if (this.templateForm.get('notificationTypeId').value === NotificationTypes.FinalIncentive) {
      this.headerHint = ConstantMessages.Desc_Commission_Calculation_Header_Hint;
      this.bodyHint = ConstantMessages.Desc_Commission_Calculation_Body_Hint;
    }
  }
  onOtherLanguagesTextSubmitted(event) {
    this.otherTemplateNameLanguages = event;
  }
  onOtherHeaderLanguagesTextSubmitted(event) {
    this.otherTemplateHeaderLanguages = event;
  }
  onOtherBodyLanguagesTextSubmitted(event) {
    this.otherTemplateBodyLanguages = event;
  }
  fillTemplateObject() {
    this.templateObject.description = this.templateForm.get('description').value;
    this.templateObject.headerDescription = this.templateForm.get('headerDescription').value;
    this.templateObject.bodyDescription = this.templateForm.get('bodyDescription').value;
    this.templateObject.notificationTypeId = this.templateForm.get('notificationTypeId').value;
    this.templateObject.inactive = this.templateForm.get('inactive').value;

    // template name
    if (this.otherTemplateNameLanguages.findIndex(x => x.languageId === this.coreSession.selectedLanguageId) < 0) {
      this.otherTemplateNameLanguages.push({
        languageId: this.coreSession.selectedLanguageId,
        description: this.templateObject.templateName
      });
    }
    this.templateObject.descriptionsList = this.otherTemplateNameLanguages;

    // template header
    if (this.otherTemplateHeaderLanguages.findIndex(x => x.languageId === this.coreSession.selectedLanguageId) < 0) {
      this.otherTemplateHeaderLanguages.push({
        languageId: this.coreSession.selectedLanguageId,
        description: this.templateObject.templateHeader
      });
    }
    this.templateObject.headerDescriptionsList = this.otherTemplateHeaderLanguages;

    // template body
    if (this.otherTemplateBodyLanguages.findIndex(x => x.languageId === this.coreSession.selectedLanguageId) < 0) {
      this.otherTemplateBodyLanguages.push({
        languageId: this.coreSession.selectedLanguageId,
        description: this.templateObject.templateBody
      });
    }
    this.templateObject.bodyDescriptionsList = this.otherTemplateBodyLanguages;
  }
  onSaveNotificationTemplate() {
    if (this.templateForm.invalid) {
      this.coreSession.markFormGroupTouched(this.templateForm);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    this.fillTemplateObject();
    this.coreSession.ModalLoading.Show();
    this.notificationService.saveNotificationTemplate(this.templateObject).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSavingTemplateEvent.emit();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
}
