import { HttpErrorResponse } from '@angular/common/http';
import { Component, OnDestroy, OnInit } from '@angular/core';
import { FormControl, FormGroup } from '@angular/forms';
import { TranslateService } from '@ngx-translate/core';
import { Subject } from 'rxjs';
import { CoreSession } from '../../../../core/core.session';
import { NotificationService } from '../../../../core/notification.service';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { NavBarService } from '../../../../core/sidebar/sidebar.service';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { NotificationTypes } from '../../../../shared/models/enums/notification-Types.enum';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { NavigationData } from '../../../../shared/models/navigation-data/navigation-data.model';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';

@Component({
  selector: 'app-notification-template-list',
  templateUrl: './notification-template-list.component.html',
  styleUrls: ['./notification-template-list.component.css']
})
export class NotificationTemplateListComponent implements OnInit, OnDestroy {

  saveSubject: Subject<void> = new Subject<void>();
  showTemplateEntrySlider = false;
  templateFilterForm: FormGroup;
  isEditMode = false;
  isViewMode = false;
  templateObject: any = null;
  notificationTypesList: any[] = [];
  hideCommCalcTypeInEntry: boolean = false;

  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Template_Name', key: 'description', isSortable: true, width: '40%' },
      { title: 'Desc_Notification_Type', key: 'notificationTypeName', isSortable: true, width: '40%' },
      { title: 'Desc_Status', key: 'templateStatuName', isSortable: true, width: '20%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  templateFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    notificationTypeId: -1,
    includeInactive: false
  };
  navLinks: any[] = [];
  constructor(
    private coreSession: CoreSession,
    private notificationService: NotificationService,
    private translateService: TranslateService,
    private sessionDataProvider: SessionDataProvider,
    private navService: NavBarService) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Notification Templates');
    this.prepareNavigation();
    var index = this.navLinks.findIndex(x => x.link == this.navService.searchMenuURL);
    if (index == -1) index = 0;
    this.navLinks[index].isActive = true;
    this.setNavigation(this.navLinks[index]);
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.notificationTemplatesURL, MenuActions.Add);

    this.addActions();
    this.fillNotificationTypesList();
    this.initFilterForm();
    this.populateTemplateList();
  }
  prepareNavigation() {
    for (let menu of this.sessionDataProvider.allMenusAvailable) {
      if (menu.menuId === ('900-001-11')) {
        let newName = {
          label: menu.menuName,
          isActive: false,
          hidden: false,
          disable: false,
        };
        this.navLinks.push(newName);
      }
    }
  }
  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.notificationTemplatesURL, MenuActions.Edit)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.edit,
          title: "Desc_Edit",
          icon: "fa-regular fa-pen-to-square",
          color: "#12344d",
        });
      this.TabelData.rowOperations.push({
        operation: RowOperation.Activate,
        title: "Desc_Reactivate",
        icon: "fa fa-toggle-on",
        color: '#02b875',
        showWhenKeyValueTrue: true,
        showHideOptionPerLine: true,
        controlKeyName: 'inactive'
      });
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Deactivate,
          title: 'Desc_Deactivate',
          icon: 'fa-solid fa-ban',
          color: '#f1685e',
          showWhenKeyValueFalse: true,
          showHideOptionPerLine: true,
          controlKeyName: 'inactive'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.notificationTemplatesURL, MenuActions.View)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.View,
          title: 'Desc_View',
          icon: 'fa-regular fa-eye',
          color: '#12344d',
        });
    }
  }
  initFilterForm() {
    this.templateFilterForm = new FormGroup({
      notificationTypeId: new FormControl(),
      includeInactive: new FormControl(false)
    });
  }
  fillNotificationTypesList() {
    this.notificationTypesList = [
      { notificationTypeId: NotificationTypes.Recall, notificationTypeName: this.translateService.instant(ConstantMessages.Desc_Recall_Caption) },
      { notificationTypeId: NotificationTypes.QualityControl, notificationTypeName: this.translateService.instant(ConstantMessages.Desc_Quality_Control_Caption) }
    ]
  }
  populateTemplateList() {
    this.templateFilter.notificationTypeId = this.templateFilterForm.get('notificationTypeId').value ? this.templateFilterForm.get('notificationTypeId').value : -1;
    this.templateFilter.includeInactive = this.templateFilterForm.get('includeInactive').value == null ? false : this.templateFilterForm.get('includeInactive').value;

    this.coreSession.ModalLoading.Show();
    this.notificationService.getNotificationTemplateListSharedTable(this.templateFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data,
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  onTableFilterChanged(query: any) {
    this.templateFilter.customListFilter = query;
    this.populateTemplateList();
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.edit:
        this.isEditMode = true;
        this.openSliderToEditTemplate(event.object);
        break;
      case RowOperation.View:
        this.isViewMode = true;
        this.openSliderToEditTemplate(event.object);
        break;
      case RowOperation.Deactivate: {
        this.UpdateActiveStatus(event.object, false);
        break;
      }
      case RowOperation.Activate: {
        this.UpdateActiveStatus(event.object, true);
        break;
      }
    }
  }
  UpdateActiveStatus(template: any, isActive: boolean) {
    var msg = '';
    if (isActive) {
      msg = ConstantMessages.MsgReactivateConfirmation;
    } else {
      msg = ConstantMessages.MsgDeactivateConfirmation;
    }
    // ask user to confirm on active/inactive
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(msg), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.updateNotificationTemplateActiveStatusInDB(template, isActive);
        }
      });
  }
  updateNotificationTemplateActiveStatusInDB(template, isActive) {
    this.coreSession.ModalLoading.Show();
    var filter = {
      templateId: template.templateId,
      inactive: !isActive
    }
    this.notificationService.updateNotificationTemplateActiveStatus(filter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
        this.populateTemplateList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  onFindButtonClicked() {
    this.templateFilter.customListFilter.page = 0;
    this.populateTemplateList();
  }
  onResetFilter() {
    this.templateFilterForm.reset();
    this.templateFilterForm.get('includeInactive').setValue(false);
  }
  openSliderToAddNewTemplate() {
    this.templateObject = null;
    this.hideCommCalcTypeInEntry = this.isListHasCommCalcType();
    this.showTemplateEntrySlider = true;
  }
  openSliderToEditTemplate(templateObj) {
    this.templateObject = templateObj;
    if (this.templateObject.notificationTypeId == NotificationTypes.FinalIncentive.valueOf()) {
      this.hideCommCalcTypeInEntry = false;
    } else if (this.templateObject.notificationTypeId != NotificationTypes.FinalIncentive.valueOf() && !this.isListHasCommCalcType()) {
      this.hideCommCalcTypeInEntry = false;
    } else {
      this.hideCommCalcTypeInEntry = true;
    }
    this.showTemplateEntrySlider = true;
  }

  isListHasCommCalcType() {
    if (this.dataSource.data.length > 0) {
      var index = this.dataSource.data.findIndex(x => x.notificationTypeId == NotificationTypes.FinalIncentive.valueOf() && !x.inactive);
      if (index > -1) {
        return true;
      } else {
        return false;
      }
    } else {
      return false;
    }
  }

  closeTemplateEntrySlider() {
    this.isEditMode = false;
    this.isViewMode = false;
    this.templateObject = null;
    this.showTemplateEntrySlider = false;
    this.coreSession.SetTitle('Notification Templates');
  }
  onSaveTemplateClick() {
    this.saveSubject.next()
  }
  afterSaveTemplate() {
    this.closeTemplateEntrySlider();
    this.populateTemplateList();
  }
  ngOnDestroy() {
    this.navService.setNavigation([]);
  }
  onChangeSeletedTab(tab) {
    this.setNavigation(tab);
  }
  setNavigation(tab) {
    let navigationList: NavigationData[] = [];
    navigationList.push(
      {
        Name: 'Desc_Settings',
        URL: ConstantURLs.settingsURL
      },
      {
        Name: tab.label,
        URL: undefined
      }
    );
    this.navService.setNavigation(navigationList);
  }
}

