import { Component, OnInit, EventEmitter, Input, Output } from '@angular/core';
import { ITextInput, CustomListFilter } from '../../../../shared/models/shared-table/custom-list.interface';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { CustomerService } from '../../../../shared/services/data-definition/customer/customers.service';
import { CoreSession } from '../../../../core/core.session';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { TranslateService } from '@ngx-translate/core';
import { ResponseModel } from '../../../../shared/models/api-models/api-models';
import { Observable, forkJoin } from 'rxjs';
import { EchoNotifications } from '../../../../echo/models/echoNotifications.model';
import { EchoNotificationService } from '../../../../shared/services/customer-app-properties/echo-notification.service';
import { ShepherdService } from 'angular-shepherd';
import { Router } from '@angular/router';
import { TourService } from '../../../../shared/shepherd-config/shepherd-config';
import { ComponentVisitService } from '../../../../shared/shepherd-config/trackTour.service';
@Component({
  selector: 'app-notification-management-entry',
  templateUrl: './notification-management-entry.component.html',
  styleUrls: ['./notification-management-entry.component.css']
})
export class NotificationManagementEntryComponent implements OnInit {
  @Input() events: Observable<void>;
  @Output() onSendCompleted = new EventEmitter()

  //start of text box properties
  bodyTextInput: ITextInput = {
    formControlName: "body",
    placeHolder: "Desc_Body_English",
    label: "Desc_Body_English",
    isTextArea: true
  };
  arBodyTextInput: ITextInput = {
    formControlName: "arBody",
    placeHolder: "Desc_Body_Arabic",
    label: "Desc_Body_Arabic",
    isTextArea: true
  };
  subjectTextInput: ITextInput = {
    formControlName: "subject",
    placeHolder: "Desc_Subject_English",
    label: "Desc_Subject_English"
  };

  arSubjectTextInput: ITextInput = {
    formControlName: "arSubject",
    placeHolder: "Desc_Subject_Arabic",
    label: "Desc_Subject_Arabic"
  };

  customersDropDownSettings = {
    singleSelection: false,
    idField: 'customerOutletIds',
    textField: 'customerOutletName',
    selectAllText: '',
    unSelectAllText: '',
    searchPlaceholderText: '',
    itemsShowLimit: 1,
    maxHeight: 150,
    allowSearchFilter: true,
  };

  groupsDropDownSettings = {
    singleSelection: false,
    idField: 'groupId',
    textField: 'groupCode',
    selectAllText: '',
    unSelectAllText: '',
    searchPlaceholderText: '',
    itemsShowLimit: 1,
    maxHeight: 150,
    allowSearchFilter: true,
  };

  channelsDropDownSettings = {
    singleSelection: false,
    idField: 'channelSubChannelIds',
    textField: 'channelSubChannelDescriptions',
    selectAllText: '',
    unSelectAllText: '',
    searchPlaceholderText: '',
    itemsShowLimit: 1,
    maxHeight: 150,
    allowSearchFilter: true,
  };

  addedGroups: any[] = [];
  addedChannels: any[] = [];
  addedCustomers: any[] = [];
  customerId: number;
  outletId: number;
  customerGroupsDataSource: any[] = [];
  channelSubChannelDataSource: any[] = [];
  customersDataSource: any[] = [];
  //end of text box properties
  //start of general properties
  echoNotificationsObj: EchoNotifications;
  sendBtnSubscription: any;
  myForm: FormGroup;

  selectedImageName: string;
  selectedVideoName: string;

  imageToUpload: File = null;
  videoToUpload: File = null;

  formData: FormData = new FormData();

  subject: string;
  body: string;
  arSubject: string;
  arBody: string;
  //end of general properties
  constructor(private customerService: CustomerService,
    private echoNotificationService: EchoNotificationService,
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private router: Router
    , private readonly shepherdService: ShepherdService,
    private tourService: TourService,
    private componentVisitService: ComponentVisitService
  ) { }

  startTour() {
    // this.tourStartTimeout = setTimeout((data => {

    //   if (this.tourService.isTourCompletedForComponent('send-notification-entry')) {
    //     return;
    //   }
    //   const elementSelectors = ['#subject-def', '#body-def', '#browseImage-def', '#browseVid-def'
    //     , '#groups-def', '#Customers-def'
    //   ] // Target each column by its unique ID
    //   const titles = [];
    //   const texts = [];
    //   for (let index = 0; index < elementSelectors.length; index++) {
    //     const { title, text } = this.getTitleAndText(index);
    //     titles.push(title);
    //     texts.push(text);
    //   }
    //   const steps = this.tourService.generateSteps(elementSelectors, titles, texts);
    //   this.shepherdService.defaultStepOptions = this.tourService.defaultStepOptions;
    //   this.shepherdService.modal = true;
    //   this.shepherdService.addSteps(steps);
    //   this.shepherdService.start();
    // }), 1000)
  }
  private tourStartTimeout: any;
  ngAfterViewInit() {
    this.startTour()
  }
  getTitleAndText(index: number): { title: string, text: string } {
    let title = '';
    let text = '';

    switch (index) {
      case 0:
        title = "Let's go to define your item";
        text = 'Here you can define your items.';
        break;
      case 1:
        title = "Let's go to define your customer";
        text = "Here you can define your customer";
        break;
      case 2:
      case 3:
      case 4:
      case 5:
        title = `Step ${index + 1} Title`;
        text = `Step ${index + 1} Content`;
        break;
      default:
        title = `Step ${index + 1}`;
        text = `Content in defult at bottom.`;
    }

    return { title, text };
  }
  ngOnInit() {
    this.initForm();
    this.setSelectAllLabelsForMultiSelect();
    this.populateEchoGroups();
    this.populateEchoChannels();
    this.populateEchoCustomers();
    this.subscribeSendClick();
  }

  ngOnDestroy() {
    this.sendBtnSubscription.unsubscribe();
    if (this.tourStartTimeout) {
      clearTimeout(this.tourStartTimeout);
    }
    this.componentVisitService.markComponentVisited('send-notification-entry');

  }

  //start of filling screen combos

  setSelectAllLabelsForMultiSelect() {
    this.customersDropDownSettings.selectAllText = this.translateService.instant(ConstantMessages.SelectAllCaption);
    this.customersDropDownSettings.unSelectAllText = this.translateService.instant(ConstantMessages.UnSelectAllCaption);
    this.customersDropDownSettings.searchPlaceholderText = this.translateService.instant(ConstantMessages.SearchCaption);

    this.channelsDropDownSettings.selectAllText = this.translateService.instant(ConstantMessages.SelectAllCaption);
    this.channelsDropDownSettings.unSelectAllText = this.translateService.instant(ConstantMessages.UnSelectAllCaption);
    this.channelsDropDownSettings.searchPlaceholderText = this.translateService.instant(ConstantMessages.SearchCaption);

    this.groupsDropDownSettings.selectAllText = this.translateService.instant(ConstantMessages.SelectAllCaption);
    this.groupsDropDownSettings.unSelectAllText = this.translateService.instant(ConstantMessages.UnSelectAllCaption);
    this.groupsDropDownSettings.searchPlaceholderText = this.translateService.instant(ConstantMessages.SearchCaption);
  }

  populateEchoGroups() {
    this.customerService.getAllEchoCustomerGroups().subscribe(response => {
      this.customerGroupsDataSource = response.data;
    }, (error: HttpErrorResponse) => {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }

  populateEchoChannels() {
    this.customerService.GetAllEchoCustomerChannels().subscribe(response => {
      this.channelSubChannelDataSource = response;
    }, (error: HttpErrorResponse) => {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }

  populateEchoCustomers() {
    this.customerService.getAllEchoCustomers().subscribe(response => {
      this.customersDataSource = response.data;
    }, (error: HttpErrorResponse) => {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  //end of filling screen combos

  //start of general events
  initForm() {
    this.myForm = new FormGroup({
      subject: new FormControl("", Validators.required),
      arSubject: new FormControl("", Validators.required),
      body: new FormControl("", Validators.required),
      arBody: new FormControl("", Validators.required),
      groups: new FormControl(this.addedGroups),
      channels: new FormControl(this.addedChannels),
      customers: new FormControl(this.addedCustomers),
      imagePath: new FormControl({ value: this.selectedImageName, disabled: true }),
      videoPath: new FormControl({ value: this.selectedVideoName, disabled: true })
    });
  }

  onChangeDetected(event: any, isImageSelected: boolean) {
    if (event.target.files[0].size > 10000000) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFileIsTooLarge));
    } else {
      if (isImageSelected) {
        if (event && event.target && event.target.files[0] && event.target.files[0].name) {
          this.selectedImageName = event.target.files[0].name;
          this.imageToUpload = event.target.files[0];
        } else {
          this.selectedImageName = "";
          this.imageToUpload = null;
        }
        this.myForm.controls['imagePath'].setValue(this.selectedImageName);
      } else {
        if (event && event.target && event.target.files[0] && event.target.files[0].name) {
          this.selectedVideoName = event.target.files[0].name;
          this.videoToUpload = event.target.files[0];
        } else {
          this.selectedVideoName = "";
          this.videoToUpload = null;
        }
        this.myForm.controls['videoPath'].setValue(this.selectedVideoName);
      }
    }
  }
  //end of general events
  //start of saving events
  subscribeSendClick() {
    this.sendBtnSubscription = this.events.subscribe(() => {
      this.onSendClicked();
    });
  }

  onSendClicked() {
    if (this.isNotificationValid()) {
      this.fillNotificationsObj();

      this.echoNotificationService.saveCustomerNotifications(this.formData).subscribe(
        (response: ResponseModel) => {
          if (response.status != null && response.status >= 0) {
            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
            this.onSendCompleted.emit();
          } else {
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
          }
        },
        (error: HttpErrorResponse) => {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
      );
    }
  }

  fillNotificationsObj() {
    this.echoNotificationsObj = new EchoNotifications();
    this.echoNotificationsObj.subject = this.subject;
    this.echoNotificationsObj.body = this.body;
    this.echoNotificationsObj.customers = this.addedCustomers;
    this.echoNotificationsObj.groups = this.addedGroups;
    this.echoNotificationsObj.channels = this.addedChannels;
    this.echoNotificationsObj.imageName = this.selectedImageName;
    this.echoNotificationsObj.videoName = this.selectedVideoName;
    this.echoNotificationsObj.arabicSubject = this.arSubject;
    this.echoNotificationsObj.arabicBody = this.arBody;
    if (this.imageToUpload) {
      this.formData.append('files', this.imageToUpload, this.imageToUpload.name);
    }
    if (this.videoToUpload) {
      this.formData.append('files', this.videoToUpload, this.videoToUpload.name);
    }
    this.formData.append('notification', JSON.stringify(this.echoNotificationsObj));
  }

  isNotificationValid() {
    this.markAsTouched();
    this.fillFormDataToProperties();
    if (!this.subject || this.subject === "" || !this.arSubject || this.arSubject === "") {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgEnterSubject));
      return false;
    }
    else if (!this.body || this.body === "" || !this.arBody || this.arBody === "") {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgEnterBody));
      return false;
    } else if ((!this.addedCustomers || this.addedCustomers.length <= 0) && (!this.addedGroups || this.addedGroups.length <= 0) && (!this.addedChannels || this.addedChannels.length <= 0)) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectOneOfCustomerFilters));
      return false;
    }

    return true;
  }

  fillFormDataToProperties() {
    this.addedCustomers = this.myForm.get('customers').value;
    this.addedGroups = this.myForm.get('groups').value;
    this.addedChannels = this.myForm.get('channels').value;
    this.subject = this.myForm.get('subject').value;
    this.body = this.myForm.get('body').value;
    this.arBody = this.myForm.get('arBody').value;
    this.arSubject = this.myForm.get('arSubject').value;
  }

  markAsTouched() {
    this.myForm.get('subject').markAsTouched();
    this.myForm.get('body').markAsTouched();
    this.myForm.get('arSubject').markAsTouched();
    this.myForm.get('arBody').markAsTouched();
  }
  //end of saving events
}
