import { Component, OnInit } from '@angular/core';
import { PaymentModel } from '../../../../shared/models/payment/payment.model';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { MultiSelectionOperation, RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { GeneralFilter } from '../../../../shared/models/general/generalFilter.model';
import { CoreSession } from '../../../../core/core.session';
import { PaymentService } from '../payment.service';
import { HttpErrorResponse } from '@angular/common/http';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { ISharedItemsProperties } from '../../../../shared/components/items/shared-items-properties.interface';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { TranslateService } from '@ngx-translate/core';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { Subject } from 'rxjs';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ReportMode } from '../../../../shared/models/enums/report-mode.enum';
import { DownPaymentService } from '../../../../shared/services/customer-operation/down-payment/down-payment.service';
import { PdfCreatorService } from '../../../../shared/services/pdf-creator.service';

@Component({
  selector: 'app-down-payments-list',
  templateUrl: './down-payments-list.component.html',
  styleUrls: ['./down-payments-list.component.css']
})
export class DownPaymentsListComponent implements OnInit {

  showSliderDownPaymenyEntry = false;
  saveSubject: Subject<void> = new Subject<void>();

  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    showEditButton: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Customer', key: 'customerNameCode', isSortable: true, width: '9%' },
      { title: 'Desc_Outlet', key: 'outletNameCode', isSortable: true, width: '9%' },
      { title: 'Desc_Employee', key: 'employeeNameCode', isSortable: true, width: '9%' },
      { title: 'Desc_Payment_ID', key: 'customerPaymentId', isSortable: true, width: '12%' },
      { title: 'Desc_Date', key: 'paymentDate', isSortable: false, width: this.isDivisionLevel() ? '11%' : '15', isDate: true },
      { title: 'Desc_Amount', key: 'appliedAmount', isSortable: false, width: this.isDivisionLevel() ? '9%' : '13', numberOfDigits: true },
      { title: 'Desc_Type', key: 'paymentTypeName', isSortable: false, width: '9%' },
      { title: 'Desc_Bank', key: 'bankName', isSortable: false, width: '8%' },
      { title: 'Desc_Branch', key: 'branchName', isSortable: false, width: '8%' },
      { title: 'Desc_Status', key: 'paymentStatusName', isSortable: false, width: '8%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  generalFilter: GeneralFilter = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
  };
  constructor(
    private coreSession: CoreSession,
    private downPaymentService: DownPaymentService,
    private translateService: TranslateService,
    private sessionData: SessionDataProvider,
    private pdfCreatorService: PdfCreatorService) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Down Payments');
    if (this.isDivisionLevel()) {
      this.TabelData.columns.push(
        { title: 'Desc_Division', key: 'divisionName', isSortable: false, width: '8%' },
      );
    }
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.downPaymentsListURL, MenuActions.Add);
    this.addDeleteAction();
    this.getDownPaymentsList();
  }
  addDeleteAction() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.downPaymentsListURL, MenuActions.Void)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Void',
          icon: 'fa-solid fa-trash',
          color: '#f1685e'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.downPaymentsListURL, MenuActions.Print)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Print,
          title: 'Desc_Print',
          icon: 'fa-solid fa-print',
          color: '#12344d'
        });
    }

  }
  isDivisionLevel() {
    return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true")
  }
  getDownPaymentsList() {
    this.coreSession.ModalLoading.Show();
    this.downPaymentService.getDownPaymentsList(this.generalFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = <SharedTableResult>response.data;
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    }
    );
  }
  onTableFilterChanged(query: any) {
    this.generalFilter.customListFilter = query;
    this.getDownPaymentsList();
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.delete: {
        this.voidDownPayments(event.object);
        break;
      }
      case RowOperation.Print: {
        this.exportCollectionReport(event);
        break;
      }
    }
  }

  exportCollectionReport(event: any) {
    let trans = {
      DivisionId: event.object.divisionId ? event.object.divisionId : -1,
      TransactionId: event.object.customerPaymentId,
      reportMode: ReportMode.DownPayment,
      isReprintReport: true
    }
    this.pdfCreatorService.ExportReport(trans);
  }
  onFindButtonClicked(result) {
    this.generalFilter.customListFilter.page = 0;
    this.generalFilter.employeeId = result.employeeId;
    this.generalFilter.transactionId = result.transactionId;
    this.generalFilter.customerId = result.customerId;
    this.generalFilter.outletId = result.outletId;
    this.generalFilter.dateModel = result.transactionDate;
    this.generalFilter.divisionId = result.divisionId;
    this.getDownPaymentsList();
  }
  voidDownPayments(downPayment: any) {
    if (downPayment.voided) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectedPaymentVoided));
      return;
    }
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.downPaymentService.voidDownPayments(downPayment).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              // voided Successfully
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
              this.getDownPaymentsList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
          }
          );
        }
      }
    );
  }
  openSliderToAddDownPayment() {
    this.showSliderDownPaymenyEntry = true;
  }
  closeDownPaymentSlider() {
    this.showSliderDownPaymenyEntry = false;
    this.coreSession.SetTitle('Down Payments');
  }
  onSavePaymentClick() {
    this.saveSubject.next()
  }
  afterSaveDownPayment() {
    this.closeDownPaymentSlider();
    this.getDownPaymentsList();
  }
}

