import { Component, OnInit, Input, Output, EventEmitter, ViewChild } from '@angular/core';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { CoreSession } from '../../../../core/core.session';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { PaymentModel } from '../../../../shared/models/payment/payment.model';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { PaymentTypes } from '../../../../shared/models/enums/payment-types.enum';
import { PaymentService } from '../payment.service';
import { BranchModel } from '../../../../shared/models/payment/branch.model';
import { BankModel } from '../../../../shared/models/payment/bank.model';
import { PaymentTypeModel } from '../../../../shared/models/payment/payment-type.model';
import { FormGroup, Validators, FormControl } from '@angular/forms';
import { DateTimeModel } from '../../../../shared/models/dateTimeModel/date-time.model';
import { IDateTimePickerProperties } from '../../../../shared/models/dateTimeModel/date-time-properties.interface';
import { TransactionModel } from '../../../../shared/models/transaction/transaction.model';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { SavePaymentModel } from '../../../../shared/models/payment/savePaymentModel';
import { TranslateService } from '@ngx-translate/core';
import { Observable } from 'rxjs';
import { CustomerService } from '../../../../shared/services/data-definition/customer/customers.service';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { CalculationService } from '../../../../shared/services/calculations.service';
import { ReportMode } from '../../../../shared/models/enums/report-mode.enum';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { PayInvoiceService } from '../../../../shared/services/customer-operation/pay-invoice/pay-invoice.service';
import { GeneralFilter } from '../../../../shared/models/general/generalFilter.model';
import { DownPaymentService } from '../../../../shared/services/customer-operation/down-payment/down-payment.service';
import { PdfCreatorService } from '../../../../shared/services/pdf-creator.service';

@Component({
  selector: 'app-payments-entry',
  templateUrl: './payments-entry.component.html',
  styleUrls: ['./payments-entry.component.css']
})
export class PaymentsEntryComponent implements OnInit {

  //#region [DECLARATIONS]
  ControlTransactionOnDivisionLevel: boolean = false;
  @Input() isDownPayment = false;
  @Input() events: Observable<void>;
  @Input() savePaymentModel: SavePaymentModel;
  @Input() totalRequiredAmount: number = 0;
  @Input() isMultiDivision: boolean = false;
  @Output() onSaveCompleted: EventEmitter<void> = new EventEmitter();
  @Output() onCancelCompleted: EventEmitter<void> = new EventEmitter();
  @ViewChild('selectCreditNoteDialog', { static: true }) selectCreditNoteDialog: NgbModal;
  numberOfDigits = Number(this.sessionData.getConfigurationValue(ConstantConfigurations.NumberOfDigits));
  forceUseAvailableCreditNotes = false;
  allowPayingExtraAtCollection = false;
  applyCreditNotePaymentInBO = false;
  saveBtnSubscription: any;

  selectedCreditNotes = [];
  selectedCustomerId = null;
  selectedOutletId = null;
  selectedCurrencyId = null;
  sharedTabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: false,
    showSearch: false,
    isOnline: false,
    showSearchBtn: false,
    showEditButton: false,
    isMultiSelection: false,
    showNewBtn: false,
    rowOperations: [
      {
        operation: RowOperation.delete,
        title: 'Desc_Delete',
        icon: 'fa-solid fa-trash',
        color: '#f1685e'
      }
    ],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Payment_Type', key: 'paymentTypeName', isSortable: false, width: '20%' },
      { title: 'Desc_Credit_Note_ID', key: 'creditNoteId', isSortable: false, width: '20%' },
      { title: 'Desc_Applied_Amount', key: 'appliedAmount', isSortable: false, width: '20%' },
      { title: 'Desc_Cheque_No', key: 'voucherNumber', isSortable: false, width: '20%' },
      { title: 'Desc_Voucher_No', key: 'voucherOwner', isSortable: false, width: '20%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  dataSourceCreditNote: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  sharedTabelDataForCreditNote: ITableProperties = {
    pageSize: 10,
    showPaginator: true,
    isOnline: false,
    showSearch: false,
    isMultiSelection: true,
    showEditButton: false,
    rowOperations: [],
    multiSelectionOperations: [],
    specificActionWhenCheckAll: true,
    specificActionWhenCheckRow: true,
    columns: [
      { title: 'Desc_Credit_Note_ID', key: 'displayTransactionID', isSortable: false, width: '20%' },
      { title: 'Desc_Outlet', key: 'outletName', isSortable: false, width: '30%' },
      { title: 'Desc_Date', key: 'transactionDate', isSortable: false, width: '18%', isDate: true },
      { title: 'Desc_Value', key: 'remainingAmount', isSortable: false, width: '15%', numberOfDigits: true },
      {
        title: 'Desc_Values_Used', key: 'valuesUsed', isSortable: false, width: '15%', numberOfDigits: true,
        isEditable: true, disableInputBasedOnProperty: true, disableInputPropertyName: 'disableValuesUsed'
      },
    ]
  };

  fillBankData: boolean = false;
  enableHeaderData: boolean = false;
  paymentDetailsForm: FormGroup;
  creditNoteForm: FormGroup;

  totalAppliedAmount: number = 0;
  paymentAppliedAmount: number = 0;

  isTypesLoading: boolean = false
  paymentTypesList: PaymentTypeModel[] = [];

  isBankLoading: boolean = false
  banksList: BankModel[] = [];

  isBranchLoading: boolean = false
  branchList: BranchModel[] = [];

  paymentsList: PaymentModel[] = [];
  selectedTransactions: TransactionModel[] = [];

  payment: PaymentModel;

  voucherNumber: string;
  voucherOwner: string;
  voucherDate: DateTimeModel = new DateTimeModel();

  voucherDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Cheque_Date',
    formControlName: 'voucherDate',
    isCalendarOnly: true,
  };
  error: boolean = false;
  //#endregion

  //#region [INITIALIZED]

  constructor(
    private calculationService: CalculationService,
    private coreSession: CoreSession,
    private paymentService: PaymentService,
    private payInvoiceService: PayInvoiceService,
    private translateService: TranslateService,
    private customerService: CustomerService,
    private sessionData: SessionDataProvider,
    private pdfCreatorService: PdfCreatorService,
    private downPaymentService: DownPaymentService,
    private modalService: NgbModal) {
  }
  ngOnInit() {
    this.ControlTransactionOnDivisionLevel = this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true";
    this.allowPayingExtraAtCollection = this.sessionData.getConfigurationValue(ConstantConfigurations.allowPayingExtraAtCollection).toLowerCase() === "true";
    this.forceUseAvailableCreditNotes = this.sessionData.getConfigurationValue(ConstantConfigurations.ForceUseAvailableCreditNotes).toLowerCase() === "true";
    this.applyCreditNotePaymentInBO = this.sessionData.getConfigurationValue(ConstantConfigurations.ApplyCreditNotePaymentInBO).toLowerCase() === "true";
    if (this.isDownPayment) {
      this.coreSession.SetTitle('Add Down Payment');
    } else {
      this.coreSession.SetTitle('Add Payment');
    }
    if (this.savePaymentModel == undefined) {
      this.savePaymentModel = new SavePaymentModel();
    }
    this.subscribeSaveClick();
    this.fillBanksList();
    this.initForm();
    this.fillPaymentTypesList();
    this.fillCreditNoteData();
  }
  subscribeSaveClick() {
    this.saveBtnSubscription = this.events.subscribe(() => {
      this.onSavePaymentsClick();
    });
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }
  initForm() {
    this.paymentDetailsForm = new FormGroup({
      customerId: new FormControl({ value: null, disabled: this.isDownPayment ? false : true }, this.isDownPayment ? Validators.required : null),
      outletId: new FormControl({ value: null, disabled: this.isDownPayment ? false : true }, this.isDownPayment ? Validators.required : null),
      divisionId: new FormControl({ value: null, disabled: this.isDownPayment ? false : true }, this.ControlTransactionOnDivisionLevel ? Validators.required : null),
      totalRequiredAmount: new FormControl({ value: this.totalRequiredAmount, disabled: true }, !this.isDownPayment ? Validators.required : null),
      totalAppliedAmount: new FormControl({ value: null, disabled: true }, Validators.required),
      paymentAppliedAmount: new FormControl(this.isDownPayment ? null : this.totalRequiredAmount, this.isDownPayment ? [Validators.required, Validators.min(0.000001)] : [Validators.required, Validators.min(0.000001), !this.allowPayingExtraAtCollection ? Validators.max(this.totalRequiredAmount) : Validators.max(999999999999999999)]),
      paymentTypeId: new FormControl(PaymentTypes.Cash.valueOf(), Validators.required),
      bankId: new FormControl(null),
      branchId: new FormControl(null),
      voucherNumber: new FormControl(null),
      voucherDate: new FormControl(null),
      multiDivision: new FormControl({ value: this.isMultiDivision ? this.translateService.instant(ConstantMessages.DescMultiDivisions) : '', disabled: true }),
      voucherOwner: new FormControl(null),
      notes: new FormControl(null)
    });
    if (!this.isDownPayment && this.ControlTransactionOnDivisionLevel) {
      this.divisionIdInput.setValue(this.savePaymentModel.divisionId);
    }

    this.creditNoteForm = new FormGroup({
      requiredAmount: new FormControl({ value: null, disabled: true }),
      appliedAmount: new FormControl({ value: null, disabled: true }),
    })
  }
  //#endregion

  //#region [Form Getters]
  get divisionIdInput() {
    return this.paymentDetailsForm.get('divisionId');
  }
  get customerIdInput() {
    return this.paymentDetailsForm.get('customerId');
  }
  get outletIdInput() {
    return this.paymentDetailsForm.get('outletId');
  }
  get totalRequiredAmountInput() {
    return this.paymentDetailsForm.get('totalRequiredAmount');
  }
  get totalAppliedAmountInput() {
    return this.paymentDetailsForm.get('totalAppliedAmount');
  }
  get paymentAppliedAmountInput() {
    return this.paymentDetailsForm.get('paymentAppliedAmount');
  }
  get paymentTypeIdInput() {
    return this.paymentDetailsForm.get('paymentTypeId');
  }
  get bankIdInput() {
    return this.paymentDetailsForm.get('bankId');
  }
  get branchIdInput() {
    return this.paymentDetailsForm.get('branchId');
  }
  get voucherNumberInput() {
    return this.paymentDetailsForm.get('voucherNumber');
  }
  get voucherDateInput() {
    return this.paymentDetailsForm.get('voucherDate');
  }
  get voucherOwnerInput() {
    return this.paymentDetailsForm.get('voucherOwner');
  }
  get notesInput() {
    return this.paymentDetailsForm.get('notes');
  }
  //#endregion

  //#region [METHODS]
  fillPaymentTypesList() {
    this.isTypesLoading = true;
    this.paymentTypesList = [];
    var outletIdForFirstTransaction = this.savePaymentModel.outletId;
    if (this.savePaymentModel.outletId == -1) {
      outletIdForFirstTransaction = this.savePaymentModel.transactionsList[0].outletId;
    }
    this.paymentService.getPaymentTypes(this.isDownPayment,
      this.savePaymentModel.customerId, outletIdForFirstTransaction).subscribe(result => {
        this.paymentTypesList = result.data;
        this.isTypesLoading = false;

        if (this.paymentTypesList.filter(x => x.paymentTypeId == PaymentTypes.Cash.valueOf()).length <= 0
          && this.paymentTypesList
          && this.paymentTypesList.length > 0
          && this.paymentDetailsForm) {
          this.paymentTypeIdInput.setValue(this.paymentTypesList[0].paymentTypeId);
          this.onChangePaymentType(this.paymentTypesList[0])
        }
      });
    if (!this.isDownPayment) {
      this.customerIdInput.setValue(this.savePaymentModel.customerCodeName);
      this.outletIdInput.setValue(this.savePaymentModel.outletCodeName);
    }
  }
  fillBanksList() {
    this.isBankLoading = true;
    this.banksList = [];
    this.paymentService.getBankWithBranches().subscribe(result => {
      this.banksList = result;
      this.isBankLoading = false;
    });
  }
  fillBranchList(bank: BankModel) {
    this.isBranchLoading = true;
    this.branchList = [];
    if (bank != null && bank.branchList != null && bank.branchList.length > 0) {
      this.branchList = bank.branchList;
    }
    this.isBranchLoading = false;
  }
  fillPaymentsList() {
    this.totalAppliedAmount = 0;
    this.dataSource = {
      data: this.paymentsList,
      totalItems: this.paymentsList.length
    }
    this.paymentsList.forEach(payment => {
      this.totalAppliedAmount = this.totalAppliedAmount + payment.appliedAmount;
    });
    this.clearForm();
  }
  clearForm() {
    this.error = false;
    this.fillBankData = false;
    this.enableHeaderData = false;
    var divisionId = this.divisionIdInput.value;
    this.paymentDetailsForm.reset();
    if (this.isMultiDivision) {
      this.paymentDetailsForm.get('multiDivision').setValue(this.translateService.instant(ConstantMessages.DescMultiDivisions));
    }
    if (this.ControlTransactionOnDivisionLevel) {
      this.divisionIdInput.setValue(divisionId);
    }
    // fill selected customer & outlet/s
    if (!this.isDownPayment) {
      this.customerIdInput.setValue(this.savePaymentModel.customerCodeName);
      this.outletIdInput.setValue(this.savePaymentModel.outletCodeName);
    }
    this.paymentAppliedAmount = 0;
    if (this.isDownPayment) {
      if (this.paymentsList.length <= 0) {
        this.enableHeaderData = true;
      } else {
        this.enableHeaderData = false;
      }
      this.enableDisableCustomerData();
    }
    else {
      if (this.totalRequiredAmount > this.totalAppliedAmount) {
        this.paymentAppliedAmount = this.totalRequiredAmount - this.totalAppliedAmount;
      }
      else if (this.totalRequiredAmount === this.totalAppliedAmount) {
        this.paymentAppliedAmount = 0;
      }
      this.totalRequiredAmount = this.calculationService.getNumberOnDigitFormat(this.totalRequiredAmount);
      this.totalRequiredAmountInput.setValue(this.totalRequiredAmount);
    }
    this.paymentAppliedAmount = this.calculationService.getNumberOnDigitFormat(this.paymentAppliedAmount);
    this.totalAppliedAmount = this.calculationService.getNumberOnDigitFormat(this.totalAppliedAmount);

    this.paymentAppliedAmountInput.setValue(this.paymentAppliedAmount > 0 ? this.paymentAppliedAmount : null);
    this.totalAppliedAmountInput.setValue(this.totalAppliedAmount > 0 ? this.totalAppliedAmount : null);

    if (this.paymentTypesList && this.paymentTypesList.length > 0 && this.paymentDetailsForm) {
      let index = this.paymentTypesList.findIndex(x => x.paymentTypeId == PaymentTypes.Cash.valueOf())
      if (index >= 0) {
        this.paymentTypeIdInput.setValue(this.paymentTypesList[index].paymentTypeId);
        this.onChangePaymentType(this.paymentTypesList[index])
      } else {
        this.paymentTypeIdInput.setValue(this.paymentTypesList[0].paymentTypeId);
        this.onChangePaymentType(this.paymentTypesList[0])
      }

    }
  }
  addPayment(creditNoreId?: string) {

    this.error = false;
    let paidAmount = Number(this.paymentAppliedAmountInput.value);

    // Validations
    if (!this.allowPayingExtraAtCollection) {
      if (!this.isDownPayment && (this.totalRequiredAmount < Number(this.calculationService.formatNumeric((this.totalAppliedAmount + paidAmount), this.numberOfDigits)))) {
        this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.AppliedMoreThanRequired));
        return;
      }
    }
    if (this.paymentTypeIdInput.value == PaymentTypes.PostedDatedCheque.valueOf() && this.voucherDate.date <= new Date()) {
      this.error = true;
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.CheckPDCDate));
      return;
    }
    if (this.paymentTypeIdInput.value == PaymentTypes.CurrentDatedCheque.valueOf() && this.voucherDate.date > new Date()) {
      this.error = true;
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.CheckCDCDate));
      return;
    }

    // Fill Payment object
    this.payment = new PaymentModel();
    this.payment.appliedAmount = paidAmount;
    this.payment.remainingAmount = paidAmount;
    this.payment.paymentTypeId = this.paymentTypeIdInput.value;
    let currenctDate = new Date();
    this.payment.paymentDateModel = new DateTimeModel();
    this.payment.paymentDateModel.year = currenctDate.getFullYear();
    this.payment.paymentDateModel.month = currenctDate.getMonth() + 1;
    this.payment.paymentDateModel.day = currenctDate.getDate();
    this.payment.paymentDateModel.hour = currenctDate.getHours();
    this.payment.paymentDateModel.minute = currenctDate.getMinutes();
    this.payment.paymentDateModel.date = currenctDate;
    this.payment.notes = this.notesInput.value;


    this.payment.bankId = this.bankIdInput.value != null ? this.bankIdInput.value : -1;
    this.payment.branchId = this.branchIdInput.value != null ? this.branchIdInput.value : -1;
    this.payment.voucherNumber = this.voucherNumberInput.value != null ? this.voucherNumberInput.value : '';
    this.payment.voucherDateModel = this.voucherDate;
    this.payment.voucherOwner = this.voucherOwnerInput.value != null ? this.voucherOwnerInput.value : '';

    if (this.paymentTypesList.find(type => type.paymentTypeId === this.payment.paymentTypeId) != null) {
      this.payment.paymentTypeName = this.paymentTypesList.find(type => type.paymentTypeId === this.payment.paymentTypeId).description;
    }

    if (this.payment.paymentTypeId == PaymentTypes.CreditNote.valueOf()) {
      this.payment.paymentTypeName = this.translateService.instant('Desc_Credit_Note');
      this.payment.creditNoteId = creditNoreId;
    }

    let indexOfCashPaymentType = -1;
    if (this.payment.paymentTypeId == PaymentTypes.Cash.valueOf()) {
      indexOfCashPaymentType = this.coreSession.GetIndexForList(this.paymentsList, 'paymentTypeId', this.payment.paymentTypeId)
    }
    if (this.payment.paymentTypeId == PaymentTypes.Cash.valueOf() && indexOfCashPaymentType != -1) {
      // if this payment type is added then add currenct payment value to the existed one
      this.paymentsList[indexOfCashPaymentType].appliedAmount += this.payment.appliedAmount;
      this.paymentsList[indexOfCashPaymentType].remainingAmount = this.paymentsList[indexOfCashPaymentType].appliedAmount;
    }
    else { // if not cash type or cash type but not exist before :
      this.paymentsList.push(this.payment);
    }
    this.fillPaymentsList();
  }
  deletedSelectedPayment(payment: PaymentModel, isDeleted: boolean) {
    let index = -1;
    if (payment.paymentTypeId == PaymentTypes.CreditNote.valueOf()) {
      index = this.paymentsList.findIndex(x => x.creditNoteId === payment.creditNoteId);
    } else {
      index = this.coreSession.GetIndexForList(this.paymentsList, 'paymentTypeId', payment.paymentTypeId);
    }
    if (index != -1 && isDeleted) {
      let rowData = this.savePaymentModel.creditNoteList.find(x => x.transactionId === this.paymentsList[index].creditNoteId);
      if (rowData) {
        rowData.isChecked = false;
        rowData.valuesUsed = 0;
        let rowSelectedIndex = this.selectedCreditNotes.findIndex(x => x.transactionId === this.paymentsList[index].creditNoteId);
        if (rowSelectedIndex >= 0) {
          this.selectedCreditNotes.splice(rowSelectedIndex, 1);
        }
      }
      this.paymentsList.splice(index, 1);
    } else if (index >= 0) {
      this.paymentsList.splice(index, 1);

    }
    this.fillPaymentsList();
  }
  setValidationForBankData() {
    if (this.fillBankData) {

      this.bankIdInput.setValidators(Validators.required);
      this.bankIdInput.updateValueAndValidity();

      this.branchIdInput.setValidators(Validators.required);
      this.branchIdInput.updateValueAndValidity();

      this.voucherNumberInput.setValidators(Validators.required);
      this.voucherNumberInput.updateValueAndValidity();

      if (this.paymentTypeIdInput.value != PaymentTypes.AllowBankTransfer.valueOf()) {
        this.voucherDateInput.setValidators(Validators.required);
        this.voucherDateInput.updateValueAndValidity();
      } else {
        this.voucherDateInput.setValidators(Validators.nullValidator);
        this.voucherDateInput.updateValueAndValidity();
      }

    } else {
      this.bankIdInput.setValidators(Validators.nullValidator);
      this.bankIdInput.updateValueAndValidity();

      this.branchIdInput.setValidators(Validators.nullValidator);
      this.branchIdInput.updateValueAndValidity();

      this.voucherNumberInput.setValidators(Validators.nullValidator);
      this.voucherNumberInput.updateValueAndValidity();

      this.voucherOwnerInput.setValidators(Validators.nullValidator);
      this.voucherOwnerInput.updateValueAndValidity();

      this.voucherDateInput.setValidators(Validators.nullValidator);
      this.voucherDateInput.updateValueAndValidity();
    }
  }
  enableDisableCustomerData() {
    if (this.selectedCustomerId === -1) {
      this.selectedCustomerId = null;
      this.selectedOutletId = null;
    }
    if (this.selectedOutletId === -1) {
      this.selectedOutletId = null;
    }
    if (this.enableHeaderData) {
      this.customerIdInput.enable({ onlySelf: true, emitEvent: false });
      this.outletIdInput.enable({ onlySelf: true, emitEvent: false });
      this.customerIdInput.setValue(this.selectedCustomerId);
      this.outletIdInput.setValue(this.selectedOutletId);

      this.divisionIdInput.enable({ onlySelf: true, emitEvent: false });
    } else {
      this.customerIdInput.disable({ onlySelf: true, emitEvent: false });
      this.outletIdInput.disable({ onlySelf: true, emitEvent: false });
      this.customerIdInput.setValue(this.selectedCustomerId);
      this.outletIdInput.setValue(this.selectedOutletId);

      this.divisionIdInput.disable({ onlySelf: true, emitEvent: false });
    }
  }
  //#endregion

  //#region [EVENTS]
  onChangeCustomer(customer) {
    if (customer) {
      this.selectedCustomerId = customer.customerId;
      this.selectedCurrencyId = customer.currencyId;
    }
    else {
      this.selectedCustomerId = null;
      this.selectedOutletId = null;
      this.selectedCurrencyId = null;
    }
  }
  onChangeOutlet(outlet) {
    if (outlet) {
      this.selectedOutletId = outlet.outletId;
    } else {
      this.selectedOutletId = null;
    }
  }
  onChangePaymentType(paymentType: PaymentTypeModel) {
    this.clearOnChangePaymentType();
    this.error = false;
    if (paymentType) {
      if (paymentType.paymentTypeId == PaymentTypes.CurrentDatedCheque.valueOf() || paymentType.paymentTypeId === PaymentTypes.PostedDatedCheque.valueOf() || paymentType.paymentTypeId === PaymentTypes.CertifiedCheque.valueOf() || paymentType.paymentTypeId === PaymentTypes.AllowBankTransfer.valueOf()) {
        this.fillBankData = true;
      } else {
        this.fillBankData = false;
      }
      this.setValidationForBankData();
    }
  }

  clearOnChangePaymentType() {
    this.voucherOwnerInput.reset();
    this.voucherNumberInput.reset();
    this.bankIdInput.reset();
    this.branchIdInput.reset();
  }

  onChangeBank(bank: BankModel) {
    this.branchIdInput.reset();
    this.fillBranchList(bank);
  }
  onChangeVoucherDate(date) {
    this.voucherDate = date;
    this.error = false;
    if (this.paymentTypeIdInput.value && this.paymentTypeIdInput.value == PaymentTypes.PostedDatedCheque.valueOf() && this.voucherDate.date <= new Date()) {
      this.error = true;
    }
    if (this.paymentTypeIdInput.value && this.paymentTypeIdInput.value == PaymentTypes.CurrentDatedCheque.valueOf() && this.voucherDate.date > new Date()) {
      this.error = true;
    }
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.delete:

        let rowIndex = this.paymentsList.findIndex(x => x.paymentTypeId != PaymentTypes.CreditNote.valueOf());
        if (this.forceUseAvailableCreditNotes && event.object.paymentTypeId == PaymentTypes.CreditNote.valueOf() && rowIndex != -1) {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption),
            this.translateService.instant('Desc_cannot_delete_credit_note_must_be_delete_other_types_so_you_can_delete_credit_note'));

        } else {
          this.deletedSelectedPayment(event.object, true);
        }


        break;
    }
  }
  onPayAllCashClick() {
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.PayAllByCashCheck), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {

          // cleare payments list
          this.paymentsList = [];

          // Fill Payment object
          this.payment = new PaymentModel();
          this.payment.appliedAmount = this.totalRequiredAmount;;
          this.payment.remainingAmount = this.totalRequiredAmount;;
          this.payment.paymentTypeId = PaymentTypes.Cash.valueOf();
          let currenctDate = new Date();
          this.payment.paymentDateModel = new DateTimeModel();
          this.payment.paymentDateModel.year = currenctDate.getFullYear();
          this.payment.paymentDateModel.month = currenctDate.getMonth() + 1;
          this.payment.paymentDateModel.day = currenctDate.getDate();
          this.payment.paymentDateModel.hour = currenctDate.getHours();
          this.payment.paymentDateModel.minute = currenctDate.getMinutes();
          this.payment.paymentDateModel.date = currenctDate;

          if (this.paymentTypesList.find(type => type.paymentTypeId === this.payment.paymentTypeId) != null) {
            this.payment.paymentTypeName = this.paymentTypesList.find(type => type.paymentTypeId === this.payment.paymentTypeId).description;
          }
          this.paymentsList.push(this.payment);
          this.fillPaymentsList();
        }
      });
  }
  onAddPaymentClick() {
    this.coreSession.markFormGroupTouched(this.paymentDetailsForm);
    if (this.paymentDetailsForm.valid) {
      var AllowDownPaymentWhenHavePendingInvoices = false;
      AllowDownPaymentWhenHavePendingInvoices = this.sessionData.getConfigurationValue(ConstantConfigurations.AllowDownPaymentWhenHavePendingInvoices).toLowerCase() === "true";
      if (this.isDownPayment && this.paymentsList.length <= 0 && !AllowDownPaymentWhenHavePendingInvoices) {
        // if customer data is still enabled and this is first payment in the list
        this.checkIfCustomerHasPendingInvoicesBeforeAddingPayment();
      } else {
        let unCheckedRows = this.savePaymentModel.creditNoteList.filter(x => x.isChecked === false);
        if (unCheckedRows.length == 0) {
          let checkedRows = this.savePaymentModel.creditNoteList.filter(x => x.isChecked === true);
          if (checkedRows && checkedRows.length > 0) {
            checkedRows.forEach(x => {
              if (x.valuesUsed < x.remainingAmount) {
                return unCheckedRows.length = 1;
              }
            })
          }
        }
          if (this.forceUseAvailableCreditNotes && this.savePaymentModel.creditNoteList.length > 0 && unCheckedRows.length > 0) {
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption),
              this.translateService.instant('Desc_You_Have_Credits_Notes_Please_Use_Them'));
          } else {
            this.addPayment();
          }
      }
    } else {
      if (!this.allowPayingExtraAtCollection) {
        if (!this.isDownPayment && this.paymentAppliedAmountInput.value > this.totalRequiredAmount) {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.WarningCaption),
            this.translateService.instant(ConstantMessages.AppliedMoreThanRequired));
          return;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption),
            this.translateService.instant(ConstantMessages.MsgFillMandatory));
          return;
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption),
          this.translateService.instant(ConstantMessages.MsgFillMandatory));
        return;
      }
    }
  }
  checkIfCustomerHasPendingInvoicesBeforeAddingPayment() {
    this.coreSession.ModalLoading.Show();
    let filter: GeneralFilter = {
      customerId: this.selectedCustomerId,
      outletId: this.selectedOutletId
    }
    this.customerService.isCustomerHasUnpaidTransactions(filter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        if (response.data) {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgCustomerHasPendingInvoices));
        }
        else {
          this.addPayment();
        }
      }
      else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  onSavePaymentsClick() {
    if (this.paymentsList == null || this.paymentsList.length <= 0) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgNoPayments));
      return;
    }
    if (!this.allowPayingExtraAtCollection) {
      if (!this.isDownPayment && this.totalRequiredAmount < this.totalAppliedAmount) {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.AppliedMoreThanRequired));
        return;
      }
    }
    if (this.isDownPayment) {
      this.saveDownPayments();
    }
    else {
      if (this.totalAppliedAmount < this.totalRequiredAmount) {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgAppliedLessThanRequired), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption))
          .then(
            (result: DialogResult) => {
              if (result === DialogResult.Yes) {
                this.savePayments();
              }
            }
          );
      } else {
        this.savePayments();
      }
    }
  }
  savePayments() {
    this.savePaymentModel.paymentsList = this.paymentsList;
    this.coreSession.ModalLoading.Show();
    this.savePaymentModel.creditNoteList = [];
    this.payInvoiceService.savePayments(this.savePaymentModel).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSaveCompleted.emit();
        let trans = {
          DivisionId: response.data.divisionId ? response.data.divisionId : -1,
          TransactionId: response.data.customerPaymentsIDsStringRef,
          reportMode: ReportMode.Payment,
          isReprintReport: false
        }
        this.pdfCreatorService.ExportReport(trans);
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  saveDownPayments() {
    this.coreSession.ModalLoading.Show();
    this.paymentsList.forEach(x => {
      x.customerId = this.selectedCustomerId;
      x.outletId = this.selectedOutletId;
      x.currencyId = this.selectedCurrencyId;
      x.divisionId = this.ControlTransactionOnDivisionLevel ? this.divisionIdInput.value : -1;
    })
    this.downPaymentService.saveDownPayments(this.paymentsList).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response && response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSaveCompleted.emit();
        let trans = {
          DivisionId: response.data.divisionId ? response.data.divisionId : -1,
          TransactionId: response.data.paymentIDs,
          reportMode: ReportMode.DownPayment,
          isReprintReport: false
        }
        this.pdfCreatorService.ExportReport(trans);


      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }

  openCreditNoteDialog() {

    if (this.selectedCreditNotes && this.selectedCreditNotes.length > 0) {
      this.selectedCreditNotes.filter(x => x.isChecked === false).forEach(result => {
        this.onCheckRowOperation(result);
      })
    }

    return this.modalService.open(this.selectCreditNoteDialog, { centered: true, size: 'xl', scrollable: true }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
          return this.selectCreditNoteDialog;
        } else {
          return -1;
        }
      });
  }
  fillCreditNoteData() {

    if (this.savePaymentModel.creditNoteList != undefined) {
      this.dataSourceCreditNote = { data: this.savePaymentModel.creditNoteList, totalItems: this.savePaymentModel.creditNoteList.length };
      this.creditNoteForm.get('appliedAmount').setValue(0);
    } else {
      this.savePaymentModel.creditNoteList = [];
    }
  }

  onCheckRowOperation(transaction: any) {
    if (transaction.isChecked) {
      transaction.valuesUsed = transaction.remainingAmount;
      transaction.disableValuesUsed = false;
      this.selectedCreditNotes.push({ ...transaction });
    } else {
      let index = this.selectedCreditNotes.findIndex(
        a => a.transactionId == transaction.transactionId
      );
      let rowIndex = this.paymentsList.find(x => x.creditNoteId == transaction.displayTransactionID);
      if (rowIndex) {
        let requiredAmount = this.calculationService.formatNumeric(+Number(this.paymentDetailsForm.get('paymentAppliedAmount').value), this.numberOfDigits);
        this.paymentDetailsForm.get('paymentAppliedAmount').setValue(Number(this.calculationService.formatNumeric(transaction.valuesUsed, this.numberOfDigits)) + requiredAmount);
      }
      transaction.valuesUsed = 0;
      this.selectedCreditNotes.splice(index, 1);
      transaction.disableValuesUsed = true;
    }
    this.fillAppliedAmount();
  }

  onCheckAllOperation(checkAll: boolean) {
    if (checkAll) {
      if (this.savePaymentModel.creditNoteList && this.savePaymentModel.creditNoteList.length > 0) {
        this.savePaymentModel.creditNoteList.forEach(result => {
          result.valuesUsed = + result.remainingAmount;
          result.disableValuesUsed = false;
        })
      }
      this.selectedCreditNotes = this.dataSourceCreditNote.data.slice();
    } else {
      this.selectedCreditNotes = [];
      this.savePaymentModel.creditNoteList.forEach(result => {
        result.valuesUsed = Number(0);
        result.disableValuesUsed = true;
      })
    }
    this.fillAppliedAmount();
  }

  fillAppliedAmount() {
    let totalAmountAppliedAmount = 0;
    if (this.selectedCreditNotes && this.selectedCreditNotes.length > 0) {
      this.selectedCreditNotes.forEach(result => {
        let indexRow = this.savePaymentModel.creditNoteList.findIndex(x => x.transactionId === result.transactionId);
        if (indexRow >= 0 && this.savePaymentModel.creditNoteList[indexRow].isChecked)
          totalAmountAppliedAmount += Number(this.savePaymentModel.creditNoteList[indexRow].valuesUsed);
      })
    }
    this.creditNoteForm.get('appliedAmount').setValue(this.calculationService.formatNumeric(totalAmountAppliedAmount, this.numberOfDigits));
  }

  checkValueByRow(event) {
    if (event.object) {
      let newValue = Number(event.object.valuesUsed);
      let originalValue = Number(event.object.remainingAmount);
      let rowIndex = this.savePaymentModel.creditNoteList.findIndex(x => x.transactionId === event.object.transactionId);
      if (newValue > originalValue) {
        this.coreSession.showWarrning(
          this.translateService.instant(ConstantMessages.WarningCaption),
          this.translateService.instant('Desc_Entered_amount_is_more_than_original'));
      }
      if (rowIndex >= 0) {
        this.savePaymentModel.creditNoteList[rowIndex].valuesUsed = newValue > originalValue ? originalValue : newValue;
        if (this.selectedCreditNotes && this.selectedCreditNotes.length > 0) {
          let indexRowForSelectedNote = this.selectedCreditNotes.findIndex(x => x.transactionId === event.object.transactionId);
          if (indexRowForSelectedNote >= 0)
            this.selectedCreditNotes[indexRowForSelectedNote].valuesUsed = this.savePaymentModel.creditNoteList[rowIndex].valuesUsed;
        }
      }
      this.fillAppliedAmount();
    }
  }

  saveCreditNote(modal: any) {
    let totalRequiredAmount = this.calculationService.formatNumeric(+Number(this.paymentDetailsForm.get('paymentAppliedAmount').value), this.numberOfDigits);
    let creditNoteSelectedToPayment = [];
    let amountValue = 0;
    if (this.selectedCreditNotes && this.selectedCreditNotes.length > 0) {
      this.selectedCreditNotes.forEach(result => {
        if (totalRequiredAmount != 0 && Number(result.valuesUsed) > 0) {
          amountValue = Number(this.calculationService.formatNumeric(result.valuesUsed, this.numberOfDigits));
          if (amountValue > totalRequiredAmount) {
            amountValue -= totalRequiredAmount;
            result.valuesUsed = Number(this.calculationService.formatNumeric(totalRequiredAmount, this.numberOfDigits));
            totalRequiredAmount = 0;
          } else {
            totalRequiredAmount = Number(this.calculationService.formatNumeric((totalRequiredAmount -= amountValue), this.numberOfDigits));
          }
          creditNoteSelectedToPayment.push(result);
        } else {
          let isUseCreditNote = this.paymentsList.findIndex(x => x.creditNoteId === result.transactionId);
          if (isUseCreditNote >= 0) {
            result.isChecked = true;
            creditNoteSelectedToPayment.push(result);

          } else {
            result.isChecked = false;
          }
        }
      })
    }
    this.removeCreditNotesFromPaymentList();
    if (creditNoteSelectedToPayment && creditNoteSelectedToPayment.length > 0) {
      creditNoteSelectedToPayment.forEach(result => {
        this.paymentTypeIdInput.setValue(PaymentTypes.CreditNote.valueOf());
        this.paymentAppliedAmountInput.setValue(this.calculationService.formatNumeric(Number(result.valuesUsed), this.numberOfDigits));
        let rowIndex = this.paymentsList.findIndex(x => x.creditNoteId === result.transactionId);
        if (rowIndex >= 0) {
          this.paymentsList.splice(rowIndex, 1);
        }
        this.addPayment(result.transactionId);
      })
    } else {
      this.removeCreditNotesFromPaymentList();
    }
    modal.close();
  }

  removeCreditNotesFromPaymentList() {
    let paymentList = this.paymentsList.filter(x => x.paymentTypeId == PaymentTypes.CreditNote.valueOf());
    if (paymentList && paymentList.length > 0) {
      paymentList.forEach(result => {
        this.deletedSelectedPayment(result, false);
      })
    }
  }
  //#endregion

}
