import { Component, OnInit, ViewChild } from '@angular/core';
import { PaymentModel } from '../../../../shared/models/payment/payment.model';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { MultiSelectionOperation, RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { GeneralFilter } from '../../../../shared/models/general/generalFilter.model';
import { CoreSession } from '../../../../core/core.session';
import { PaymentService } from '../payment.service';
import { HttpErrorResponse } from '@angular/common/http';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { MainFilterComponent } from '../../../../shared/components/filters/main-filter/main-filter.component';
import { ISharedItemsProperties } from '../../../../shared/components/items/shared-items-properties.interface';
import { FormProcessMode } from '../../../../shared/models/enums/form-process-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { TranslateService } from '@ngx-translate/core';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ReportMode } from '../../../../shared/models/enums/report-mode.enum';
import { PdfCreatorService } from '../../../../shared/services/pdf-creator.service';
import { IDateTimePickerProperties } from '../../../../shared/models/dateTimeModel/date-time-properties.interface';

@Component({
  selector: 'app-payments-list',
  templateUrl: './payments-list.component.html',
  styleUrls: ['./payments-list.component.css']
})
export class PaymentsListComponent implements OnInit {

  selectedPaymentsToVoid: PaymentModel[] = [];

  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    showEditButton: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Customer', key: 'customerNameCode', isSortable: true, width: this.isDivisionLevel() ? '10%' : '13%' },
      { title: 'Desc_Outlet', key: 'outletNameCode', isSortable: true, width: this.isDivisionLevel() ? '10%' : '13%' },
      { title: 'Desc_Employee', key: 'employeeNameCode', isSortable: true, width: '10%' },
      { title: 'Desc_Payment_ID', key: 'customerPaymentId', isSortable: true, width: '12%' },
      { title: 'Desc_Date', key: 'paymentDate', isSortable: false, width: '12%', isDate: true },
      { title: 'Desc_Amount', key: 'appliedAmount', isSortable: false, width: this.isDivisionLevel() ? '10%' : '12%', numberOfDigits: true },
      { title: 'Desc_Type', key: 'paymentTypeName', isSortable: false, width: '9%' },
      { title: 'Desc_Bank', key: 'bankName', isSortable: false, width: '9%' },
      { title: 'Desc_Branch', key: 'branchName', isSortable: false, width: '10%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  fromToDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Date',
    formControlName: 'fromToDate',
    isCalendarOnly: true,
    isRange: true,
  };
  generalFilter: GeneralFilter = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
  };

  itemFormProperties: ISharedItemsProperties = {
    addedItemList: [],
    generalFilter: {
      getPrices: true,
    },
  };
  constructor(
    private coreSession: CoreSession,
    private paymentService: PaymentService,
    private translateService: TranslateService,
    private sessionData: SessionDataProvider,
    private pdfCreatorService: PdfCreatorService) {
  }

  ngOnInit() {
    this.coreSession.SetTitle('Payments');
    if (this.isDivisionLevel()) {
      this.TabelData.columns.push(
        { title: 'Desc_Division', key: 'divisionName', isSortable: false, width: '8%' },
      );
    }
    this.addActions();
    this.getPaymentsList();
  }
  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.paymentsListURL, MenuActions.Void)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Void',
          icon: 'fa-solid fa-trash',
          color: '#f1685e',
          showWhenKeyValueTrue: true,
          showHideOptionPerLine: true,
          controlKeyName: "showVoidBtn"
        });

    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.paymentsListURL, MenuActions.Bounce)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Bounce,
          title: 'Desc_Bounce',
          icon: 'fa fa-refresh',
          color: '#ff0000',
          showWhenKeyValueTrue: true,
          showHideOptionPerLine: true,
          controlKeyName: 'isCheque'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.paymentsListURL, MenuActions.Print)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Print,
          title: 'Desc_Print',
          icon: 'fa-solid fa-print',
          color: '#12344d'
        });
    }

  }
  isDivisionLevel() {
    return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true")
  }
  getPaymentsList() {
    this.coreSession.ModalLoading.Show();
    this.paymentService.getPaymentsList(this.generalFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0 && response.data.data) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data
        };
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    }
    );
  }
  onTableFilterChanged(query: any) {
    this.generalFilter.customListFilter = query;
    this.getPaymentsList();
  }
  onRowOperation(event: any) {
    this.selectedPaymentsToVoid = [];
    this.selectedPaymentsToVoid.push(event.object);
    switch (event.operation) {
      case RowOperation.delete: {
        this.voidPayments(this.selectedPaymentsToVoid);
        break;
      }
      case RowOperation.Bounce: {
        this.bounceCheque(this.selectedPaymentsToVoid);
        break;
      }
      case RowOperation.Print: {
        this.exportCollectionReport(event);
        break;
      }
    }
  }

  exportCollectionReport(event: any) {
    let trans = {
      DivisionId: event.object.divisionId ? event.object.divisionId : -1,
      TransactionId: event.object.customerPaymentId,
      reportMode: ReportMode.Payment,
      isReprintReport: true
    }
    this.pdfCreatorService.ExportReport(trans);
  }
  onMultiSelectionOperation(event: any) {
    switch (event.operation) {
      case MultiSelectionOperation.delete: {
        this.selectedPaymentsToVoid = [];
        this.selectedPaymentsToVoid = event.object;
        this.voidPayments(this.selectedPaymentsToVoid);
        break;
      }
    }
  }
  onFindButtonClicked(result) {
    this.generalFilter.customListFilter.page = 0;
    this.generalFilter.employeeId = result.employeeId;
    this.generalFilter.transactionId = result.transactionId;
    this.generalFilter.customerId = result.customerId;
    this.generalFilter.outletId = result.outletId;
    this.generalFilter.fromToDate = result.transactionDate;
    this.generalFilter.divisionId = result.divisionId;
    this.generalFilter.includeCashPayment = result.includeCashPayment;

    this.getPaymentsList();
  }
  voidPayments(selectedPayments: PaymentModel[]) {
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.paymentService.voidPayments(selectedPayments).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              // voided Successfully
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.MsgVoidedSuccessfully), response.message);
              this.getPaymentsList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
          }
          );
        }
      }
    );
  }
  bounceCheque(selectedPayments: PaymentModel[]) {
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgBounceConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.paymentService.bounceCheques(selectedPayments).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              // Bounced Successfully
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly), response.message);
              this.getPaymentsList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
          }
          );
        }
      }
    );
  }

}
