import { Component, OnInit, ViewChild, Input } from '@angular/core';
import { CoreSession } from '../../../../core/core.session';
import { TransactionModel } from '../../../../shared/models/transaction/transaction.model';
import { Router } from '@angular/router';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { FormProcessMode } from '../../../../shared/models/enums/form-process-mode.enum';
import { GeneralFilter } from '../../../../shared/models/general/generalFilter.model';
import { OperationSummaryComponent } from '../../../../shared/components/operation-summary/operation-summary.component';
import { IOperationSummary } from '../../../../shared/models/operation-summary/operation-summary-interface';
import { TranslateService } from '@ngx-translate/core';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { ValidationService } from '../../../../shared/services/validation.service';
import { DocumentTypes } from '../../../../shared/models/enums/document-types.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ReportMode } from '../../../../shared/models/enums/report-mode.enum';
import { TransactionService } from '../../transactions/transaction.service';
import { PaymentService } from '../../payments/payment.service';
import { getDate } from 'ngx-bootstrap/chronos/utils/date-getters';
import { YEAR } from 'ngx-bootstrap/chronos/units/constants';
import { PaymentModel } from '../../../../shared/models/payment/payment.model';
import { SendEmailService } from '../../../admin-settings/email/send-email.service';
import { CustomerTransactionsService } from '../../../../shared/services/customer-operation/customer-transactions/customer-transactions.service';

@Component({
    selector: 'app-transaction-list',
    templateUrl: './transaction-list.component.html',
    styleUrls: ['./transaction-list.component.css']
})
export class TransactionListComponent implements OnInit {
    TabelData: ITableProperties = {
        pageSize: this.coreSession.pageSize,
        showPaginator: true,
        isOnline: true,
        showSearch: true,
        showSearchBtn: true,
        showNewBtn: false,
        isMultiSelection: false,
        openFilterDirectly: true,
        rowOperations: [],
        multiSelectionOperations: [],
        columns: [
            { title: 'Desc_Document_ID', key: 'transactionId', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
            { title: 'Desc_Customer', key: 'customerCodeName', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
            { title: 'Desc_Outlet', key: 'outletCodeName', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
            { title: 'Desc_Employee', key: 'employeeName', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
            { title: 'Desc_TransactionType', key: 'transPaymentType', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
            { title: 'Desc_Date', key: 'transactionDate', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%', isExpiryDate: true },
            { title: 'Desc_Amount', key: 'netTotal', isSortable: false, width: this.isDivisionLevel() ? '9%' : '10%', numberOfDigits: true },
            { title: 'Desc_Status', key: 'voidedString', isSortable: true, width: this.isDivisionLevel() ? '7%' : '8%' }
        ]
    };
    dataSource: SharedTableResult = {
        totalItems: 0,
        data: []
    };
    transactionId = '';
    operationSummaryData: IOperationSummary = {
        operationData: {},
        formProcessMode: FormProcessMode.Sales,
        formTitle: 'Desc_Sales_Invoice'
    };

    selectedTransaction: TransactionModel[] = [];
    currentTime = new Date();
    query: GeneralFilter = {
        customListFilter: {
            searchFilter: '',
            page: 0,
            pageSize: this.coreSession.pageSize
        },
        fromToDate:
            [{
                year: new Date().getFullYear(),
                month: new Date().getMonth() + 1,
                day: new Date().getDate(),
                minute: new Date().getMinutes(),
                hour: new Date().getHours(),
                second: new Date().getSeconds(),
                date: new Date(),
            },
            {
                year: new Date().getFullYear(),
                month: new Date().getMonth() + 1,
                day: new Date().getDate(),
                minute: new Date().getMinutes(),
                hour: new Date().getHours(),
                second: new Date().getSeconds(),
                date: new Date(),
            }
            ],
    };
    selectedPaymentsToVoid: PaymentModel[] = [];
    showTransactionEntrySlider = false;
    showViewSummarySlider = false;
    autoCollectBOInvoices = false;
    constructor(
        private sessionData: SessionDataProvider,
        private validationService: ValidationService,
        private coreSession: CoreSession,
        private router: Router,
        private translateService: TranslateService,
        private customerTransactionsService: CustomerTransactionsService
    ) {
    }

    ngOnInit() {
        this.coreSession.SetTitle('Invoices');
        if (this.isDivisionLevel())
            this.TabelData.columns.push({ title: "Desc_Division", key: "divisionName", isSortable: true, width: "10%" });
        this.populateCustomerTransactionsList();
        this.addViewAction();
        this.addDeleteAction();
        this.autoCollectBOInvoices = this.sessionData.getConfigurationValue(ConstantConfigurations.AutoCollectBOInvoices).toLowerCase() === "true";
    }
    addDeleteAction() {
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.transactionListURL, MenuActions.Void)) {
            this.TabelData.rowOperations.push(
                {
                    operation: RowOperation.delete,
                    title: 'Desc_Void',
                    icon: 'fa-solid fa-trash',
                    color: '#f1685e',
                    showWhenKeyValueTrue: true,
                    showHideOptionPerLine: true,
                    controlKeyName: "showVoidBtn"
                });

        }
    }
    addViewAction() {
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.transactionListURL, MenuActions.View)) {
            this.TabelData.rowOperations.push(
                {
                    operation: RowOperation.View,
                    title: 'Desc_View',
                    icon: 'fa-regular fa-eye',
                    color: '#12344d',
                    showWhenKeyValueTrue: true,
                    showHideOptionPerLine: true,
                    controlKeyName: "showViewBtn"
                });
        }
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.transactionListURL, MenuActions.Print)) {
            this.TabelData.rowOperations.push(
                {
                    operation: RowOperation.Print,
                    title: 'Desc_Print',
                    icon: 'fa-solid fa-print',
                    color: '#12344d'
                });
        }
    }
    isDivisionLevel() {
        return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true")
    }
    populateCustomerTransactionsList() {
        this.coreSession.ModalLoading.Show();
        this.customerTransactionsService.getTransactionsList(this.query).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
                this.dataSource = <SharedTableResult>response.data;
            } else {
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
        }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
        );
    }
    onTableFilterChanged(query: any) {
        this.query.customListFilter = query;
        this.populateCustomerTransactionsList();
    }
    onRowOperation(event: any) {
        this.selectedPaymentsToVoid = [];
        this.selectedPaymentsToVoid.push(event.object);
        switch (event.operation) {
            case RowOperation.View:
                if (this.coreSession.checkActivitiesAvailability(ConstantURLs.transactionListURL, MenuActions.View)) {
                    if (!event.object.isPayment) {
                        if (event.object.transactionTypeId != 3) {
                            this.viewTransaction(event.object.signature);
                        }
                        else {
                            this.viewExchangeTransaction(event.object.signature);
                        }
                    }
                    else {
                        this.viewPayment(event.object.signature);
                    }
                }
                break;
            case RowOperation.delete:
                if (this.coreSession.checkActivitiesAvailability(ConstantURLs.transactionListURL, MenuActions.Void)) {
                    if (!event.object.isPayment) {
                        if (event.object.transactionTypeId != 3) {
                            this.voidTransaction(event.object);
                        } else {
                            this.voidExchange(event.object);
                        }
                    }
                    else {
                        this.voidPayment(this.selectedPaymentsToVoid);
                    }
                }
                break;
            case RowOperation.Print:
                if (!event.object.isPayment) {
                    let trans = {
                        divisionId: event.object.divisionId ? event.object.divisionId : -1,
                        transactionId: event.object.transactionId,
                        transactionTypeId: event.object.transactionTypeId,
                        customerId: event.object.customerId,
                        outletId: event.object.outletId
                    }
                    this.customerTransactionsService.ShowReport(trans, true);
                    break;
                }
                else {
                    let payments = {
                        divisionId: event.object.divisionId ? event.object.divisionId : -1,
                        transactionId: event.object.transactionId,
                        transactionTypeId: event.object.paymentTypeId,
                        customerId: event.object.customerId,
                        outletId: event.object.outletId,
                        reportTypeID: 6
                    }
                    this.customerTransactionsService.ShowReport(payments, true);
                    break;
                }
        }
    }
    voidTransaction(transaction: TransactionModel) {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo).then(
            (res: DialogResult) => {
                if (res === DialogResult.Yes) {
                    this.coreSession.ModalLoading.Show();
                    transaction.isCustomerTransactions = true;
                    this.customerTransactionsService.voidTransaction(transaction).subscribe(response => {
                        this.coreSession.ModalLoading.Hide();
                        if (response.status != null && response.status >= 0) {
                            // voided Successfully
                            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
                            if (response.data) {
                                transaction.voided = true;
                            }
                            this.populateCustomerTransactionsList();
                        } else {
                            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                        }
                    }, (error: HttpErrorResponse) => {
                        this.coreSession.ModalLoading.Hide();
                        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
                    }
                    );
                }
            }
        );
    }
    voidExchange(transaction: TransactionModel) {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo).then(
            (res: DialogResult) => {
                if (res === DialogResult.Yes) {
                    this.coreSession.ModalLoading.Show();
                    transaction.isCustomerTransactions = true;
                    this.customerTransactionsService.voidExchange(transaction).subscribe(response => {
                        this.coreSession.ModalLoading.Hide();
                        if (response.status != null && response.status >= 0) {
                            // voided Successfully
                            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
                            if (response.data) {
                                transaction.voided = true;
                            }
                            this.populateCustomerTransactionsList();
                        } else {
                            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                        }
                    }, (error: HttpErrorResponse) => {
                        this.coreSession.ModalLoading.Hide();
                        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
                    }
                    );
                }
            }
        );
    }
    voidPayment(selectedPayments: PaymentModel[]) {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
            (res: DialogResult) => {
                if (res === DialogResult.Yes) {
                    this.coreSession.ModalLoading.Show();
                    this.customerTransactionsService.voidPayments(selectedPayments).subscribe(response => {
                        this.coreSession.ModalLoading.Hide();
                        if (response.status != null && response.status >= 0) {
                            // voided Successfully
                            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.MsgVoidedSuccessfully), response.message);
                            this.populateCustomerTransactionsList();
                        } else {
                            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                        }
                    }, (error: HttpErrorResponse) => {
                        this.coreSession.ModalLoading.Hide();
                        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
                    }
                    );
                }
            }
        );
    }
    viewTransaction(signature: string) {
        this.coreSession.ModalLoading.Show();
        this.customerTransactionsService.getTransactionData(signature).subscribe(
            response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                    this.selectedTransaction = response.data;
                    this.operationSummaryData = {
                        operationData: this.selectedTransaction,
                        formProcessMode: FormProcessMode.Sales,
                        formTitle: 'Desc_Invoice'
                    };
                    this.showViewSummarySlider = true;
                } else {
                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                }
            },
            (error: HttpErrorResponse) => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
        );
    }
    viewExchangeTransaction(signature: string) {
        this.coreSession.ModalLoading.Show();
        fromMode: FormProcessMode.ExchangeSale;
        this.customerTransactionsService.getTransactionData(signature).subscribe(
            response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                    this.selectedTransaction = response.data;
                    this.operationSummaryData = {
                        operationData: this.selectedTransaction,
                        formProcessMode: FormProcessMode.ExchangeSale,
                        formTitle: 'Desc_Invoice'
                    };
                    this.showViewSummarySlider = true;
                } else {
                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                }
            },
            (error: HttpErrorResponse) => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
        );
    }
    viewPayment(signature: string) {
        this.coreSession.ModalLoading.Show();
        this.customerTransactionsService.getPaymentTransactions(signature).subscribe(
            response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                    this.selectedTransaction = response.data;
                    this.operationSummaryData = {
                        operationData: this.selectedTransaction,
                        formProcessMode: FormProcessMode.Payment,
                        formTitle: 'Desc_Payment'
                    };
                    this.showViewSummarySlider = true;
                } else {
                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                }
            },
            (error: HttpErrorResponse) => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
        );
    }
    closeEntrySlider() {
        this.transactionId = '';
        this.showTransactionEntrySlider = false;
    }
    onFindButtonClicked(result) {
        this.query.customListFilter.page = 0;
        this.query.employeeId = result.employeeId;
        this.query.transactionId = result.transactionId;
        this.query.customerId = result.customerId;
        this.query.outletId = result.outletId;
        this.query.fromToDate = result.transactionDate;
        this.query.reportTypeId = result.reportTypeId;
        this.query.routeId = result.routeId;
        this.query.includeVoided = result.includeVoided;
        if (this.isDivisionLevel())
            this.query.divisionId = result.divisionId;
        else
            this.query.divisionId = -1
        this.populateCustomerTransactionsList();
    }
    closeViewSummarySlider() {
        this.selectedTransaction = null;
        this.operationSummaryData = null;
        this.showViewSummarySlider = false;
    }
}




