import { Component, OnInit, ViewChild, Input } from '@angular/core';
import { CoreSession } from '../../../../core/core.session';
import { TransactionService } from '../transaction.service';
import { TransactionModel } from '../../../../shared/models/transaction/transaction.model';
import { Router } from '@angular/router';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { FormProcessMode } from '../../../../shared/models/enums/form-process-mode.enum';
import { GeneralFilter } from '../../../../shared/models/general/generalFilter.model';
import { OperationSummaryComponent } from '../../../../shared/components/operation-summary/operation-summary.component';
import { IOperationSummary } from '../../../../shared/models/operation-summary/operation-summary-interface';
import { TranslateService } from '@ngx-translate/core';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { ValidationService } from '../../../../shared/services/validation.service';
import { DocumentTypes } from '../../../../shared/models/enums/document-types.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ReportMode } from '../../../../shared/models/enums/report-mode.enum';
import { InvoiceService } from '../../../../shared/services/customer-operation/invoice/invoice.service';
import { PdfCreatorService } from '../../../../shared/services/pdf-creator.service';
import { TestInvoiceService } from '../../../../shared/services/customer-operation/invoice/test-invoice.service';

@Component({
  selector: 'app-customer-transaction-list',
  templateUrl: './customer-transaction-list.component.html',
  styleUrls: ['./customer-transaction-list.component.css']
})
export class CustomerTransactionListComponent implements OnInit {
  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: true,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Customer', key: 'customerCodeName', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
      { title: 'Desc_Outlet', key: 'outletCodeName', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%' },
      { title: 'Desc_Employee', key: 'employeeName', isSortable: false, width: this.isDivisionLevel() ? '11%' : '14%' },
      { title: 'Desc_Transaction_ID', key: 'transactionId', isSortable: true, width: this.isDivisionLevel() ? '15%' : '16%' },
      { title: 'Desc_Date', key: 'transactionDate', isSortable: true, width: this.isDivisionLevel() ? '13%' : '14%', isExpiryDate: true },
      { title: 'Desc_Net_Total', key: 'netTotal', isSortable: false, width: this.isDivisionLevel() ? '9%' : '10%', numberOfDigits: true },
      { title: 'Desc_Remaining', key: 'remainingAmount', isSortable: false, width: this.isDivisionLevel() ? '9%' : '10%', numberOfDigits: true },
      { title: 'Desc_Status', key: 'voidedString', isSortable: false, width: this.isDivisionLevel() ? '7%' : '8%' }
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  transactionId = '';
  operationSummaryData: IOperationSummary = {
    operationData: {},
    formProcessMode: FormProcessMode.Sales,
    formTitle: 'Desc_Sales_Invoice'
  };

  selectedTransaction: TransactionModel;

  query: GeneralFilter = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
  };

  showTransactionEntrySlider = false;
  showViewSummarySlider = false;
  autoCollectBOInvoices = false;
  @Input() formProcessMode: FormProcessMode = FormProcessMode.Sales;
  constantURL: string = '';

  constructor(
    private sessionData: SessionDataProvider,
    private validationService: ValidationService,
    private invoiceService: InvoiceService,
    private testInvoiceService: TestInvoiceService,
    private coreSession: CoreSession,
    private router: Router,
    private translateService: TranslateService,
    private pdfCreatorService: PdfCreatorService) {
  }

  ngOnInit() {
    if (this.formProcessMode == FormProcessMode.Sales) {
      this.coreSession.SetTitle('Invoices');
      this.constantURL = ConstantURLs.invoiceListURL;
    }
    else {
      this.coreSession.SetTitle('TestInvoices');
      this.constantURL = ConstantURLs.testInvoicesListURL;
    }
    if (this.isDivisionLevel())
      this.TabelData.columns.push({ title: "Desc_Division", key: "divisionName", isSortable: true, width: "10%" });
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.Add);
    this.addDeleteAction();
    this.addViewAction();
    if (this.formProcessMode == FormProcessMode.Sales) {
      this.populateCustomerTransactionsList();
    }
    else {
      this.populateCustomerTestTransactionsList();
    }
    this.autoCollectBOInvoices = this.sessionData.getConfigurationValue(ConstantConfigurations.AutoCollectBOInvoices).toLowerCase() === "true";
  }
  addDeleteAction() {
    if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.Void)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Void',
          icon: 'fa-solid fa-trash',
          color: '#f1685e'
        });
    }
  }
  addViewAction() {
    if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.View)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.View,
          title: 'Desc_View',
          icon: 'fa-regular fa-eye',
          color: '#12344d'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.Print)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Print,
          title: 'Desc_Print',
          icon: 'fa-solid fa-print',
          color: '#12344d'
        });
    }
  }
  isDivisionLevel() {
    return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true")
  }
  populateCustomerTransactionsList() {
    this.coreSession.ModalLoading.Show();
    this.invoiceService.getCustomerTransactions(this.query).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = <SharedTableResult>response.data;
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    }
    );
  }
  populateCustomerTestTransactionsList() {
    this.coreSession.ModalLoading.Show();
    this.testInvoiceService.getTestCustomerTransactions(this.query).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = <SharedTableResult>response.data;
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    }
    );
  }
  onTableFilterChanged(query: any) {
    this.query.customListFilter = query;
    if (this.formProcessMode == FormProcessMode.Sales) {
      this.populateCustomerTransactionsList();
    }
    else {
      this.populateCustomerTestTransactionsList();
    }
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.View:
        if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.View)) {
          if (this.formProcessMode == FormProcessMode.Sales)
            this.viewTransaction(event.object.signature);
          else
            this.viewTestTransaction(event.object.signature);
        }
        break;
      case RowOperation.delete:
        if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.View)) {
          if (this.formProcessMode == FormProcessMode.Sales)
            this.voidTrasnaction(event.object);
          else
            this.voidTestTrasnaction(event.object);
        }
        break;
      case RowOperation.Print:
        if (this.coreSession.checkActivitiesAvailability(this.constantURL, MenuActions.Print)) {
          let trans = {
            divisionId: event.object.divisionId ? event.object.divisionId : -1,
            transactionId: event.object.transactionId,
            customerId: event.object.customerId,
            outletId: event.object.outletId,
            reportMode: this.formProcessMode == FormProcessMode.Sales ? ReportMode.Transaction : ReportMode.TestTransaction,
            isReprintReport: true
          }
          this.pdfCreatorService.ExportReport(trans);
        }

        break;
    }
  }
  voidTrasnaction(transaction: TransactionModel) {
    if (transaction.voided) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectedInvoiceVoided));
      return;
    }
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.invoiceService.voidTransaction(transaction).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              // voided Successfully
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
              if (response.data) {
                transaction.voided = true;
              }
              this.populateCustomerTransactionsList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
          }
          );
        }
      }
    );
  }
  voidTestTrasnaction(transaction: TransactionModel) {
    if (transaction.voided) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectedInvoiceVoided));
      return;
    }
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgVoidConfirmation), DialogMode.YesNo).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.testInvoiceService.voidTestTransaction(transaction).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              // voided Successfully
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
              if (response.data) {
                transaction.voided = true;
              }
              this.populateCustomerTestTransactionsList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          }, (error: HttpErrorResponse) => {
            this.coreSession.ModalLoading.Hide();
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
          }
          );
        }
      }
    );
  }
  viewTransaction(signature: string) {
    this.coreSession.ModalLoading.Show();
    this.invoiceService.getTransactionData(signature).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.selectedTransaction = response.data;
          this.operationSummaryData = {
            operationData: this.selectedTransaction,
            formProcessMode: FormProcessMode.Sales,
            formTitle: 'Desc_Sales_Invoice'
          };
          this.showViewSummarySlider = true;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  viewTestTransaction(signature: string) {
    this.coreSession.ModalLoading.Show();
    this.testInvoiceService.getTestTransactionData(signature).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.selectedTransaction = response.data;
          this.operationSummaryData = {
            operationData: this.selectedTransaction,
            formProcessMode: FormProcessMode.Sales,
            formTitle: 'Desc_Sales_Invoice'
          };
          this.showViewSummarySlider = true;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  getMaxDocumentSequence() {
    let documentSequenceFilter = {
      divisionId: -1,
      employeeId: this.coreSession.CurrentOperator.employeeId,
      documentType: DocumentTypes.Invoice
    };
    this.coreSession.ModalLoading.Show();
    // check for document sequence and account
    this.validationService.checkAccountAndDocumentSequence(documentSequenceFilter).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.transactionId = response.data;
          if (!this.autoCollectBOInvoices) {
            this.showTransactionEntrySlider = true;
          } else {
            this.checkDocumentSequenceToPayment();
          }
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  checkDocumentSequenceToPayment() {

    let documentSequenceFilter = {
      divisionId: -1,
      employeeId: this.coreSession.CurrentOperator.employeeId,
      DocumentType: DocumentTypes.Collection
    };
    this.coreSession.ModalLoading.Show();
    this.validationService.getMaxDocumentSequence(documentSequenceFilter).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.showTransactionEntrySlider = true;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  openNewInvoiceScreen() {
    if (this.formProcessMode == FormProcessMode.Sales && !this.applyDivisionsOnDocumentSequence()) {
      this.getMaxDocumentSequence();
    } else {
      this.showTransactionEntrySlider = true;
    }
  }
  closeEntrySlider() {
    this.transactionId = '';
    this.showTransactionEntrySlider = false;
  }
  onSaveComplete() {
    this.closeEntrySlider();
    if (this.formProcessMode == FormProcessMode.Sales) {
      this.populateCustomerTransactionsList();
    }
    else {
      this.populateCustomerTestTransactionsList();
    }
  }
  onFindButtonClicked(result) {
    this.query.customListFilter.page = 0;
    this.query.employeeId = result.employeeId;
    this.query.transactionId = result.transactionId;
    this.query.customerId = result.customerId;
    this.query.outletId = result.outletId;
    this.query.fromToDate = result.transactionDate;
    this.query.includeVoided = result.includeVoided;
    this.query.newCustomerOnly = result.newCustomerOnly;
    if (this.isDivisionLevel())
      this.query.divisionId = result.divisionId;
    else
      this.query.divisionId = -1
    if (this.formProcessMode == FormProcessMode.Sales) {
      this.populateCustomerTransactionsList();
    }
    else {
      this.populateCustomerTestTransactionsList();
    }
  }
  closeViewSummarySlider() {
    this.selectedTransaction = null;
    this.operationSummaryData = null;
    this.showViewSummarySlider = false;
  }
  applyDivisionsOnDocumentSequence() {
    return (this.isDivisionLevel() && this.sessionData.getConfigurationValue(ConstantConfigurations.ApplyDivisionsOnDocumentSequence).toLowerCase() === "true")
  }
}




