import { FormControl, FormGroup } from '@angular/forms';
import { Component, OnInit } from '@angular/core';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { CoreSession } from '../../../../core/core.session';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ResponseModel } from '../../../../shared/models/api-models/api-models';
import { CustomerService } from '../../../../shared/services/data-definition/customer/customers.service';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { DateTimeModel } from '../../../../shared/models/dateTimeModel/date-time.model';
import { IDateTimePickerProperties } from '../../../../shared/models/dateTimeModel/date-time-properties.interface';
import { TranslateService } from '@ngx-translate/core';
import { Subject } from 'rxjs';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { CustomerICashService } from '../../../../shared/services/data-definition/customer/customer-icash.service';

@Component({
  selector: 'app-customer-icash',
  templateUrl: './customer-icash-list.component.html',
  styleUrls: ['./customer-icash-list.component.css']
})

export class CustomerICashListComponent implements OnInit {

  //#region [Declaration]

  isEditMode = false;
  showSlider = false;
  fromToDate: DateTimeModel[];
  icashFilterForm: FormGroup;
  saveSubject: Subject<void> = new Subject<void>();
  menuURL = "";
  DateProperties: IDateTimePickerProperties = {
    label: 'Desc_Date',
    formControlName: 'Date',
    isCalendarOnly: true,
    isRange: true
  };
  icashListObject: any;
  dataSource: SharedTableResult = { totalItems: 0, data: [] };
  TableData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    openFilterDirectly: false,
    rowOperations: [
    ],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_ICash_Code', key: 'iCashCode', isSortable: true, width: '30%' },
      { title: 'Desc_Vendor', key: 'organizationCodeName', isSortable: true, width: '30%' },
      { title: 'Desc_Start_Date', key: 'startDate', isSortable: true, isExpiryDate: true, width: '20%', },
      { title: 'Desc_End_Date', key: 'endDate', isSortable: true, isExpiryDate: true, width: '20%' },
    ]
  };

  icashFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    fromToDate: [],
    isInactive: false
  };

  constructor(
    private coreSession: CoreSession,
    private customerService: CustomerService,
    private customerICashService: CustomerICashService,
    private translateService: TranslateService
  ) { }


  //#endregion

  //#region [Events]

  ngOnInit() {
    this.coreSession.SetTitle('ICash List');
    this.initFilterForm();
    this.populateICashCustomer();
    this.menuURL = ConstantURLs.CustomerICashURL;
    this.TableData.showNewBtn = this.coreSession.checkActivitiesAvailability(this.menuURL, MenuActions.Add);
    this.addDeleteAction();
    this.addEditAction();
  }

  onTransactionDateSelected(date) {
    this.icashFilter.fromToDate = date;
  }

  onTableFilterChanged(query: any) {
    this.icashFilter.customListFilter = query;
    this.populateICashCustomer();
  }

  onReset() {
    this.icashFilterForm.reset();
    this.icashFilterForm.get('isInactive').setValue(false);
    this.icashFilter.fromToDate = [];

  }

  onCloseEntrySlider() {
    this.showSlider = false;
    this.isEditMode = false;
    this.coreSession.SetTitle('ICash List');
  }

  onNewBtnClicked() {
    this.showSlider = true;
    this.isEditMode = false;
    this.coreSession.SetTitle('Add ICash');
  }

  onSaveClick() {
    this.saveSubject.next();
  }

  onRowOperation(event: any) {

    switch (event.operation) {
      case RowOperation.edit:
        if (this.coreSession.checkActivitiesAvailability(this.menuURL, MenuActions.Edit)) {
          this.getICashData(event.object);
        }
        break;
      case RowOperation.delete:
        this.onDeleteClicked(event.object);
        break;
    }

  }

  onDeleteClicked(event: any) {
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (result: DialogResult) => {
        if (result === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          this.customerICashService
            .DeleteICashData(event.iCashListID)//here
            .subscribe(
              response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                  this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
                  this.populateICashCustomer();
                } else {
                  this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
                }
              },
              error => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError("", ConstantMessages.ErrorHappened);
              }
            );
        }
      });
  }

  //#endregion

  //#region [Functions]

  initFilterForm() {

    this.icashFilterForm = new FormGroup({
      Date: new FormControl([null, null]),
      isInactive: new FormControl(false)
    })
  }

  populateICashCustomer() {

    this.icashFilter.isInactive = this.icashFilterForm.get('isInactive').value != null ? this.icashFilterForm.get('isInactive').value : false;
    this.coreSession.ModalLoading.Show();
    this.customerICashService.getICashCustomers(this.icashFilter).subscribe(
      (response: ResponseModel) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.dataSource = {
            totalItems: response.data.totalItems,
            data: response.data.data
          };
        } else {
          this.dataSource = {
            totalItems: 0,
            data: []
          };
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });

  }

  applySerachFilterOnFind() {
    this.populateICashCustomer();

  }

  afterSaveItem() {

    this.onCloseEntrySlider();
    this.populateICashCustomer();
  }

  getICashData(object) {
    this.coreSession.ModalLoading.Show();
    this.customerICashService.getICashDataInEdit(object.iCashListID).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.icashListObject = response.data;
          this.isEditMode = true;
          this.showSlider = true;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      error => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );

  }

  addDeleteAction() {
    if (this.coreSession.checkActivitiesAvailability(this.menuURL, MenuActions.Delete)) {
      this.TableData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Delete',
          icon: 'fa-solid fa-trash',
          color: '#f1685e'
        });
    }


  }
  addEditAction() {
    if (this.coreSession.checkActivitiesAvailability(this.menuURL, MenuActions.Edit)) {
      this.TableData.rowOperations.push({
        operation: RowOperation.edit,
        title: "Desc_Edit",
        icon: "fa-regular fa-pen-to-square",
        color: "#12344d"
      });

    }

  }

  //#endregion


}
