import { Component, OnInit, EventEmitter, Input, Output } from '@angular/core';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { CustomerService } from '../../../../shared/services/data-definition/customer/customers.service';

import { HttpErrorResponse } from '@angular/common/http';

import { Observable } from 'rxjs';
import { CustomerGroupModel } from '../../../../shared/models/customer/customer.group.model';
import { CoreSession } from '../../../../core/core.session';
import { ResponseModel } from '../../../../shared/models/api-models/api-models';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { TranslateService } from '@ngx-translate/core';
import { MasterDataService } from '../../master-data.service';


@Component({
  selector: 'app-customer-group-entry',
  templateUrl: './customer-group-entry.component.html',
  styleUrls: ['./customer-group-entry.component.css']
})
export class CustomerGroupEntryComponent implements OnInit {
  groupForm:FormGroup;

  groupObj:CustomerGroupModel;
  observable:any;
  isOrganizationLoading : boolean = false;
  organizationsList : any[] = [] ;

  @Input() isEditMode=false;
  @Input() isViewMode=false;
  @Output() onAddingNewCustomerGroup=new EventEmitter();
  @Input() saveEvent: Observable<void>;
  @Input() showChannelSubChannel: boolean = false;

   saveBtnSubscription: any;

  channelsDropdownSetting = {
    singleSelection: true,
    idField: 'channelSubChannelIds',
    textField: 'channelSubChannelDescriptions',
    selectAllText: '',
    unSelectAllText: '',
    position: 'top',
    allowSearchFilter: true,
    searchPlaceholderText: '',
    itemsShowLimit: 2,
    
  };
  allChannelsList: any[] = [];
  constructor(private customerService:CustomerService,private coreSession:CoreSession,
    private translateService: TranslateService,
    private masterDataService:MasterDataService) { }

  ngOnInit() {
    if(this.showChannelSubChannel) {
      this.fillChannelsSubChannelsList();
      this.channelsDropdownSetting.selectAllText = this.translateService.instant(ConstantMessages.SelectAllCaption);
      this.channelsDropdownSetting.unSelectAllText = this.translateService.instant(ConstantMessages.UnSelectAllCaption);
      this.channelsDropdownSetting.searchPlaceholderText = this.translateService.instant(ConstantMessages.SearchCaption);
    }
    this.subscribeSaveClick();
    this.groupForm = new FormGroup({
      groupName: new FormControl(
        { value: "", disabled: this.isViewMode },
         Validators.required
      ),
      groupCode: new FormControl(
        { value: "", disabled: this.isViewMode },
        Validators.required
      ),
      organizationId: new FormControl(
        { value: null, disabled: this.isViewMode },
         Validators.required
      ),
      channel: new FormControl({value:null , disabled: this.isViewMode },
        this.showChannelSubChannel? Validators.required : Validators.nullValidator)
    });
    this.fillOrganizationList();
    if(this.isEditMode){
      this.fillUIInEditForm();
    }
  }
  fillOrganizationList() {
    this.organizationsList = [];
    this.isOrganizationLoading = true;
    this.masterDataService.getOrganizationList().subscribe(result => {
      this.organizationsList = result;

      this.isOrganizationLoading = false;
    });
  }
  fillUIInEditForm(){
    if(this.groupObj){
      this.groupForm.value.groupCode=this.groupObj.groupCode;
      this.groupForm.value.groupName=this.groupObj.groupName;
      this.groupForm.value.organizationId=this.groupObj.organizationId;
    }
  }
  fillGroupObj(){
    if(!this.groupObj)this.groupObj=new CustomerGroupModel();
    this.groupObj.groupCode=this.groupForm.value.groupCode;
    this.groupObj.groupName=this.groupForm.value.groupName;
    this.groupObj.organizationId=this.groupForm.value.organizationId;

    if(this.showChannelSubChannel) {
      var channelSubChannelObj = this.groupForm.get('channel').value[0];
      this.groupObj.channelId = parseInt(channelSubChannelObj.channelSubChannelIds.split('-')[0]);
      this.groupObj.subChannelId = parseInt(channelSubChannelObj.channelSubChannelIds.split('-')[1]);
    } else {
      this.groupObj.channelId = -1;
      this.groupObj.subChannelId = -1;
    }
  }
  IsValidToSaveGroupCustomer(){
    if(this.groupForm.invalid){
      this.markFormGroupTouched();
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    this.fillGroupObj();
    let codeFilter = {
      tableName :'CustomerGroup',
      columnName :'GroupCode',
      insertedValue : this.groupObj.groupCode,
      excludedString : ' And GroupId != ' + '-1',
    }
    this.masterDataService.checkIfUniqueValue(codeFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {

        if (!response.data) {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgUniqueCode));
        }
        else {
          this.SaveGroupCustomerData();
        }
      }
      else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(response.message));
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    }
    );


  }
  onSaveClicked(){
    this.IsValidToSaveGroupCustomer();
  }
  SaveGroupCustomerData() {

  this.observable=new Observable<ResponseModel>();

  if(this.isEditMode)
    this.observable=this.customerService.UpdateCustomerGroup(this.groupObj);
  else
    this.observable=this.customerService.InsertNewCustomerGroup(this.groupObj);

  this.observable.subscribe(
    response => {
    this.coreSession.ModalLoading.Hide();
    if (response.status != null && response.status >= 0) {
      //swal('Success!', response.message, 'success');
      this.groupObj=response.data;
      this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption),response.message);
      this.onAddingNewCustomerGroup.emit(this.groupObj);
    } else {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption),response.message);
    }
    },
      (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption),this.translateService.instant(ConstantMessages.ErrorHappened));
     }
    );
  }
  private markFormGroupTouched() {
    (<FormGroup>(
      this.groupForm
    )).controls.groupName.markAsTouched();
    (<FormGroup>(
      this.groupForm
    )).controls.groupCode.markAsTouched();
    (<FormGroup>(
      this.groupForm
    )).controls.organizationId.markAsTouched();
    if(this.showChannelSubChannel) {
      (<FormGroup>(this.groupForm)).controls.channel.markAsTouched();
    }
  }
  fillChannelsSubChannelsList() {
    this.customerService.GetAllCustomerChannels(null).subscribe(result => {
      this.allChannelsList = result;
    });
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }

  subscribeSaveClick() {
    this.saveBtnSubscription = this.saveEvent.subscribe(() => {
      this.onSaveClicked();
    });
  }
}
