import { Component, OnInit, Input, Output, EventEmitter } from '@angular/core';
import { Observable } from 'rxjs';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { ITextInput } from '../../../../../shared/models/shared-table/custom-list.interface';
import { CoreSession } from '../../../../../core/core.session';
import { TranslateService } from '@ngx-translate/core';
import { ItemService } from '../../../../../shared/services/data-definition/product/item.service';
import { ConstantMessages } from '../../../../../shared/models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { SessionDataProvider } from '../../../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../../../shared/models/constants/constant-configuration';
import { BrandService } from '../../../../../shared/services/data-definition/product/brand.service';
import { MasterDataService } from '../../../master-data.service';

@Component({
  selector: 'app-brand-entry',
  templateUrl: './brand-entry.component.html',
  styleUrls: ['./brand-entry.component.css']
})
export class BrandEntryComponent implements OnInit {
  @Input() events: Observable<void>;
  @Input() brandObject: any;
  @Output() onSaveCompleted = new EventEmitter();

  saveBtnSubscription: any;
  isEditMode = false;
  form: FormGroup;
  otherLanguages: any[] = [];
  allowCustomerAppLogin = false;
  brandNameInput: ITextInput = {
    formControlName: "name",
    placeHolder: "Desc_Brand_Name",
    label: "Desc_Brand_Name",
    isRequierd: true
  };
  organizationsList: any[] =[];

  brandId = -1;
  organizationDropdownSettings = {
    singleSelection: false,
    idField: "organizationId",
    textField: "organizationCodeName",
    selectAllText: this.translateService.instant(ConstantMessages.SelectAllCaption),
    unSelectAllText: this.translateService.instant(ConstantMessages.UnSelectAllCaption),
    searchPlaceholderText: this.translateService.instant(ConstantMessages.SearchCaption),
    allowSearchFilter: true,
    itemsShowLimit: 2,
  };
  selectedImage: File = null;
  hasImage = false;
  isImageChanged = false;
  imagePath: any = "";
  constructor(private coreSession: CoreSession,
    private sessionProvider: SessionDataProvider,
    private translateService: TranslateService,
    private masterDataService: MasterDataService,
    private brandService: BrandService) { }

  ngOnInit() {
    this.fillOrganizations();
    this.allowCustomerAppLogin = this.sessionProvider.getConfigurationValue(ConstantConfigurations.AllowCustomerAppLogin).toLowerCase() === "true";
    this.checkIfEditMode();
    this.subscribeSaveClick();
    this.initForm();
  }

  checkIfEditMode() {
    if (this.brandObject) {
      this.isEditMode = true;
    } else {
      this.brandObject = {};
    }
  }

  initForm() {
    let brandName = "";
    let brandCode = "";
    let showInCustomerApp = false;
    if (this.isEditMode) {
      brandName = this.brandObject.brandName;
      brandCode = this.brandObject.brandCode;
      this.organizationsList = this.brandObject.organizationsList;
      showInCustomerApp = this.brandObject.showInCustomerApp;
      this.brandId = this.brandObject.brandId;
    }
    this.form = new FormGroup({
      name: new FormControl(brandName, Validators.required),
      showInCustomerApp: new FormControl(showInCustomerApp),
      code: new FormControl(brandCode, Validators.required),
      organizationsList: new FormControl(this.organizationsList, Validators.required),

    });
    if (this.isEditMode && this.brandObject.hasImage) {
      this.hasImage = true;
      this.imagePath = document.getElementsByTagName("base")[0].href + "Resources/brands-img/" + this.brandObject.brandId + ".png" + "?random=" + new Date().getTime();
    } else {
      this.setDefaultImage();
    }
  }
  fillOrganizations() {
    this.masterDataService.getOrganizationList(false).subscribe((result) => {
      debugger
      this.organizationsList = result;
      this.form.get("organizationsList").setValue(this.organizationsList.filter((org) =>
        org.organizationId == this.coreSession.CurrentOperator.organizationId));
    });
    // debugger
    // this.organizationsList.push(this.coreSession.CurrentOperator.organizationId);
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }

  subscribeSaveClick() {
    this.saveBtnSubscription = this.events.subscribe(() => {
      this.onSaveClicked();
    });
  }

  onSaveClicked() {
    if (this.form.invalid) {
      this.coreSession.markFormGroupTouched(this.form as FormGroup);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    } else {

      this.fillBrandObject();
      let codeFilter = {
        tableName: "Brand",
        columnName: "BrandCode",
        insertedValue: this.form.get("code").value,
        excludedString: " And BrandId != " + this.brandId,
      };
      this.coreSession.ModalLoading.Show();
      this.masterDataService.checkIfUniqueValue(codeFilter).subscribe(
        (response) => {
          this.coreSession.ModalLoading.Hide();
          if (response.status != null && response.status >= 0) {
            if (!response.data)
              this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgUniqueCode));
            else
              if (this.isEditMode)
                this.editBrandData();
              else
                this.saveBrandData();
          } else
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        },
        (error: HttpErrorResponse) => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
      );
    }
  }

  saveBrandData() {
    var formData: FormData = new FormData();
    if (this.brandObject.hasImage && this.brandObject.isImageChanged) {
      formData.append("image", this.selectedImage, this.selectedImage.name);
    }
    formData.append('brand', JSON.stringify(this.brandObject));
    this.coreSession.ModalLoading.Show();
    this.brandService.saveBrand(formData).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
          this.onSaveCompleted.emit();

        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  editBrandData() {
    var formData: FormData = new FormData();
    if (this.brandObject.hasImage && this.brandObject.isImageChanged) {
      formData.append("image", this.selectedImage, this.selectedImage.name);
    }
    formData.append('brand', JSON.stringify(this.brandObject));
    this.coreSession.ModalLoading.Show();
    this.brandService.editBrand(formData).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
          this.onSaveCompleted.emit();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  fillBrandObject() {
    this.brandObject.hasImage = this.hasImage;
    this.brandObject.isImageChanged = this.isImageChanged;
    this.brandObject.brandName = this.form.value.name;
    this.brandObject.brandCode = this.form.value.code;
    this.brandObject.organizationsList = this.form.value.organizationsList;
    this.brandObject.showInCustomerApp = this.form.value.showInCustomerApp;
    this.brandObject.descriptionLanguages = this.otherLanguages;
    this.brandObject.descriptionLanguages.push({
      languageId: this.coreSession.selectedLanguageId,
      description: this.form.value.name
    });
  }

  onOtherLanguagesTextSubmitted(event) {
    this.otherLanguages = event;
  }

  markAsTouched() {
    this.form.get("name").markAsTouched();
    this.form.get("code").markAsTouched();
  }
  onUploadFinished(event: any) {
    if (event.target.files && event.target.files[0]) {
      const reader = new FileReader();
      this.selectedImage = event.target.files[0];
      this.hasImage = true;
      this.isImageChanged = true;
      reader.onload = (event: ProgressEvent) => {
        this.imagePath = (<FileReader>event.target).result;
      };
      reader.readAsDataURL(event.target.files[0]);
    }
  }
  onClearImage() {
    this.isImageChanged = true;
    this.hasImage = false;
    this.selectedImage = null;
    this.setDefaultImage();
  }
  setDefaultImage() {
    this.imagePath = document.getElementsByTagName("base")[0].href + "Resources/noProduct.jpg";
  }
}

