import { Component, OnInit, Input, Output, EventEmitter } from '@angular/core';
import { Observable, Subscription } from 'rxjs';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { ITextInput } from '../../../../../shared/models/shared-table/custom-list.interface';
import { CoreSession } from '../../../../../core/core.session';
import { TranslateService } from '@ngx-translate/core';
import { ItemService } from '../../../../../shared/services/data-definition/product/item.service';
import { ConstantMessages } from '../../../../../shared/models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { SessionDataProvider } from '../../../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../../../shared/models/constants/constant-configuration';
import { ItemCategoryService } from '../../../../../shared/services/data-definition/product/item-category.service';
import { Router } from "@angular/router";
import { ShepherdService } from 'angular-shepherd';
import { TourService } from '../../../../../shared/shepherd-config/shepherd-config';
import { ComponentVisitService } from '../../../../../shared/shepherd-config/trackTour.service';
@Component({
  selector: 'app-category-entry',
  templateUrl: './category-entry.component.html',
  styleUrls: ['./category-entry.component.css']
})
export class CategoryEntryComponent implements OnInit {
  @Input() events: Observable<void>;
  @Input() categoryObject: any;
  @Output() onSaveCompleted = new EventEmitter();

  saveBtnSubscription: any;
  isEditMode = false;
  form: FormGroup;
  otherLanguages: any[] = [];
  allowCustomerAppLogin = false;
  categoryNameInput: ITextInput = {
    formControlName: "name",
    placeHolder: "Desc_Category_Name",
    label: "Desc_Category_Name",
    isRequierd : true
  };

  categoryId = -1;

  selectedImage: File = null;
  hasImage = false;
  isImageChanged = false;
  imagePath: any = "";

  constructor(
    private coreSession: CoreSession,
    private sessionProvider: SessionDataProvider,
    private translateService: TranslateService,
    private itemService: ItemService,
    private itemCategoryService: ItemCategoryService,
    private router : Router,
    private shepherdService : ShepherdService,
    private tourService: TourService,
    private componentVisitService: ComponentVisitService
  ) { }
  private tourStartTimeout: any;

  startTour(forceTour?:boolean){

    if(!forceTour)
    if (this.tourService.isTourCompletedForComponent('category-entry')) {
      // Tour already completed for this component
      return;
    }

    this.tourStartTimeout= setTimeout((data=>{

      if(this.tourService.isFirstTour()||forceTour){
        const elementSelectors = ['#image','#def-category','#def-division','#Customer-App']
        const titles = [];
      const texts = [];
      for (let index = 0; index < elementSelectors.length; index++) {
        const { title, text } = this.getTitleAndText(index);
        titles.push(title);
        texts.push(text);
      }
      // Target each column by its unique ID
      const steps = this.tourService.generateSteps(elementSelectors,titles,texts);
      this.shepherdService.defaultStepOptions = this.tourService.defaultStepOptions;
      this.shepherdService.modal = true;
      this.shepherdService.addSteps(steps);
      this.shepherdService.start();
    }
    }),1000)
  }
  ngAfterViewInit() {
    this.startTour()

  }
  getTitleAndText(index: number): { title: string, text: string } {
    let title = '';
    let text = '';

    switch (index) {
      case 0:
          title = "Desc_Add_Image";
          text = 'Desc_content_Add_Image';
          break;
      case 1:
          title = "Desc_Category";
          text = "Desc_content_Category";
          break;
      case 2:
          title = "Desc_Division";
          text = "Desc_Division";
          break;
      case 3:
          title = "Desc_Customer_App_Caption";
          text = "Desc__content_Customer_App";
          break;
      default:
          title = `Step ${index + 1}`;
          text = `Content in default at bottom.`;
  }

    return { title, text };
  }
  ngOnInit() {
    this.tourSubscription = this.tourService.startTourEntry$.subscribe(() => {
        this.startTour(true)
      }
    )
    this.allowCustomerAppLogin = this.sessionProvider.getConfigurationValue(ConstantConfigurations.AllowCustomerAppLogin).toLowerCase() === "true";
    this.checkIfEditMode();
    this.subscribeSaveClick();
    this.initForm();
  }

  checkIfEditMode() {
    if (this.categoryObject) {
      this.isEditMode = true;
    } else {
      this.categoryObject = {};
    }
  }

  initForm() {
    let categoryCode = "";
    let categoryName = "";
    let divisionId: number;
    let active = true;
    let showInCustomerApp = true;
    if (this.isEditMode) {
      categoryCode = this.categoryObject.itemCategoryCode;
      categoryName = this.categoryObject.itemCategoryName;
      active = this.categoryObject.active;
      divisionId = this.categoryObject.divisionId;
      showInCustomerApp = this.categoryObject.showInCustomerApp;
      this.categoryId = this.categoryObject.itemCategoryId;
    }
    this.form = new FormGroup({
      code: new FormControl(categoryCode, Validators.required),
      name: new FormControl(categoryName, Validators.required),
      divisionId: new FormControl(divisionId, Validators.required),
      active: new FormControl(active, Validators.required),
      showInCustomerApp: new FormControl(showInCustomerApp),
    });
    if (this.isEditMode && this.categoryObject.hasImage) {
      this.hasImage = true;
      this.imagePath = document.getElementsByTagName("base")[0].href + "Resources/categories-img/" + this.categoryObject.itemCategoryId + ".png" + "?random=" + new Date().getTime();
    } else {
      this.setDefaultImage();
    }
  }

  private tourSubscription: Subscription;
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
    if (this.tourStartTimeout) {
      clearTimeout(this.tourStartTimeout);
    }
    this.componentVisitService.markComponentVisited('category-entry');

    if (this.tourSubscription) {
      this.tourSubscription.unsubscribe();
    }
  }

  subscribeSaveClick() {
    this.saveBtnSubscription = this.events.subscribe(() => {
      this.onSaveClicked();
    });
  }

  onSaveClicked() {
    this.markAsTouched();
    if (this.form.invalid) {
      return;
    }

    this.checkCategoryCodeIfExist();
  }

  checkCategoryCodeIfExist() {
    this.itemCategoryService.checkIfCategoryCodeExist(this.form.value.code, this.categoryId).subscribe(
      response => {
        if (response.status != null && response.status >= 0) {
          if (response.data.isExist) {
            this.coreSession.showWarrning(
              this.translateService.instant("Desc_Warning"),
              this.translateService.instant("Desc_Code_Exist")
            );
            return;
          } else {
            this.fillCategoryObject();

            if (this.isEditMode) {
              this.editCategoryData();
            } else {
              this.saveCategoryData();
            }
          }
        }
      },
      error => {
        this.coreSession.showError("", ConstantMessages.ErrorHappened);
      }
    );
  }

  saveCategoryData() {
    var formData: FormData = new FormData();
    if (this.categoryObject.hasImage && this.categoryObject.isImageChanged) {
      formData.append("image", this.selectedImage, this.selectedImage.name);
    }
    formData.append('category', JSON.stringify(this.categoryObject));
    this.itemCategoryService.saveCategory(formData).subscribe(
      response => {
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
          this.onSaveCompleted.emit();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  editCategoryData() {
    var formData: FormData = new FormData();
    if (this.categoryObject.hasImage && this.categoryObject.isImageChanged) {
      formData.append("image", this.selectedImage, this.selectedImage.name);
    }
    formData.append('category', JSON.stringify(this.categoryObject));
    this.itemCategoryService.editCategory(formData).subscribe(
      response => {
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
          this.onSaveCompleted.emit();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  fillCategoryObject() {
    this.categoryObject.hasImage = this.hasImage;
    this.categoryObject.isImageChanged = this.isImageChanged;
    this.categoryObject.itemCategoryCode = this.form.value.code;
    this.categoryObject.itemCategoryName = this.form.value.name;
    this.categoryObject.active = this.form.value.active;
    this.categoryObject.divisionId = this.form.value.divisionId;
    this.categoryObject.showInCustomerApp = this.form.value.showInCustomerApp;
    this.categoryObject.descriptionLanguages = this.otherLanguages;
    this.categoryObject.descriptionLanguages.push({
      languageId: this.coreSession.selectedLanguageId,
      description: this.form.value.name
    });
  }

  onOtherLanguagesTextSubmitted(event) {
    this.otherLanguages = event;
  }

  markAsTouched() {
    this.form.get("code").markAsTouched();
    this.form.get("name").markAsTouched();
    this.form.get("divisionId").markAsTouched();
  }
  onUploadFinished(event: any) {
    if (event.target.files && event.target.files[0]) {
      const reader = new FileReader();
      this.selectedImage = event.target.files[0];
      this.hasImage = true;
      this.isImageChanged = true;
      reader.onload = (event: ProgressEvent) => {
        this.imagePath = (<FileReader>event.target).result;
      };
      reader.readAsDataURL(event.target.files[0]);
    }
  }
  onClearImage() {
    this.isImageChanged = true;
    this.hasImage = false;
    this.selectedImage = null;
    this.setDefaultImage();
  }
  setDefaultImage() {
    this.imagePath = document.getElementsByTagName("base")[0].href + "Resources/NoProduct-Echo.jpg";
  }
}
