import { FlatTreeControl, NestedTreeControl } from '@angular/cdk/tree';
import { HttpErrorResponse } from '@angular/common/http';
import { Component, OnInit, ViewChild } from '@angular/core';
import { FormControl, FormGroup, Validators } from '@angular/forms';
import { MatTreeNestedDataSource } from '@angular/material/tree';
import { TranslateService } from '@ngx-translate/core';
import { BehaviorSubject, Observable, of as observableOf, Subject } from 'rxjs';
import { CoreSession } from '../../../../../core/core.session';
import { SessionDataProvider } from '../../../../../core/session-data-provider.service';
import { NavBarService } from '../../../../../core/sidebar/sidebar.service';
import { SelectCustomersComponent } from '../../../../../shared/components/customers/select-customers/select-customers.component';
import { SharedDialogComponent } from '../../../../../shared/components/shared-dialog/shared-dialog.component';
import { ResponseModel } from '../../../../../shared/models/api-models/api-models';
import { ConstantConfigurations } from '../../../../../shared/models/constants/constant-configuration';
import { ConstantMessages } from '../../../../../shared/models/constants/constant-message';
import { ConstantURLs } from '../../../../../shared/models/constants/constant-URL';
import { DialogMode } from '../../../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../../../shared/models/enums/dialog-result.enum';
import { GeographicalLocationHierarchyLevel } from '../../../../../shared/models/enums/geographical-location-hierarchy-Level';
import { MenuActions } from '../../../../../shared/models/enums/menu-actions-enum';
import { NavigationData } from '../../../../../shared/models/navigation-data/navigation-data.model';
import { ITextInput } from '../../../../../shared/models/shared-table/custom-list.interface';
import { TreeNode } from '../../../../../shared/models/tree-model/tree-model';
import { GeolocationService } from '../../../../../shared/services/geolocation.service';
import { CustomerService } from '../../../../../shared/services/data-definition/customer/customers.service';

@Component({
  selector: 'app-geographical-location-hierarchy',
  templateUrl: './geographical-location-hierarchy.component.html',
  styleUrls: ['./geographical-location-hierarchy.component.css']
})
export class GeographicalLocationHierarchyComponent implements OnInit {



  //#region  [Declarations]

  selectedGroup = -1;
  dataSource: any;
  nestedDataSource: MatTreeNestedDataSource<TreeNode>;
  nestedDataSourceTemp: MatTreeNestedDataSource<TreeNode>;
  nestedTreeControl: NestedTreeControl<TreeNode>;
  childrenChange = new BehaviorSubject<TreeNode[]>([]);
  dataChange: BehaviorSubject<TreeNode[]> = new BehaviorSubject<TreeNode[]>([]);
  @ViewChild('templateEntry', { static: true }) templateEntry: SharedDialogComponent;
  saveSubject: Subject<void> = new Subject<void>();
  title = '';
  isEditMode = false;
  entryFormType: GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.Country;
  nodeData: any;
  treeControl = new FlatTreeControl<TreeNode>(node => node.level, node => node.isExpanded);
  locationForm: FormGroup;
  otherLanguages: any[] = [];
  nameFieldProp: ITextInput = {
    formControlName: "name",
    placeHolder: '',
    label: '',
    isRequierd: true
  };
  searchText = '';
  expandNode = false;
  selectedNode: any;
  addAction = false;
  deleteAction = false;
  editAction = false;
  highLevelLabel: string = "";
  isLoading = false;
  isCountryMode = false;

  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private customerService: CustomerService,
    private sessionData: SessionDataProvider,
    private locationService: GeolocationService,
    private navService: NavBarService
  ) {
    this.nestedTreeControl = new NestedTreeControl<TreeNode>(this._getChildren);
    this.nestedDataSource = new MatTreeNestedDataSource();
  }

  private _getChildren = (node: TreeNode) => { return observableOf(node.children); };

  hasNestedChild = (_: number, nodeData: TreeNode) => { return (nodeData.isParent); };

  isExpandable = (node: TreeNode) => node.isExpanded;

  //#endregion

  //#region [Events]

  ngOnInit() {


    this.childrenChange.subscribe(items => {
      this.nestedDataSource.data = null;
      this.nestedDataSource.data = items;
    });

    this.setNavigation();
    this.initializeForm();
    this.addActions();
    this.setLabels();
    this.populateGeoLocationHierarchyData();
  }

  onSaveClicked() {
    this.saveSubject.next();
  }

  onCloseClicked() {
    this.templateEntry.Close();
  }

  onFilterChanged() {

    this.coreSession.ModalLoading.Show();
    this.filterTree();
    if (this.searchText) {
      this.nestedTreeControl.expandAll();
    } else {
      this.nestedTreeControl.collapseAll();
    }
  }

  onOtherLanguagesTextSubmitted(event) {
    this.otherLanguages = event;
  }

  onDialogResult(event) {
    if (event && event.saveCLicked) {
      this.onSaveClicked();
    }
  }

  onAddingSuccessfully(nodeData: any) {

    this.onCloseClicked();
    if (nodeData.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.Country
      && this.isEditMode && !nodeData.isStateLevel) {
      this.populateGeoLocationHierarchyData()
    } else {
      this.loadMore(nodeData, false, true);
    }
  }

  addNode(event: any) {

    this.isEditMode = false;
    if (event) {
      switch (event.geographicalLocationHierarchyLevel) {
        case GeographicalLocationHierarchyLevel.Country:
          this.entryFormType = GeographicalLocationHierarchyLevel.State;
          break;
        case GeographicalLocationHierarchyLevel.State:
          this.entryFormType = GeographicalLocationHierarchyLevel.City;
          break;
        case GeographicalLocationHierarchyLevel.City:
          this.entryFormType = GeographicalLocationHierarchyLevel.Area;
          break;
        case GeographicalLocationHierarchyLevel.Area:
          this.entryFormType = GeographicalLocationHierarchyLevel.Street;
          break;
        case GeographicalLocationHierarchyLevel.Street:
          this.entryFormType = GeographicalLocationHierarchyLevel.Street;
          break;
      }

      this.nodeData = event;
      this.prepareEntryData(this.entryFormType);
      this.showEntryModal();
      this.expandNode = true;
    }
  }

  editNode(event: any) {
    this.isEditMode = true;
    if (event) {
      this.nodeData = event;
      this.prepareEntryData(event.geographicalLocationHierarchyLevel)
      this.showEntryModal();
      this.expandNode = true;
    }
  }

  deleteNode(event: any) {

    if (event) {
      var locationHierarchyData =
      {
        countryId: event.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.Country ? event.nodeId : -1,
        stateId: event.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.State ? event.nodeId : -1,
        cityId: event.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.City ? event.nodeId : -1,
        areaId: event.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.Area ? event.nodeId : -1,
        streetId: event.geographicalLocationHierarchyLevel == GeographicalLocationHierarchyLevel.Street ? event.nodeId : -1,
        code: '',
        geographicalLocationHierarchyLevel: event.geographicalLocationHierarchyLevel,
        isEditMode: false,
        parentId: event.parentId,
        descriptions: []
      }
      this.nodeData = event;
      this.expandNode = true;
      this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
        (result: DialogResult) => {
          if (result === DialogResult.Yes) {
            this.coreSession.ModalLoading.Show();
            this.locationService.deletLocationHierarchyData(locationHierarchyData).subscribe(
              result => {
                this.coreSession.ModalLoading.Hide();
                if (result.status != null && result.status >= 0) {
                  this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
                  if (result.data) {
                    this.loadMore(result.data, false, true);
                  } else {
                    this.populateGeoLocationHierarchyData();
                  }
                } else {
                  this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), result.message);
                }
              },
              error => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
              }
            );
          }
        });
    }

  }



  addHighLevelLocation() {
    if (this.locationForm.valid) {
      var locationHierarchyData =
      {
        countryId: -1,
        stateId: -1,
        cityId: -1,
        areaId: -1,
        streetId: -1,
        code: this.locationForm.controls.code.value,
        geographicalLocationHierarchyLevel: this.geographicalLocationHierarchyLevel(),
        isEditMode: false,
        descriptions: this.fillDescriptions()
      }
      this.coreSession.ModalLoading.Show();
      this.locationService.saveUpdateLocationHierarchyData(locationHierarchyData).subscribe(
        result => {
          this.coreSession.ModalLoading.Hide();
          if (result.status != null && result.status >= 0) {
            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
            this.locationForm.reset();
            this.populateGeoLocationHierarchyData();
          } else {
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), result.message);
          }
        },
        error => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
      );
    } else {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption),
        this.translateService.instant(ConstantMessages.MsgFillMandatory));
      this.locationForm.markAllAsTouched();
    }
  }

  //#endregion

  //#region [Functions]

  populateGeoLocationHierarchyData() {
    var locationHierarchyFilter =
    {
      locationHierarchyLevel: 0,
      parentId: null,
      searchFilter: ''
    }
    this.coreSession.ModalLoading.Show();
    this.locationService.getGeoLocationHierarchyData(locationHierarchyFilter).subscribe(
      (response: ResponseModel) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.nestedDataSource.data = this.nestedTreeControl.dataNodes = this.nestedDataSourceTemp = response.data;
          if (this.expandNode) {
            this.expandSpecificNode(this.nodeData);
          }
        } else {
          this.nestedDataSource.data = [];
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }

  prepareEntryData(entryFormType: GeographicalLocationHierarchyLevel) {
    switch (entryFormType) {
      case GeographicalLocationHierarchyLevel.Country:
        this.title = this.isEditMode ? 'Desc_Edit_Geographical_Location_Five' : 'Desc_Add_Geographical_Location_Five';

        this.entryFormType = GeographicalLocationHierarchyLevel.Country;
        break;
      case GeographicalLocationHierarchyLevel.State:
        this.title = this.isEditMode ? 'Desc_Edit_Geographical_Location_Four' : 'Desc_Add_Geographical_Location_Four';
        this.entryFormType = GeographicalLocationHierarchyLevel.State;
        break;
      case GeographicalLocationHierarchyLevel.City:
        this.title = this.isEditMode ? 'Desc_Edit_Geographical_Location_Three' : 'Desc_Add_Geographical_Location_Three';
        this.entryFormType = GeographicalLocationHierarchyLevel.City;
        break;
      case GeographicalLocationHierarchyLevel.Area:
        this.title = this.isEditMode ? 'Desc_Edit_Geographical_Location_Two' : 'Desc_Add_Geographical_Location_Two';
        this.entryFormType = GeographicalLocationHierarchyLevel.Area;
        break;
      case GeographicalLocationHierarchyLevel.Street:
        this.title = this.isEditMode ? 'Desc_Edit_Geographical_Location_One' : 'Desc_Add_Geographical_Location_One';
        this.entryFormType = GeographicalLocationHierarchyLevel.Street;
        break;
    }
  }

  showEntryModal() {
    this.templateEntry.Show(true).then((res) => { });
  }

  initializeForm() {
    this.locationForm = new FormGroup({
      name: new FormControl("", Validators.required),
      code: new FormControl("", Validators.required),
    });
  }

  fillDescriptions() {

    let descriptions = this.otherLanguages;
    let oldValue = descriptions.findIndex(x => x.languageId === this.coreSession.selectedLanguageId);
    if (oldValue >= 0) descriptions.splice(oldValue, 1)
    if (descriptions.findIndex(x => x.languageId === this.coreSession.selectedLanguageId) < 0) {
      descriptions.push({
        languageId: this.coreSession.selectedLanguageId,
        description: this.locationForm.controls.name.value
      });
    }
    return descriptions;
  }

  filterTree() {
    if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Street) {
      let filteredTreeData;
      if (this.searchText) {
        // Filter the tree
        function filter(array, text) {

          const getChildren = (result, object) => {
            if (object.nodeName.toLocaleLowerCase().includes(text.toLocaleLowerCase())) {
              result.push(object);
              return result;
            }
            if (Array.isArray(object.children)) {
              const children = object.children.reduce(getChildren, []);
              if (children.length) result.push({ ...object, children });
            }
            return result;
          };

          return array.reduce(getChildren, []);
        }

        filteredTreeData = filter(this.nestedDataSourceTemp, this.searchText);
      } else {
        // Return the initial tree
        filteredTreeData = this.nestedDataSourceTemp;
      }

      // Build the tree nodes from Json object. The result is a list of `TodoItemNode` with nested
      // file node as children.
      this.nestedDataSource.data = this.nestedTreeControl.dataNodes = filteredTreeData;
      this.coreSession.ModalLoading.Hide();
    } else {
      this.filterDataFromDb()
    }

  }

  setLabels() {
    switch (this.geographicalLocationHierarchyLevel()) {
      case GeographicalLocationHierarchyLevel.Country:
        this.highLevelLabel = 'Desc_Geographical_Location_Code_Five';
        this.nameFieldProp.placeHolder = this.nameFieldProp.label = 'Desc_Geographical_Location_Name_Five';
        break;
      case GeographicalLocationHierarchyLevel.State:
        this.highLevelLabel = 'Desc_Geographical_Location_Code_Four';
        this.nameFieldProp.placeHolder = this.nameFieldProp.label = 'Desc_Geographical_Location_Name_Four';
        break;
      case GeographicalLocationHierarchyLevel.City:
        this.highLevelLabel = 'Desc_Geographical_Location_Code_Three';
        this.nameFieldProp.placeHolder = this.nameFieldProp.label = 'Desc_Geographical_Location_Name_Three';
        break;
      case GeographicalLocationHierarchyLevel.Area:
        this.highLevelLabel = 'Desc_Geographical_Location_Code_Two';
        this.nameFieldProp.placeHolder = this.nameFieldProp.label = 'Desc_Geographical_Location_Name_Two';
        break;
      case GeographicalLocationHierarchyLevel.Street:
        this.highLevelLabel = 'Desc_Geographical_Location_Code_One';
        this.nameFieldProp.placeHolder = this.nameFieldProp.label = 'Desc_Geographical_Location_Name_One';
        break;
    }
  }

  addActions() {
    this.addAction = (this.geographicalLocationHierarchyLevel() !== GeographicalLocationHierarchyLevel.Street) && this.coreSession.checkActivitiesAvailability(ConstantURLs.geoLocationURL, MenuActions.Add);
    this.editAction = this.coreSession.checkActivitiesAvailability(ConstantURLs.geoLocationURL, MenuActions.Edit);
    this.deleteAction = this.coreSession.checkActivitiesAvailability(ConstantURLs.geoLocationURL, MenuActions.Delete);
  }

  fillTooltip(level) {
    let tootltipText = '';
    switch (level) {
      case GeographicalLocationHierarchyLevel.Country:
        tootltipText = 'Desc_Add_Geographical_Location_Five';
        break;
      case GeographicalLocationHierarchyLevel.State:
        tootltipText = 'Desc_Add_Geographical_Location_Four';
        break;
      case GeographicalLocationHierarchyLevel.City:
        tootltipText = 'Desc_Add_Geographical_Location_Three';
        break;
      case GeographicalLocationHierarchyLevel.Area:
        tootltipText = 'Desc_Add_Geographical_Location_Two';
        break;
      case GeographicalLocationHierarchyLevel.Street:
        tootltipText = 'Desc_Add_Geographical_Location_One';

        break;
    }

    return tootltipText;
  }

  setNavigation() {

    let navigationList: NavigationData[] = [];
    navigationList.push(
      {
        Name: 'Desc_Data_Definition',
        URL: ConstantURLs.dataDefinitionURL
      },
      {
        Name: 'Desc_Master_Data',
        URL: ConstantURLs.dataDefinitionURL + '/' + ConstantURLs.masterURL
      },
      {
        Name: 'Desc_Geographical_Location_Hierarchy',
        URL: undefined
      }
    );
    this.navService.setNavigation(navigationList);
  }



  expandSpecificNode(node: any) {
    this.loadMore(node, true, false);
  }

  loadMore(node: any, expandNodes: boolean, openAfterOperation: boolean) {


    var locationHierarchyFilter =
    {
      locationHierarchyLevel: node.geographicalLocationHierarchyLevel - 1,
      parentId: node.nodeId,
      searchFilter: ''
    }
    this.coreSession.ModalLoading.Show();
    this.locationService.getGeoLocationHierarchyData(locationHierarchyFilter).subscribe(
      (response: ResponseModel) => {
        this.coreSession.ModalLoading.Hide();

        if (response.status != null && response.status >= 0) {
          switch (node.geographicalLocationHierarchyLevel) {
            case GeographicalLocationHierarchyLevel.Country:
              this.fillChilds(node, response.data);
              break;
            case GeographicalLocationHierarchyLevel.State:
              this.fillChilds(node, response.data);
              break;
            case GeographicalLocationHierarchyLevel.City:
              this.fillChilds(node, response.data);
              break;
            case GeographicalLocationHierarchyLevel.Area:
              this.fillChilds(node, response.data);
              break;
          }
          this.nestedTreeControl.dataNodes = this.nestedDataSource.data;
          //this.nestedDataSourceTemp.data = this.nestedDataSource.data;
          this.childrenChange.next(this.nestedTreeControl.dataNodes);
          if (expandNodes) {
            this.nestedTreeControl.expandAll();
          }
          if (openAfterOperation && !this.nestedTreeControl.isExpanded(node))
            this.nestedTreeControl.toggle(node);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });

  }

  fillChilds(node: any, childrens: any) {
    let firstLevelIndex = -1;
    let seconfLevelIndex = -1;
    let thirdLevelIndex = -1;
    let fourthLevelIndex = -1;

    switch (node.geographicalLocationHierarchyLevel - 1) {
      case GeographicalLocationHierarchyLevel.State:
        if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Country) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.countryId == node.countryId);
          this.nestedDataSource.data[firstLevelIndex].children = childrens;
        }
        break;
      case GeographicalLocationHierarchyLevel.City:
        if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.State) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.stateId == node.stateId && x.nodeId == node.nodeId);
          this.nestedDataSource.data[firstLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Country) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.countryId == node.countryId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.countryId == node.countryId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children = childrens;
        }
        break;
      case GeographicalLocationHierarchyLevel.Area:

        if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.City) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.stateId == node.stateId && x.nodeId == node.nodeId);
          this.nestedDataSource.data[firstLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.State) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.stateId == node.stateId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.cityId == node.cityId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Country) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.countryId == node.countryId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.countryId == node.countryId);
          thirdLevelIndex = this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.countryId == node.countryId && x.cityId == node.cityId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children[thirdLevelIndex].children = childrens;
        }
        break;
      case GeographicalLocationHierarchyLevel.Street:

        if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Area) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.areaId == node.areaId);
          this.nestedDataSource.data[firstLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.City) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.cityId == node.cityId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.areaId == node.areaId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.State) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.stateId == node.stateId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.cityId == node.cityId);
          thirdLevelIndex = this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children.findIndex(x => x.stateId == node.stateId && x.cityId == node.cityId && x.areaId == node.areaId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children[thirdLevelIndex].children = childrens;
        } else if (this.geographicalLocationHierarchyLevel() == GeographicalLocationHierarchyLevel.Country) {
          firstLevelIndex = this.nestedDataSource.data.findIndex(x => x.countryId == node.countryId);
          seconfLevelIndex = this.nestedDataSource.data[firstLevelIndex].children.findIndex(x => x.stateId == node.stateId);
          thirdLevelIndex = this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children.findIndex(x => x.cityId == node.cityId);
          fourthLevelIndex = this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children[thirdLevelIndex].children.findIndex(x => x.areaId == node.areaId);
          this.nestedDataSource.data[firstLevelIndex].children[seconfLevelIndex].children[thirdLevelIndex].children[fourthLevelIndex].children = childrens;
        }
        break;
    }

  }

  filterDataFromDb() {

    var locationHierarchyFilter =
    {
      locationHierarchyLevel: 0,
      parentId: null,
      searchFilter: this.searchText
    }
    if (this.searchText) {
      this.locationService.getGeoLocationHierarchyData(locationHierarchyFilter).subscribe(
        (response: ResponseModel) => {
          if (response.status != null && response.status >= 0) {
            this.nestedDataSource.data = this.nestedTreeControl.dataNodes = response.data;
            if (response.data && response.data.length > 0)
              this.nestedTreeControl.expandAll();
            this.coreSession.ModalLoading.Hide();
          }
        }, (error: HttpErrorResponse) => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        });
    } else {
      this.populateGeoLocationHierarchyData();
    }
  }



  geographicalLocationHierarchyLevel() : number {
    return + this.sessionData.getConfigurationValue(ConstantConfigurations.GeographicalLocationHierarchyLevel).toLowerCase();
  }

  //#endregion

}
