import { Component, OnInit } from '@angular/core';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { FormControl, FormGroup } from '@angular/forms';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../../core/core.session';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { Subject } from 'rxjs';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { VehicleService } from '../../../../shared/services/data-definition/warehouse-and-vehicle/vehicle.service';

@Component({
  selector: 'app-vehicle-list',
  templateUrl: './vehicle-list.component.html',
  styleUrls: ['./vehicle-list.component.css']
})
export class VehicleListComponent implements OnInit {

  showVehicleEntrySlider = false;
  saveSubject: Subject<void> = new Subject<void>();
  vehiclesFilterForm: FormGroup;
  isEditMode = false;
  vehicleObject: any = null;
  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Vehicle_Name', key: 'vehicleName', isSortable: true, width: '15%' },
      { title: 'Desc_Vehicle_Code', key: 'vehicleCode', isSortable: true, width: '15%' },
      { title: 'Desc_Barcode', key: 'barcode', isSortable: true, width: '15%', },
      { title: 'Desc_Organization', key: 'organizationCodeName', isSortable: true, width: '15%' },
      { title: 'Desc_Employee', key: 'employeeCodeName', isSortable: true, width: '15%' },
      { title: 'Desc_Vehicle_Type', key: 'vehicleTypeName', isSortable: true, width: '15%' },
      { title: 'Desc_Status', key: 'vehicleStatusName', isSortable: false, width: '10%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  vehiclesFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    includeInactive: false,
    employeeId: -1,
    organizationId: -1
  };
  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private vehicleService: VehicleService,
    private sessionData: SessionDataProvider) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Vehicles');
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.vehiclesURL, MenuActions.Add);
    this.addDeleteAction();
    this.initFilterForm();
    this.populateVehiclesList();
  }
  addDeleteAction() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.vehiclesURL, MenuActions.Delete)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Delete',
          icon: 'fa fa-close',
          color: '#f1685e',
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.vehiclesURL, MenuActions.Edit)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Deactivate,
          title: 'Desc_Deactivate',
          icon: 'fa fa-toggle-off',
          color: '#f1685e',
          showWhenKeyValueFalse: true,
          showHideOptionPerLine: true,
          controlKeyName: 'inactive'
        },
        {
          operation: RowOperation.Activate,
          title: "Desc_Reactivate",
          icon: "fa fa-toggle-on",
          color: "#02b875",
          controlKeyName: "inactive",
          showWhenKeyValueTrue: true,
          showHideOptionPerLine: true
        },{
          operation: RowOperation.edit,
          title: "Desc_Edit",
          icon: "fa fa-pencil",
          color: "#12344d"
        })
    }
  }
  populateVehiclesList() {
    this.vehiclesFilter.employeeId = this.vehiclesFilterForm.get('employeeId').value == null ? -1 : this.vehiclesFilterForm.get('employeeId').value,
      this.vehiclesFilter.organizationId = this.vehiclesFilterForm.get('organizationId').value == null ? -1 : this.vehiclesFilterForm.get('organizationId').value,
      this.vehiclesFilter.includeInactive = this.vehiclesFilterForm.get('includeInactive').value == null ? false : this.vehiclesFilterForm.get('includeInactive').value;

    this.coreSession.ModalLoading.Show();
    this.vehicleService.getVehiclesListSharedTable(this.vehiclesFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data,
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  initFilterForm() {
    this.vehiclesFilterForm = new FormGroup({
      organizationId: new FormControl(),
      employeeId: new FormControl(),
      includeInactive: new FormControl(false)
    });
  }
  onTableFilterChanged(query: any) {
    this.vehiclesFilter.customListFilter = query;
    this.populateVehiclesList();
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.delete: {
        this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
          (result: DialogResult) => {
            if (result === DialogResult.Yes) {
        this.deleteVehicle(event.object);
            }
          });
        break;
      }
      case RowOperation.Deactivate: {
        this.deactivateVehicle(event.object);
        break;
      }
      case RowOperation.Activate: {
        this.activateVehicle(event.object);
        break;
      }
      case RowOperation.edit:
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.vehiclesURL, MenuActions.Edit)) {
          this.editVehicle(event.object);
        }
        break;
    }
  }
  editVehicle(vehicle) {
    this.coreSession.ModalLoading.Show();
    this.vehicleService.getVehicleData(vehicle.vehicleId).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          vehicle.defaultLoadingWarehouseId = response.data.defaultLoadingWarehouseId;
          vehicle.defaultOffloadingWarehouseId = response.data.defaultOffloadingWarehouseId;
          vehicle.loadingWarehouses = response.data.loadingWarehouses;
          vehicle.offloadWarehouses = response.data.offloadWarehouses;
          vehicle.damagedWarehouses = response.data.damagedWarehouses;
          vehicle.descriptionLanguages = response.data.descriptionLanguages;
          if(this.linkVehicleToAccount())
          {
            vehicle.accountId = response.data.accountId;
            vehicle.balance = response.data.balance;
            vehicle.creditLimit = response.data.creditLimit;
            vehicle.currencyId = response.data.currencyId;
          }
          this.openSliderToEditVehicle(vehicle)
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  deleteVehicle(vehicle) {
    this.coreSession.ModalLoading.Show();
    this.vehicleService.deleteVehicle(vehicle.warehouseId).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
        this.populateVehiclesList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  deactivateVehicle(vehicle) {
    this.coreSession.ModalLoading.Show();
    this.vehicleService.deactivateVehicle(vehicle.warehouseId).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
        this.populateVehiclesList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  activateVehicle(vehicle) {
    this.coreSession.ModalLoading.Show();
    this.vehicleService.activateVehicle(vehicle.warehouseId).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
        this.populateVehiclesList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  onFindButtonClicked() {
    this.vehiclesFilter.customListFilter.page = 0;
    this.populateVehiclesList();
  }
  onReset() {
    this.vehiclesFilterForm.reset();
    this.vehiclesFilterForm.get('includeInactive').setValue(false);
  }
  openSliderToAddNewVehicle() {
    this.isEditMode = false;
    this.vehicleObject = null;
    this.showVehicleEntrySlider = true;
  }
  openSliderToEditVehicle(vehicle) {
    this.isEditMode = true;
    this.vehicleObject = vehicle;
    this.showVehicleEntrySlider = true;
  }
  closeVehicleEntrySlider() {
    this.showVehicleEntrySlider = false;
    this.isEditMode = false;
    this.coreSession.SetTitle('Vehicles');
  }
  onSaveVehicleClick() {
    this.saveSubject.next()
  }
  afterSaveVehicle() {
    this.closeVehicleEntrySlider();
    // this.onReset();
    this.populateVehiclesList();
  }

  linkVehicleToAccount()
  {
    return  this.sessionData.getConfigurationValue(ConstantConfigurations.LinkVehicleToAccount).toLowerCase() === "true";
  }
}
