import { Component, OnInit } from '@angular/core';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { FormControl, FormGroup } from '@angular/forms';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../../core/core.session';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { Subject } from 'rxjs';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { WarehouseService } from '../../../../shared/services/data-definition/warehouse-and-vehicle/warehouse.service';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';

@Component({
  selector: 'app-warehouse-list',
  templateUrl: './warehouse-list.component.html',
  styleUrls: ['./warehouse-list.component.css']
})
export class WarehouseListComponent implements OnInit {

  showWarehouseEntrySlider = false;
  saveSubject: Subject<void> = new Subject<void>();
  warehouseFilterForm: FormGroup;
  isEditMode = false;
  warehouseObject: any = null;
  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Warehouse_Name', key: 'warehouseName', isSortable: true, width: '15%' },
      { title: 'Desc_Warehouse_Code', key: 'warehouseCode', isSortable: true, width: '15%' },
      { title: 'Desc_Barcode', key: 'barcode', isSortable: true, width: '15%', },
      { title: 'Desc_Organization', key: 'organizationCodeName', isSortable: true, width: '20%' },
      { title: 'Desc_Warehouse_Type', key: 'warehouseTypeName', isSortable: true, width: '15%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  warehouseFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    warehouseTypeId: -1,
    organizationId: -1
  };
  allowSelectAlternativeWarehouseInFO: boolean = false;

  addedWarehouseAlternatives: any[] = [];
  availableWarehouseAlternatives: any[] = [];
  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private warehouseService: WarehouseService,
    private sessionData: SessionDataProvider) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Warehouses');
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.warehousesURL ,MenuActions.Add);
    this.addActions();
    this.initFilterForm();
    this.populateWarehousesList();
    this.allowSelectAlternativeWarehouseInFO = this.sessionData.getConfigurationValue(ConstantConfigurations.AllowSelectAlternativeWarehouseInFO).toLowerCase() === "true";
  }
  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.warehousesURL ,MenuActions.Delete)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Delete',
          icon: 'fa fa-close',
          color: '#f1685e',
        });
    }

    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.warehousesURL ,MenuActions.Edit)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.edit,
          title: "Desc_Edit",
          icon: "fa fa-pencil",
          color: "#12344d"
        });
    }
  }
  populateWarehousesList() {
    this.warehouseFilter.organizationId = this.warehouseFilterForm.get('organizationId').value == null ? -1 : this.warehouseFilterForm.get('organizationId').value,
    this.warehouseFilter.warehouseTypeId = this.warehouseFilterForm.get('warehouseTypeId').value == null ? -1 : this.warehouseFilterForm.get('warehouseTypeId').value,
    this.coreSession.ModalLoading.Show();

    this.warehouseService.getWarehousesListSharedTable(this.warehouseFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data,
        }

      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });

  }
  initFilterForm() {
    this.warehouseFilterForm = new FormGroup({
      organizationId: new FormControl(),
      warehouseTypeId: new FormControl(),
    });
  }
  onTableFilterChanged(query: any) {
    this.warehouseFilter.customListFilter = query;
    this.populateWarehousesList();
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.delete: {
        this.deleteWarehouse(event.object);
        break;
      }
      case RowOperation.edit:
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.warehousesURL ,MenuActions.Edit)) {
          this.editWarehouse(event.object);
        }
        break;
    }
  }
  editWarehouse(warehouse) {
    this.coreSession.ModalLoading.Show();
    this.warehouseService.getWarehouseData(warehouse.warehouseId, warehouse.warehouseTypeId, warehouse.organizationId).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
         warehouse.storeKeepers=response.data.warehouse.storeKeepers;
          warehouse.sources=response.data.warehouse.sources;
          warehouse.destinations=response.data.warehouse.destinations;
          warehouse.validToChangeToTruck = response.data.warehouse.validToChangeToTruck;
          warehouse.validToChangeFromTruck = response.data.warehouse.validToChangeFromTruck;
          warehouse.itemCategories = response.data.warehouse.itemCategories;
          warehouse.descriptionLanguages = response.data.warehouse.descriptionLanguages;
          warehouse.addressLanguages = response.data.warehouse.addressLanguages;
          warehouse.allowEditOrganization = response.data.warehouse.allowEditOrganization;
          this.addedWarehouseAlternatives = response.data.addedAlternativeWarehouses ? response.data.addedAlternativeWarehouses : [];
          this.availableWarehouseAlternatives = response.data.availableAlternativeWarehouses ? response.data.availableAlternativeWarehouses : [];
          this.openSliderToEditWarehouse(warehouse)
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  deleteWarehouse(warehouse) {
    this.coreSession.ModalLoading.Show();
    this.warehouseService.deleteWarehouse(warehouse.warehouseId).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
        this.populateWarehousesList();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  onFindButtonClicked() {
    this.warehouseFilter.customListFilter.page = 0;
    this.populateWarehousesList();
  }
  onReset() {
    this.warehouseFilterForm.reset();
  }
  openSliderToAddNewWarehouse() {
    this.isEditMode = false;
    this.warehouseObject = null;
    this.showWarehouseEntrySlider = true;
  }
  openSliderToEditWarehouse(selectedWarehouse) {
    this.isEditMode = true;
    this.warehouseObject = selectedWarehouse;
    this.showWarehouseEntrySlider = true;
  }
  closeWarehouseEntrySlider() {
    this.warehouseObject = null;
    this.showWarehouseEntrySlider = false;
    this.isEditMode = false;
    this.coreSession.SetTitle('Warehouses');
  }
  onSaveWarehouseClick() {
    this.saveSubject.next()
  }
  afterSaveWarehouse() {
    this.closeWarehouseEntrySlider();
    this.populateWarehousesList();
  }
}


