import { Component, OnInit, ViewChild } from '@angular/core';
import { FormControl, FormGroup, Validators } from '@angular/forms';
import { NavigationEnd, Router } from '@angular/router';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../../core/core.session';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { SerialOperationService } from '../../../serial-operation/serial-operation.service';
import { WarehouseService } from '../../../../shared/services/data-definition/warehouse-and-vehicle/warehouse.service';
import { ConstantConfigurations } from '../../../../shared/models/constants/constant-configuration';

@Component({
  selector: 'app-warehouse-stock-list',
  templateUrl: './warehouse-stock-list.component.html',
  styleUrls: ['./warehouse-stock-list.component.css']
})
export class WarehouseStockListComponent implements OnInit {

  //#region [Declarations]
  warhouseId: number;
  filterForm: FormGroup;
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  stockFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    warehouseId: -1,
    divisionId: -1,
    categoryId: -1,
    itemId: -1
  };

  tableData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    openFilterDirectly: true,
    showEditButton: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Item_Name', key: 'itemName', isSortable: true, width: this.useDefaultExpiryAndBatchForItems() ? '20%' : '18%' },
      { title: 'Desc_Item_Code', key: 'itemCode', isSortable: true, width: this.useDefaultExpiryAndBatchForItems() ? '15%' : '12%' },
      { title: 'Desc_UOM', key: 'uom', isSortable: true, width: this.useDefaultExpiryAndBatchForItems() ? '12%' : '8%' },
      { title: 'Desc_Category', key: 'category', isSortable: true, width: this.useDefaultExpiryAndBatchForItems() ? '15%' : '13%' },
      { title: 'Desc_Division', key: 'division', isSortable: true, width: this.useDefaultExpiryAndBatchForItems() ? '15%' : '13%' },
      {
        title: 'Desc_Quantity', key: 'quantity', isSortable: true, numberOfStockDigits: true,
        width: this.useDefaultExpiryAndBatchForItems() ? '13%' : '12%'
      },
    ]
  };
  selectedItemId = -1;
  currentFormControlName = 'itemId';
  dataSourceSerials: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  sharedTabelDataForSerials: ITableProperties = {
    pageSize: 10,
    showPaginator: true,
    isOnline: false,
    showSearch: true,
    showEditButton: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Serial_No', key: 'fromSerial', isSortable: false, width: '50%' },
      //{ title: 'Desc_Transaction_ID', key: 'transactionId', isSortable: false, width: '50%' },
    ]
  };
  @ViewChild('serialsDialog', { static: true }) serialsDialog: NgbModal;
  getDamagedWH = false;
  //#endregion

  //#region [Events]
  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private WarehouseService: WarehouseService,
    private sessionData: SessionDataProvider,
    private serialOperationService: SerialOperationService,
    private modalService: NgbModal
  ) {
  }

  ngOnInit() {
    this.coreSession.SetTitle('Current Stock');
    this.showOrHideExpiryAndBatch();
    this.addAction();

    this.filterForm = new FormGroup({
      'allData': new FormGroup({
        warehouseId: new FormControl([], Validators.required),
        divisionId: new FormControl(),
        itemCategoryId: new FormControl(),
        itemId: new FormControl()
      })
    });

    this.getDamagedWH = this.sessionData.getConfigurationValue(ConstantConfigurations.RestoreNonSellableReturns).toLowerCase() === "true";
  }

  onFindButtonClicked() {

    if (this.filterForm.get('allData').get('warehouseId').invalid) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
    } else {
      this.stockFilter.warehouseId = this.filterForm.get('allData').get('warehouseId').value == null ? -1 : this.filterForm.get('allData').get('warehouseId').value;
      this.stockFilter.divisionId = this.filterForm.get('allData').get('divisionId').value == null ? -1 : this.filterForm.get('allData').get('divisionId').value;
      this.stockFilter.categoryId = this.filterForm.get('allData').get('itemCategoryId').value == null ? -1 : this.filterForm.get('allData').get('itemCategoryId').value;
      this.stockFilter.itemId = this.filterForm.get('allData').get('itemId').value == null ? -1 : this.filterForm.get('allData').get('itemId').value;
      this.coreSession.ModalLoading.Show();
      this.WarehouseService.GetCurrentStock(this.stockFilter).subscribe(
        response => {
          this.coreSession.ModalLoading.Hide();
          if (response.status != null && response.status >= 0) {
            this.dataSource = {
              totalItems: response.data.totalItems,
              data: response.data.data,
            }
          } else {
            this.dataSource = {
              totalItems: 0,
              data: [],
            };
          }
        },
        error => {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
      );
    }

  }


  onReset() {
    this.filterForm.reset();
  }

  onTableFilterChanged(query: any) {
    this.stockFilter.customListFilter = query;
    this.onFindButtonClicked();
  }

  onRowClicked(event: any) {
    switch (event.operation) {
      case RowOperation.View:
        this.fillSerialsAndOpenDialog(event.object);
        break;
    }
  }
  //#endregion


  //#region [Functions]

  showOrHideExpiryAndBatch() {
    if (!this.useDefaultExpiryAndBatchForItems()) {
      this.tableData.columns.push({ title: 'Desc_BatchNo', key: 'batchNo', isSortable: true, width: '12%' });
      this.tableData.columns.push({ title: 'Desc_ExpiryDate', key: 'expiryDate', isSortable: true, isExpiryDate: true, width: '12%' });
    }
  }

  addAction() {
      this.tableData.rowOperations.push(
        {
          operation: RowOperation.View,
          title: 'Desc_Serials',
          icon: 'fa fa-barcode',
          color: '#f1685e',
          showWhenKeyValueTrue: true,
          showHideOptionPerLine: true,
          controlKeyName: "hasSerialNumber"
        });
  }

  useDefaultExpiryAndBatchForItems() {
    return this.sessionData.getConfigurationValue('UseDefaultExpiryAndBatchForItems').toString().toLowerCase() === "true";
  }

  fillSerialsAndOpenDialog(itemObj: any) {
    var filter =
    {
      packId: itemObj.packId,
      expiryDateModel: this.coreSession.getDateTimeModel(new Date(itemObj.expiryDate)),
      filterByBatch: true,
      batchNo: itemObj.batchNo,
      itemSerial: '',
      customerId: -1,
      outletId: -1,
      warehouseId: itemObj.warehouseId,
      isCurrentStockMode : true,
      isVehicle : itemObj.isVehicle

    }
    this.coreSession.ModalLoading.Show();
    this.serialOperationService.getSerialsForLocationItem(filter).subscribe(response => {
      if (response.status != null && response.status >= 0) {
        if (response.data && response.data.length > 0) {
          this.coreSession.ModalLoading.Hide();
          this.dataSourceSerials = { data: response.data, totalItems: response.data.length };
          return this.modalService.open(this.serialsDialog, { centered: true, scrollable: true }).result.then(
            (result) => {
              if (result === DialogResult.Ok) {
                return this.serialsDialog;
              } else {
                return -1;
              }
            });
        } else {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgNoSerials));
        }
      } else {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      error => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      })

  }
  //#endregion

  onCategorySelected(item) {
    this.filterForm.get('allData').get('itemId').setValue(null);
  }

  onDivisionSelected(item) {
    this.filterForm.get('allData').get('itemCategoryId').setValue(null);
    this.filterForm.get('allData').get('itemId').setValue(null);
  }

}
