import { Component, OnInit, ViewChild } from '@angular/core';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { IDateTimePickerProperties } from '../../../shared/models/dateTimeModel/date-time-properties.interface';
import { ActivatedRoute, Router, Params } from '@angular/router';
import { Location } from '@angular/common';
import { OrderListFilter } from '../../../shared/models/shared-table/custom-list.interface';
import { OrderService } from '../order.service';
import { ISharedItemsProperties } from '../../../shared/components/items/shared-items-properties.interface';
import { FormProcessMode } from '../../../shared/models/enums/form-process-mode.enum';
import { CoreSession } from '../../../core/core.session';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { GeneralFilter } from '../../../shared/models/general/generalFilter.model';
import { NgbDropdownConfig } from '@ng-bootstrap/ng-bootstrap';
import { TotalsModel } from '../../../shared/models/Item/totals.model';
import { TakePromotionComponent } from '../../../shared/components/promotion/take-promotion/take-promotion.component';
import { ConstantURLs } from '../../../shared/models/constants/constant-URL';
import { ValidationService } from '../../../shared/services/validation.service';
import { TransationCommonData } from '../../../shared/models/transaction/transaction-common-data';
import { CalculationService } from '../../../shared/services/calculations.service';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';
import { TranslateService } from '@ngx-translate/core';
import { Subject } from 'rxjs';
import { OrderOperation } from '../../../shared/models/enums/order-operation .enum';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { OrderPendingService } from '../../../shared/services/order/order-pending.service';
import { OrderStatus } from '../../../shared/models/enums/order-status.enum';

@Component({
  selector: 'app-add-order',
  templateUrl: './add-order.component.html',
  styleUrls: ['./add-order.component.css']
})
export class AddOrderComponent implements OnInit {

  //#region [DECLARATIONS]
  @ViewChild('takePromotions', { static: true }) takePromotions: TakePromotionComponent;
  tabsArray: boolean[] = [false, false, false];
  tabsTitle: string[] = ['Desc_General_Data', 'Desc_Details_Data', 'Desc_Order_Summary'];

  customerPromotions: any[] = [];
  selectedTabIndex: number;
  resetForms = false;
  isEditMode = false;
  isloading = false;
  isSaveAndHome = false;
  issaveAndNew = false;
  disableHeaderData = false;
  orderForm: FormGroup;
  minDate = new Date();
  recurringObj: any;
  showSlider = false;
  saveSubject: Subject<void> = new Subject<void>();


  controlTransactionOnDivisionLevel = false;

  orderId: string;
  customerId: number;
  outletId: number;
  employeeId: number;
  divisionId = -1;
  // orderDate: DateTimeModel;
  desiredDeliveryDate: DateTimeModel;

  getPrices = false;
  totals: TotalsModel;
  itemFormProperties: ISharedItemsProperties = {
    addedItemList: [],
    promotedItemList: [],
    generalFilter: new GeneralFilter()
  };

  desiredDeliveryDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Desired_Delivery_Date',
    formControlName: 'desiredDeliveryDate',
  };

  query: OrderListFilter = {};

  order: any = {};
  transactionCommonData: TransationCommonData;

  //#endregion

  //#region [Life Cycle Hooks]
  constructor(
    private route: ActivatedRoute,
    private location: Location,
    private orderService: OrderService,
    private validationService: ValidationService,
    public coreSession: CoreSession,
    config: NgbDropdownConfig,
    private router: Router,
    private calculationService: CalculationService,
    private translateService: TranslateService,
    public sessionData: SessionDataProvider,
    public orderPendingService: OrderPendingService
  ) {
    // customize default values of dropdowns used by this component tree
    config.placement = 'left';
    config.autoClose = true;
  }

  ngOnInit() {
    this.totals = new TotalsModel();
    this.tabsArray[0] = true;
    this.selectedTabIndex = 0;
    this.isSaveAndHome = false;
    this.issaveAndNew = false;
    this.route.params.subscribe((param: Params) => {
      this.query.orderSignature = param['orderSignature'];
      this.isEditMode = param['orderSignature'] != null;
    });
    if (this.isEditMode) {
      this.prepareOrderDataInEditMode();
      this.coreSession.SetTitle('Edit Pending Order');
    } else {
      this.initForm();
      this.coreSession.SetTitle('Add Pending Order');
    }
  }

  //#endregion

  //#region [Reactive Form]
  initForm() {
    this.orderForm = new FormGroup({
      headerData: new FormGroup({
        customerId: new FormControl(
          { value: this.customerId, disabled: this.isEditMode },
          this.isEditMode ? null : Validators.required
        ),
        outletId: new FormControl(
          { value: this.outletId, disabled: this.isEditMode },
          this.isEditMode ? null : Validators.required
        ),
        employeeId: new FormControl(
          { value: this.employeeId, disabled: this.isEditMode },
          this.isEditMode ? null : Validators.required
        ),
        divisionId: new FormControl({
          value: this.divisionId,
          disabled: this.isEditMode
        }),
        desiredDeliveryDate: new FormControl(
          {
            value: this.desiredDeliveryDate,
            disabled: false
          },
          Validators.required
        )
      })
    });
  }

  markAsTouched() {
    (<FormGroup>(
      this.orderForm.get('headerData')
    )).controls.customerId.markAsTouched();
    (<FormGroup>(
      this.orderForm.get('headerData')
    )).controls.outletId.markAsTouched();
    (<FormGroup>(
      this.orderForm.get('headerData')
    )).controls.employeeId.markAsTouched();
    (<FormGroup>(
      this.orderForm.get('headerData')
    )).controls.divisionId.markAsTouched();
    // (<FormGroup>(
    //   this.orderForm.get('headerData')
    // )).controls.orderDate.markAsTouched();
    (<FormGroup>(
      this.orderForm.get('headerData')
    )).controls.desiredDeliveryDate.markAsTouched();
  }

  //#endregion

  //#region [Fill order data]
  prepareOrderDataInEditMode() {
    this.isloading = true;
    this.coreSession.ModalLoading.Show();
    this.orderService.getOrder(this.query).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.order = response.data;
          this.itemFormProperties.addedItemList = this.order.soldItems.slice();
          this.itemFormProperties.promotedItemList = this.order.promotedItems.slice();
          this.order.isEditMode = true;
          this.fillFormProperties();
          this.fillDatesPropertie();
          this.initForm();
          this.fillRecurringData();
          this.isloading = false;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  fillFormProperties() {
    this.customerId = this.order.customerId;
    this.outletId = this.order.outletId;
    this.employeeId = this.order.employeeId;
    this.divisionId = this.order.divisionId;
  }

  fillDatesPropertie() {
    this.desiredDeliveryDate = this.order.desiredDeliveryDateModel;
    // this.orderDate = this.order.desiredDeliveryDateModel;

    this.desiredDeliveryDateProperties.dateValue = this.desiredDeliveryDate;
    // this.orderDateProperties.dateValue = this.orderDate;

  }

  fillOrderHeaderDataProperties() {
    this.order.customerId = (<FormGroup>this.headerData).controls.customerId.value;
    this.order.outletId = (<FormGroup>this.headerData).controls.outletId.value;
    this.order.employeeId = (<FormGroup>this.headerData).controls.employeeId.value;
    const divisionId = (<FormGroup>this.headerData).controls.divisionId.value;
    this.order.divisionId = divisionId === null || divisionId === '' ? -1 : divisionId;
    this.order.desiredDeliveryDateModel = this.desiredDeliveryDate;
  }
  onDesiredDeliveryDateSelected(date) {
    this.desiredDeliveryDate = date;
  }
  //#endregion

  //#region [Tabs Validations]
  previous() {
    if (this.selectedTabIndex === 2 && this.order.isPromotionTaken) {
      this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgCancelPromotionConfirmation), DialogMode.YesNo).then(
        (res: DialogResult) => {
          if (res === DialogResult.Yes) {
            // cancel promotions
            this.calculateTotalAfterCancelPromotions();
            this.order.isPromotionTaken = false;
            this.tabsArray.fill(false);
            this.tabsArray[--this.selectedTabIndex] = true;
          }
        }
      );
    } else {
      this.tabsArray.fill(false);
      this.tabsArray[--this.selectedTabIndex] = true;
    }

  }

  next() {
    this.markAsTouched();
    if (!this.orderForm.invalid) {
      this.fillOrderHeaderDataProperties();
      this.checkOrderHeader();
    }
  }

  fillGeneralFilterData() {
    if (!this.itemFormProperties) {
      this.itemFormProperties = {
        addedItemList: [],
        promotedItemList: [],
        generalFilter: new GeneralFilter()
      };
    }
    this.itemFormProperties.generalFilter = new GeneralFilter();
    this.itemFormProperties.generalFilter.customerId = this.order.customerId;
    this.itemFormProperties.generalFilter.outletId = this.order.outletId;
    this.itemFormProperties.generalFilter.divisionId = this.order.divisionId;
    this.itemFormProperties.generalFilter.employeeId = this.order.employeeId;
    this.itemFormProperties.generalFilter.vehicleId = this.order.vehicleId;
    this.itemFormProperties.generalFilter.warehouseId = this.order.warehouseId;
    this.itemFormProperties.generalFilter.formProcessMode = FormProcessMode.Orders;
    this.itemFormProperties.generalFilter.getPrices = true;
    this.getPrices = true;
  }

  back() {
    this.router.navigate(['../'], { relativeTo: this.route });
  }

  moveToNextTab() {
    this.tabsArray.fill(false);
    this.tabsArray[++this.selectedTabIndex] = true;
  }

  checkOut() {
    if (this.itemFormProperties.addedItemList.length === 0) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgSelectItem));
      return;
    }
    this.order.soldItems = this.itemFormProperties.addedItemList;
    (this.itemFormProperties.addedItemList);
    this.order.promotedItems = this.itemFormProperties.promotedItemList;
    this.order.allItems = [];
    this.order.allItems = this.order.allItems.concat(this.order.soldItems);
    this.order.allItems = this.order.allItems.concat(this.order.promotedItems);
    this.order.netTotal = this.totals.netTotal;
    this.order.grossTotal = this.totals.grossTotal;
    this.order.tax = this.totals.taxTotal;
    this.order.calculatedRetailTax = this.totals.retailTaxTotal;
    this.order.discount = this.totals.discountTotal;
    this.order.promotedDiscount = this.totals.promotedDiscountTotal;
    this.FillTransactionCommonData();
    if (this.order.isPromotionTaken) { // ToDo And Check allow promotions Configuration
      this.moveToNextTab();
    } else {
      this.checkoutOrder();
    }

  }

  //#endregion

  //#region [PROMOTIONS]
  checkoutOrder() {
    this.coreSession.ModalLoading.Show();
    this.validationService.checkOutPromotions(this.transactionCommonData).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          if (response.data.customerPromotions.length > 0) {
            this.customerPromotions = response.data.customerPromotions;
            this.openTakePromotions();
          } else {
            this.moveToNextTab();
          }
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), error.message);
      }
    );
  }

  FillTransactionCommonData() {
    this.transactionCommonData.customerId = this.order.customerId;
    this.transactionCommonData.outletId = this.order.outletId;
    this.transactionCommonData.divisionId = this.order.divisionId;
    this.transactionCommonData.employeeId = this.order.employeeId;
    this.transactionCommonData.itemFormProcessMode = FormProcessMode.Orders;
    this.transactionCommonData.soldItems = this.order.soldItems;
    this.transactionCommonData.promotedItems = this.order.promotedItems;
    this.transactionCommonData.netTotal = this.order.netTotal;
    this.transactionCommonData.grossTotal = this.order.grossTotal;
    this.transactionCommonData.orderStatus = this.order.orderStatus;
    this.transactionCommonData.tax = this.order.tax;
  }
  checkOrderHeader() {
    this.transactionCommonData = new TransationCommonData();
    this.FillTransactionCommonData();
    this.coreSession.ModalLoading.Show();
    this.validationService.checkHeaderData(this.transactionCommonData).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.order.outlet = response.data.outlet;
          this.transactionCommonData.noDocumentSequance = response.data.noDocumentSequance;
          this.transactionCommonData.draftTransactionId = response.data.draftTransactionId;

          this.checkCustomerValidations();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  openTakePromotions() {
    // this.takePromotions.Show().then(
    //   (result) => {
    //     if (result !== 0) {
    //     }
    //   }
    // );
  }

  onPromotionTaken(selectedBenifets) {
    if (selectedBenifets) {
      this.order.customerPromotions = selectedBenifets;
      this.transactionCommonData.customerPromotions = selectedBenifets;
      this.takeSelectedPromotions();
    }
  }


  takeSelectedPromotions() {
    this.coreSession.ModalLoading.Show();
    this.validationService.takePromotions(this.transactionCommonData).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.transactionCommonData = response.data;
          this.order.soldItems = this.transactionCommonData.soldItems;
          this.order.promotedItems = this.transactionCommonData.promotedItems;
          // set taken promotions in Back end to order promotion
          this.order.customerPromotions = this.transactionCommonData.customerPromotions;
          this.order.allItems = [];
          this.order.allItems = this.order.allItems.concat(this.order.soldItems);
          this.order.allItems = this.order.allItems.concat(this.order.promotedItems);
          this.calculateTotalAfterPromotions();
          this.order.isPromotionTaken = true;
          this.moveToNextTab();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  //#endregion

  //#region [SAVING]
  save() {
    this.isSaveAndHome = true;
    if (this.isEditMode)
      this.updateOrder();
    else
      this.saveOrder();
  }

  saveAndNew() {
    this.issaveAndNew = true;
    if (this.isEditMode)
      this.updateOrder();
    else
      this.saveOrder();
  }

  saveOrder() {
    this.coreSession.ModalLoading.Show();
    // save just to pending order
    this.orderPendingService.insertOrder(this.order).subscribe(
      (response) => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
          if (this.isSaveAndHome) {
            this.router.navigate(['../'], { relativeTo: this.route });
          } else {
            this.router.navigateByUrl('/RefrshComponent', { skipLocationChange: true }).then(() =>
              this.router.navigate([ConstantURLs.orderManagementURL + '/' + ConstantURLs.addPendingOrderURL]));
            // RELOAD SCREEN
          }
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  updateOrder() {
    this.coreSession.ModalLoading.Show();
    this.order.orderOperation = OrderOperation.EditValues;
    if (this.order.newStatusId == OrderStatus.New.valueOf) {
      this.orderPendingService.updateOrder(this.order).subscribe(
        (response) => {
          this.onUpdateOrderSuccess(response);
        }, (error: HttpErrorResponse) => {
          this.onUpdateOrderError(error);
        }
      );
    } else {
      this.orderService.updateOrder(this.order).subscribe(
        (response) => {
          this.onUpdateOrderSuccess(response);
        }, (error: HttpErrorResponse) => {
          this.onUpdateOrderError(error);
        }
      );
    }

  }

  onUpdateOrderSuccess(response) {
    this.coreSession.ModalLoading.Hide();
    if (response.status != null && response.status >= 0) {
      this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
      if (this.isSaveAndHome) {
        this.router.navigate(['../'], { relativeTo: this.route });
      } else {
        this.router.navigateByUrl('/RefrshComponent', { skipLocationChange: true }).then(() =>
          this.router.navigate([ConstantURLs.orderManagementURL + '/' + ConstantURLs.addPendingOrderURL]));
        // RELOAD SCREEN
      }
    } else {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
    }
  }

  onUpdateOrderError(error) {
    this.coreSession.ModalLoading.Hide();
    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
  }
  //#endregion

  //#region [Customer Validation]
  checkCustomerValidations() {
    if (this.transactionCommonData.noDocumentSequance === true) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgNoDocumentSequence));
      return;
    }
    if (this.order.outlet.customerValidations.noAccount === true) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgNoCustomerAccount));
      return;
    }
    if (this.order.outlet.customerValidations.isOnHold === true) {
      // Customer onHold
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgOnHoldCustomer));
      return;
    }
    if (this.order.outlet.customerValidations.isViolatePaymentTerms === true) {
      // Customer Violte payment terms
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgCustomerViolatePaymentTerms));
      return;
    }
    if (this.order.outlet.customerValidations.isExceedsCustomerLimit === true) {
      // Customer Violte payment terms
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgCustomerExceedsCreditLimit));
      return;
    }
    this.fillGeneralFilterData();
    this.tabsArray.fill(false);
    this.tabsArray[++this.selectedTabIndex] = true;
  }

  calculateTotalAfterPromotions() {
    // reset totals
    this.totals = new TotalsModel();
    // calculate items totals after take benefits
    this.calculationService.calculateTotalAfterPromotions(this.totals, this.order.allItems);
    // set transaction Totals
    this.order.netTotal = this.totals.netTotal;
    this.order.grossTotal = this.totals.grossTotal;
    this.order.tax = this.totals.taxTotal;
    this.order.calculatedRetailTax = this.totals.retailTaxTotal;
    this.order.discount = this.totals.discountTotal;
    this.order.promotedDiscount = this.totals.promotedDiscountTotal;

    this.itemFormProperties.addedItemList = this.order.soldItems;
    this.itemFormProperties.promotedItemList = this.order.promotedItems;
  }
  calculateTotalAfterCancelPromotions() {
    // reset totals
    this.totals = new TotalsModel();
    this.order.promotedItems = [];
    this.order.allItems = [];
    this.order.allItems = this.order.allItems.concat(this.order.soldItems);
    // calculate items totals after take benefits
    this.calculationService.calculateTotalAfterCancelPromotions(this.totals, this.order.allItems);
    this.itemFormProperties.addedItemList = this.order.soldItems;
    this.itemFormProperties.promotedItemList = this.order.promotrdItems;
    // set transaction Totals
    this.order.netTotal = this.totals.netTotal;
    this.order.grossTotal = this.totals.grossTotal;
    this.order.tax = this.totals.taxTotal;
    this.order.calculatedRetailTax = this.totals.retailTaxTotal;
    this.order.discount = this.totals.discountTotal;
    this.order.promotedDiscount = this.totals.promotedDiscountTotal;

    this.customerPromotions = [];
    this.order.customerPromotions = [];
    this.transactionCommonData.customerPromotions = [];
    this.transactionCommonData.promotionsForAllLevels = [];
    this.order.promotionsForAllLevels = [];
  }


  //#endregion

  //#region [GETTERS]
  get headerData() {
    return this.orderForm.get('headerData');
  }
  //#endregion

  onRecurringOrder() {
    this.showSlider = true;
  }
  fillRecurringData() {
    this.recurringObj = new Object();
    this.recurringObj.typeId = this.order.recurringTypeId
    this.recurringObj.recurringValues = this.order.recurringValue
    this.recurringObj.isActive = this.order.isRecurringActive
  }
  closeSlider() {
    this.showSlider = false;
  }
  onSaveClick() {
    this.saveSubject.next()
  }
  onItemSaved($event) {
    this.order.recurringTypeId = $event.typeId;
    this.order.recurringValue = $event.recurringValues;
    this.order.isRecurringActive = $event.isActive;
    this.fillRecurringData();
    this.closeSlider();
  }
}
