import { HttpErrorResponse } from '@angular/common/http';
import { Component, OnInit } from '@angular/core';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../core/core.session';
import { ConstantURLs } from '../../../shared/models/constants/constant-URL';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { FormProcessMode } from '../../../shared/models/enums/form-process-mode.enum';
import { MenuActions } from '../../../shared/models/enums/menu-actions-enum';
import { OrderOperation } from '../../../shared/models/enums/order-operation .enum';
import { OrderStatus } from '../../../shared/models/enums/order-status.enum';
import { RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { IOperationSummary } from '../../../shared/models/operation-summary/operation-summary-interface';
import { OrderListFilter } from '../../../shared/models/shared-table/custom-list.interface';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { OrderCancelledService } from '../../../shared/services/order/order-cancelled.service';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { ReportMode } from '../../../shared/models/enums/report-mode.enum';
import { PdfCreatorService } from '../../../shared/services/pdf-creator.service';
import { NavBarService } from '../../../core/sidebar/sidebar.service';

@Component({
  selector: 'app-cancel-order',
  templateUrl: './cancel-order.component.html',
  styleUrls: ['./cancel-order.component.css']
})
export class CancelOrderComponent implements OnInit {

  orderId = "";
  selectedOrder: any;

  tableData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    showSearch: true,
    isOnline: true,
    showSearchBtn: true,
    showNewBtn: false,
    showExport: true,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      {
        title: "Desc_Order_ID",
        key: "orderId",
        isSortable: true,
        width: "14%"
      },
      {
        title: "Desc_Customer",
        key: "customerName",
        isSortable: false,
        width: "14%"
      },
      {
        title: "Desc_Outlet",
        key: "outletName",
        isSortable: false,
        width: "14%"
      },
      {
        title: "Desc_Employee_Name",
        key: "employeeName",
        isSortable: false,
        width: "14%"
      },
      {
        title: "Desc_Order_Date",
        key: "orderDate",
        isSortable: true,
        width: "14%",
        isDate: true
      },
      {
        title: "Desc_Order_Type",
        key: "orderType",
        isSortable: false,
        width: "10%"
      },
      {
        title: "Desc_Note",
        key: "note",
        isSortable: false,
        width: "18%"
      },
    ]
  };

  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  query: OrderListFilter = {
    customListFilter: {
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    orderOperation: OrderOperation.Cancel,
    orderStatus: OrderStatus.Canceled,
    customerId: -1,
    outletId: -1,
    employeeId: -1
  };

  operationSummaryData: IOperationSummary = {
    operationData: {},
    formProcessMode: FormProcessMode.CancelledOrder,
    formTitle: "Desc_Order_Cancel_Summary"
  };
  showViewSummarySlider = false;

  isEditMode = false;
  showOrderSlider = false;
  orderSignature = "";

  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private orderCancelledService: OrderCancelledService,
    private sessionData: SessionDataProvider,
    private pdfCreatorService: PdfCreatorService,
    private navService : NavBarService
  ) { }
  showSharedTable: boolean = false

  ngOnInit() {
    this.navService.endtour.subscribe(() => {
      // Call the onTourEnd() function or perform any action
      this.navService.tourStatus = true;
      this.showSharedTable = this.navService.tourStatus}
    )

      this.showSharedTable = this.navService.tourStatus;
    this.coreSession.SetTitle("Cancelled Orders");
    if (this.isDivisionLevel())
    this.tableData.columns.push({ title: "Desc_Division", key: "division", isSortable: true, width: "15%" });
    this.populateNewOrderList();
    this.addActions();
  }

  isDivisionLevel() {
    return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true")
  }

  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.cancelledOrdersURL, MenuActions.View)) {
      this.tableData.rowOperations.push({
        operation: RowOperation.View,
        title: 'Desc_View',
        icon: 'fa-regular fa-eye',
        color: '#12344d'
      });
    }

    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.cancelledOrdersURL, MenuActions.Print)) {
      this.tableData.rowOperations.push({
        operation: RowOperation.Print,
        title: 'Desc_Print',
        icon: 'fa-solid fa-print',
        color: '#12344d'
      });
    }
  }

  populateNewOrderList() {
    this.coreSession.ModalLoading.Show();
    this.orderCancelledService.getOrders(this.query).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.dataSource = <SharedTableResult>response.data;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  onTableFilterChanged(query: any) {
    this.query.customListFilter = query;
    this.populateNewOrderList();
  }

  onFindButtonClicked(result) {
    this.query.customListFilter.page = 0;
    this.query.customerId = result.customerId;
    this.query.outletId = result.outletId;
    this.query.employeeId = result.employeeId;
    this.query.orderDateRange = result.transactionDate;
    if (this.isDivisionLevel())
      this.query.divisionId = result.divisionId;
    else
      this.query.divisionId = -1;
    this.populateNewOrderList();
  }

  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.View:
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.cancelledOrdersURL, MenuActions.View)) {
          this.viewTransaction(event.object.orderSignature);
        }
        break;
      case RowOperation.Print:
        this.exportOrderReport(event);
        break;
    }
  }

  closeViewSummarySlider() {
    this.selectedOrder = null;
    this.operationSummaryData = null;
    this.showViewSummarySlider = false;
  }

  viewTransaction(signature: string) {
    this.coreSession.ModalLoading.Show();
    let query: OrderListFilter = {};
    query.orderSignature = signature;
    query.isViewMode = true;
    query.orderStatus = OrderStatus.Canceled;
    query.orderOperation = OrderOperation.Cancel;
    this.orderCancelledService.getOrder(query).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.selectedOrder = response.data;
          this.operationSummaryData = {
            operationData: this.selectedOrder,
            formProcessMode: FormProcessMode.CancelledOrder,
            formTitle: "Desc_Order_Cancel_Summary"
          };
          this.showViewSummarySlider = true;
        } else {
          this.coreSession.showWarrning(
            this.translateService.instant(ConstantMessages.WarningCaption),
            response.message
          );
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  exportOrderReport(event: any) {
    let trans = {
      divisionId: event.object.divisionId ? event.object.divisionId : -1,
      transactionId: event.object.orderId,
      reportMode: ReportMode.SalesOrder,
      customerId: event.object.customerId ? event.object.customerId : -1,
      outletId: event.object.outletId ? event.object.outletId : -1,
      isReprintReport: true
    }
    this.pdfCreatorService.ExportReport(trans);
  }

}
