import { Component, OnInit, ViewChild } from "@angular/core";
import {
  CdkDragDrop,
  moveItemInArray,
  transferArrayItem
} from "@angular/cdk/drag-drop";
import { FormGroup, FormControl, Validators } from "@angular/forms";
import { OrderListFilter } from "../../../shared/models/shared-table/custom-list.interface";
import { OrderStatus } from "../../../shared/models/enums/order-status.enum";
import { OrderService } from "../order.service";
import { CoreSession } from "../../../core/core.session";
import { DialogMode } from "../../../shared/models/enums/dialog-mode.enum";
import { HttpErrorResponse } from "@angular/common/http";
import { ConstantMessages } from "../../../shared/models/constants/constant-message";
import {
  NgbModal,
  NgbModalConfig,
  NgbModalRef
} from "@ng-bootstrap/ng-bootstrap";
import { IDateTimePickerProperties } from "../../../shared/models/dateTimeModel/date-time-properties.interface";
import { DateTimeModel } from "../../../shared/models/dateTimeModel/date-time.model";
import { OrderOperation } from "../../../shared/models/enums/order-operation .enum";
import { TranslateService } from "@ngx-translate/core";
import { SessionDataProvider } from "../../../core/session-data-provider.service";
import { ConstantConfigurations } from "../../../shared/models/constants/constant-configuration";
import { DeliveryModes } from '../../../shared/models/enums/delivery-modes.enum';

@Component({
  selector: "app-dispatch-order",
  templateUrl: "./dispatch-order.component.html",
  styleUrls: ["./dispatch-order.component.css"]
})
export class DispatchOrderComponent implements OnInit {
  @ViewChild('mainFilter', { static: true }) mainFilter: NgbModal;
  modalRef: NgbModalRef;
  resetForms = false;
  approvedOrdersForm: FormGroup;
  assignedOrderForm: FormGroup;
  reschduleForm: FormGroup;

  event: CdkDragDrop<any>;
  selectedOrder: any;
  selectedDriverId = null;

  selectedSchdualeDateModel: DateTimeModel;

  approvedQuery: OrderListFilter = {
    orderOperation: OrderOperation.GetApproved,
    orderStatus: OrderStatus.Approved
  };
  assignedQuery: OrderListFilter = {
    orderOperation: OrderOperation.GetAssigned
  };

  deliveryDateProperties: IDateTimePickerProperties = {
    label: "Default Deleviry Date",
    formControlName: "deliveryDate",
    isCalendarOnly: true
  };

  schduleDateProperties: IDateTimePickerProperties = {
    label: "Reschdule Delivery Date",
    formControlName: "schduleDate",
    min: new Date()
  };
  approvedOrders: any[] = [];
  assignedOrders: any[] = [];

  showWarehouseSelection = false;
  constructor(
    private orderService: OrderService,
    private coreSession: CoreSession,
    private modalService: NgbModal,
    private translateService: TranslateService,
    private sessionData: SessionDataProvider
  ) {}

  ngOnInit() {
    this.coreSession.SetTitle("Dispatch");
    this.initOrderForm();
    this.initAssignedOrderForm();
    this.initReschduleForm();
    this.showWarehouseSelection =
    this.sessionData
    .getConfigurationValue(ConstantConfigurations.ForceLoadRequestForOrders)
    .toLowerCase() === "true" && (
      (
        this.sessionData
      .getConfigurationValue(ConstantConfigurations.AffectVanStockInDeliveryInFO)
      .toLowerCase() === "true" && this.sessionData
      .getConfigurationValue(ConstantConfigurations.DeliveryMode)
      .toLowerCase().includes(DeliveryModes.BOInvoicingFODelivery.valueOf()+"")
      )
      || this.sessionData
      .getConfigurationValue(ConstantConfigurations.DeliveryMode)
      .toLowerCase().includes(DeliveryModes.FOInvoicingAndDelivery.valueOf()+"")
      );
  }

  initOrderForm() {
    this.approvedOrdersForm = new FormGroup({
      customerData: new FormGroup({
        customerId: new FormControl(),
        outletId: new FormControl()
      }),
      employeeData: new FormGroup({
        employeeId: new FormControl()
      })
    });
  }

  initAssignedOrderForm() {
    this.assignedOrderForm = new FormGroup({
      employeeData: new FormGroup({
        employeeId: new FormControl()
      })
    });
  }

  initReschduleForm() {
    this.reschduleForm = new FormGroup({
      deliveryDate: new FormControl({
        value: null,
        disabled: true
      }),
      schduleDate: new FormControl(),
      warehouseId: new FormControl({ value: null }, Validators.required)
    });
  }

  dropToAssigned(event: CdkDragDrop<any>) {
    if (event.previousContainer === event.container) {
      // Sorting
      this.moveItemInArray(event);
    } else {
      // Move
      if (this.selectedDriverId) {
        this.event = event;
        this.selectedOrder = event.item.data;
        this.deliveryDateProperties.dateValue = this.selectedOrder.desiredDeliveryDateModel;

        this.selectedSchdualeDateModel = null;
        this.selectedOrder.driverId = this.selectedDriverId;
        this.isOrderValidToAssign();
      } else {
        this.coreSession.showWarrning(
          this.translateService.instant(ConstantMessages.WarningCaption),
          this.translateService.instant(ConstantMessages.Desc_Select_Driver)
        );
      }
    }
  }

  dropToApproved(event: CdkDragDrop<any>) {
    if (event.previousContainer === event.container) {
      // Sorting
      this.moveItemInArray(event);
    } else {
      // Move
      this.event = event;
      this.selectedOrder = event.item.data;
      this.selectedOrder.driverId = this.selectedDriverId;
      this.removeOrderAssignment();
    }
  }

  moveItemInArray(event: CdkDragDrop<any>) {
    moveItemInArray(
      event.container.data,
      event.previousIndex,
      event.currentIndex
    );
  }

  transferArrayItem(event: CdkDragDrop<any>) {
    transferArrayItem(
      event.previousContainer.data,
      event.container.data,
      event.previousIndex,
      event.currentIndex
    );
  }

  findApprovedOrders() {
    this.approvedQuery.customerId = this.approvedOrdersForm.value.customerData.customerId;
    this.approvedQuery.outletId = this.approvedOrdersForm.value.customerData.outletId;
    this.approvedQuery.employeeId = this.approvedOrdersForm.value.employeeData.employeeId;
    this.approvedQuery.getAll = true;
    this.populateNewOrderList();
  }

  findAssignedOrders() {
    this.selectedDriverId = this.assignedOrderForm.value.employeeData.employeeId;
    if (this.selectedDriverId) {
      this.assignedQuery.employeeId = this.selectedDriverId;
      this.assignedQuery.getAll = true;
      this.populateAssignedOrderList();
    } else {
      this.coreSession.showWarrning(
        this.translateService.instant(ConstantMessages.WarningCaption),
        this.translateService.instant(ConstantMessages.Desc_Select_Driver)
      );
    }
  }

  resetApprovedOrdersForm() {
    this.approvedOrdersForm.reset();
    this.resetForms = true;
    this.approvedOrders = null;
  }

  resetAssignedOrdersForm() {
    this.assignedOrderForm.reset();
    this.selectedDriverId = null;
    this.assignedOrders = null;
  }

  populateNewOrderList() {
    this.coreSession.ModalLoading.Show();
    this.orderService.getOrders(this.approvedQuery).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.approvedOrders = response.data.data;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  populateAssignedOrderList() {
    this.coreSession.ModalLoading.Show();
    this.orderService.getOrders(this.assignedQuery).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.assignedOrders = response.data.data;
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  isOrderValidToAssign() {
    this.coreSession.ModalLoading.Show();
    this.orderService.isOrderValidToAssign(this.selectedOrder).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.selectedOrder = response.data;
          this.modalRef = this.modalService.open(this.mainFilter, {
            centered: true
          });
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  assignOrderToDriver() {
    this.coreSession.ModalLoading.Show();
    this.orderService.assignOrderToDriver(this.selectedOrder).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(
            this.translateService.instant(ConstantMessages.SuccessCaption),
            this.translateService.instant(ConstantMessages.OrderDispatched)
          );
          this.transferArrayItem(this.event);
          // this.populateNewOrderList();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  removeOrderAssignment() {
    this.coreSession.ModalLoading.Show();
    var orderList = [];
    orderList.push(this.selectedOrder);
    this.orderService.removeOrderAssignment(orderList).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.coreSession.showSuccess(
            this.translateService.instant(ConstantMessages.SuccessCaption),
            this.translateService.instant(ConstantMessages.Remove)
          );
          this.transferArrayItem(this.event);
          // this.populateNewOrderList();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  onReschduleClicked() {
    if (!this.selectedSchdualeDateModel) {
      this.coreSession.showError(
        this.translateService.instant(ConstantMessages.ErrorCaption),
        this.translateService.instant(ConstantMessages.SelectSchedule)
      );
      return;
    }

    this.onCloseModal();
    this.selectedOrder.referenceWarehouseId = this.reschduleForm.get('warehouseId').value;
    this.selectedOrder.scheduleDateModel = this.selectedSchdualeDateModel;
    this.assignOrderToDriver();
  }

  onKeepDefualtDataClicked() {
    if (this.event.item.data.desiredDeliveryDateModel.date < Date.now) {
      this.coreSession.showError(
        this.translateService.instant(ConstantMessages.ErrorCaption),
        this.translateService.instant(
          ConstantMessages.DeliveryDateCantBeLessThanAssignment
        )
      );
      return;
    }
    this.onCloseModal();
    this.selectedOrder.referenceWarehouseId = this.reschduleForm.get('warehouseId').value;
    this.selectedOrder.scheduleDateModel = this.selectedOrder.desiredDeliveryDateModel;
    this.assignOrderToDriver();
  }

  onCloseModal() {
    this.modalRef.dismiss();
    this.reschduleForm.reset();
  }

  onSchduleDateSelected(event: any) {
    this.selectedSchdualeDateModel = event;
  }
}
