import { Component, OnInit } from "@angular/core";
import { OrderService } from "../order.service";
import { CoreSession } from "../../../core/core.session";
import { TranslateService } from "@ngx-translate/core";
import { DialogMode } from "../../../shared/models/enums/dialog-mode.enum";
import { ConstantMessages } from "../../../shared/models/constants/constant-message";

@Component({
  selector: "app-order-dashboard",
  templateUrl: "./order-dashboard.component.html",
  styleUrls: ["./order-dashboard.component.css"]
})
export class OrderDashboardComponent implements OnInit {
  isToday = false;
  isMonthly = true;
  isAnnual = false;

  pieOptions: any = {};
  barOptions: any = {};

  pieOptionsData: any = {};
  barOptionsData: any = {};

  dashboardsData: any = {};

  overAllOrders: any = {};
  allPendingOrders: any = {};
  readyToDelivery: any = {};
  deliveredOrders: any = {};

  constructor(
    private orderService: OrderService,
    private coreSession: CoreSession,
    private translateService: TranslateService
  ) {}

  ngOnInit() {
    this.initBarOptions();
    this.initPieOptions();
    this.populateDashboardsData();
  }

  populateDashboardsData() {
    this.coreSession.ModalLoading.Show();
    this.orderService
      .getDashboardData(this.isToday, this.isMonthly, this.isAnnual)
      .subscribe(
        response => {
          this.coreSession.ModalLoading.Hide();
          if (response.status != null && response.status >= 0) {
            this.fillDashboardsData(response.data);
          } else {
            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
          }
        },
        error => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
        }
      );
  }

  onTodayClick() {
    this.isToday = true;
    this.isMonthly = false;
    this.isAnnual = false;
    this.populateDashboardsData();
  }

  onMonthlyClick() {
    this.isToday = false;
    this.isMonthly = true;
    this.isAnnual = false;
    this.populateDashboardsData();
  }

  onAnnualClick() {
    this.isToday = false;
    this.isMonthly = false;
    this.isAnnual = true;
    this.populateDashboardsData();
  }

  initBarOptions() {
    this.barOptions = {
      backgroundColor: "#FFF",
      color: ["#3398DB"],
      title: {
        text: this.translateService.instant('Desc_Top_Items_Order'),
        left: "center",
        top: 20,
        textStyle: {
          color: "#2b3643"
        }
      },
      tooltip: {
        trigger: "axis",
        axisPointer: {
          type: "shadow"
        }
      },
      grid: {
        left: "3%",
        right: "4%",
        bottom: "3%",
        containLabel: true
      },
      xAxis: [
        {
          type: "category",
          
          axisTick: {
            alignWithLabel: true
          }
        }
      ],
      yAxis: [
        {
          type: "value"
        }
      ],
      series: [
        {
          name: "Quantity",
          type: "bar",
          barWidth: "60%",
        }
      ],
      toolbox: {
        show: true,
        showTitle: false, // hide the default text so they don't overlap each other
        feature: {
          saveAsImage: {
            show: true,
            title: "Save As Image"
          }
        }
      }
    };
  }

  initPieOptions() {
    this.pieOptions = {
      backgroundColor: "#FFF",
      title: {
        text: this.translateService.instant('Desc_Order_Per_Status'),
        left: "center",
        top: 20,
        textStyle: {
          color: "#2b3643"
        }
      },

      tooltip: {
        trigger: "item",
        formatter: "{a} <br/>{b} : {c} ({d}%)"
      },
      series: [
        {
          name: "Orders",
          type: "pie",
          radius: "55%",
          roseType: "area",
          center: ["50%", "50%"],
          // data: [
          //   { value: 335, name: "C-1" },
          //   { value: 310, name: "C-2" },
          //   { value: 274, name: "C-3" },
          //   { value: 235, name: "C-4" },
          //   { value: 400, name: "C-5" }
          // ],
          label: {
            fontSize: 15
          },
          // labelLine: {
          //   normal: {
          //     lineStyle: {
          //       color: 'rgba(255, 255, 255, 0.3)'
          //     },
          //     smooth: 0.2,
          //     length: 10,
          //     length2: 20
          //   }
          // },
          itemStyle: {
            emphasis: {
              shadowBlur: 10,
              shadowOffsetX: 0,
              shadowColor: "rgba(0, 0, 0, 0.5)"
            }
          },

          animationType: "scale",
          animationEasing: "elasticOut",
          animationDelay: function(idx) {
            return Math.random() * 200;
          }
        }
      ]
    };
  }

  fillDashboardsData(data) {

    this.overAllOrders = data.overAllOrders;
    this.allPendingOrders = data.currentPending;
    this.readyToDelivery = data.readyToDelivery;
    this.deliveredOrders = data.deliveredOrders;
    // update series data:
    this.barOptionsData = {
      xAxis:[
        {
          data: data.topItems.xAxis,
        }
      ],
      series: [
        {
          data: data.topItems.yAxis
        }
      ]
    };

    this.pieOptionsData = {
      series: [
        {
          data: data.orderByStatus
        }
      ]
    }
  }
}
