import { Component, OnInit, ViewChild, OnDestroy } from '@angular/core';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { CoreSession } from '../../../core/core.session';
import { IncidentService } from '../incident.service';
import { TranslateService } from '@ngx-translate/core';
import { NgbModal, NgbModalConfig } from '@ng-bootstrap/ng-bootstrap';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantURLs } from '../../../shared/models/constants/constant-URL';
import { MenuActions } from '../../../shared/models/enums/menu-actions-enum';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { Subject } from 'rxjs';
import { RecallService } from '../recall.service';
import { RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { IncidentStatusEnum } from '../../../shared/models/enums/incident-status-enum';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';




@Component({
  selector: 'app-incidents-list',
  templateUrl: './incidents-list.component.html',
  styleUrls: ['./incidents-list.component.css']
})
export class IncidentsListComponent implements OnInit {

  saveSubject: Subject<void> = new Subject<void>();
  showIncidentEntrySlider = false;
  isEditMode = false;
  isViewMode = false;
  OkResult: DialogResult = DialogResult.Ok;
  currentIncident: any = null;

  TableData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    showSearch: true,
    isOnline: true,
    showSearchBtn: false,
    showNewBtn: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      {
        title: "Desc_Incident",
        key: "incidentName",
        isSortable: true,
        width: "33%"
      },
      {
        title: "Desc_Item",
        key: "itemCodeName",
        isSortable: true,
        width: "33%"
      },
      {
        title: "Desc_Date",
        key: "createdDate",
        isSortable: true,
        width: "33%",
        isDate: true
      }
    ]
  };

  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  incidentFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    dateRange: DateTimeModel,
    ItemId: -1,
    employeeId: -1,
    incidentStatus: -1
  };

  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private incidentService: IncidentService) { }

  ngOnInit() {
    this.coreSession.SetTitle('Incidents');
    this.TableData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.incidentsListURL, MenuActions.Add);
    this.populateIncidentList();
    this.addActions();
  }

  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.incidentsListURL, MenuActions.Edit)) {
      this.TableData.rowOperations.push({
        operation: RowOperation.edit,
        title: "Desc_Edit",
        icon: "fa fa-pencil",
        color: "#12344d"
      });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.incidentsListURL, MenuActions.Delete)) {
      this.TableData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Cancel',
          icon: 'fa fa-close',
          color: '#f1685e'
        });
    }
  }

  populateIncidentList() {
    this.coreSession.ModalLoading.Show();
    this.incidentService.getIncidentListSharedTable(this.incidentFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data
        };
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }

  openNewIncidentScreen() {
    this.isEditMode = false;
    this.isViewMode = false;
    this.showIncidentEntrySlider = true;
  }

  onSaveIncidentClick() {
    this.saveSubject.next();
  }

  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.edit:
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.incidentsListURL, MenuActions.Edit)) {
          if (event.object && event.object != null) {
            this.openEditIncidentScreen(event.object);
          }
        }
        break;
      case RowOperation.delete:
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.incidentsListURL, MenuActions.Delete)) {
          if (event.object && event.object != null) {
            if (event.object.status === 1) {
              this.updateIncidentStatus(event.object.incidentId, IncidentStatusEnum.canceled);
            } else {

            }
          }
        }
        break;
    }
  }

  onTableFilterChanged(query: any) {
    this.incidentFilter.customListFilter = query;
    this.populateIncidentList();
  }

  closeIncidentEntrySlider() {
    this.currentIncident = null;
    this.isEditMode = false;
    this.isViewMode = false;
    this.showIncidentEntrySlider = false;
    this.coreSession.SetTitle('Incidents');
  }

  afterSaveIncident() {
    this.closeIncidentEntrySlider();
    this.populateIncidentList();
  }

  openEditIncidentScreen(selectedIncident: any) {
    this.currentIncident = selectedIncident;
    this.isEditMode = true;
    this.isViewMode = false;
    this.showIncidentEntrySlider = true;
  }

  updateIncidentStatus(incidentId: any, newStatus: any) {
    var msg = '';
    if (newStatus === IncidentStatusEnum.canceled) {
      msg = ConstantMessages.MsgCancelConfirmation;
    }
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(msg), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          var filter = {
            incidentId: incidentId,
            incidentStatus: newStatus
          }
          this.incidentService.updateIncidentStatus(filter).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
              this.populateIncidentList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          },
            (error: HttpErrorResponse) => {
              this.coreSession.ModalLoading.Hide();
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            });
        }
      }
    );
  }
}
