import { Component, OnInit, OnDestroy, Output, Input, EventEmitter, ViewChild } from '@angular/core';
import { Observable } from 'rxjs';
import { ITextInput } from '../../../shared/models/shared-table/custom-list.interface';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { RowOperation, MultiSelectionOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { CoreSession } from '../../../core/core.session';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { TranslateService } from '@ngx-translate/core';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';
import { HttpErrorResponse } from '@angular/common/http';
import * as cloneDeep from 'lodash/cloneDeep';
import { RecallTaskStatus } from '../../../shared/models/enums/recall-task-status.enum';
import { RecallService } from '../recall.service';
import { RecallInternalPartyTypes, RecallInternalOperationTypes } from '../../../shared/models/enums/recall-internal-party-types.enum';
import { RecallPartyTypes } from '../../../shared/models/enums/recall-party-types.enum';
import { RecallNotificationTypes } from '../../../shared/models/enums/recall-notification-types.enum';
import { IDateTimePickerProperties } from '../../../shared/models/dateTimeModel/date-time-properties.interface';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { DatePipe } from '@angular/common';
import { MasterDataService } from '../../master-data-definition/master-data.service';
import { SerialStatus } from '../../../shared/models/enums/SerialStatus';
import { NgbModalConfig, NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { CalculationService } from '../../../shared/services/calculations.service';
import { NotificationService } from '../../../core/notification.service';
import { NotificationTypes } from '../../../shared/models/enums/notification-Types.enum';
import { EmailService } from '../../admin-settings/email/email.service';
import { EmailTypes } from '../../../shared/models/enums/email-types.enum';

@Component({
  selector: 'app-recall-entry',
  templateUrl: './recall-entry.component.html',
  styleUrls: ['./recall-entry.component.css'],
  providers: [DatePipe]
})
export class RecallEntryComponent implements OnInit, OnDestroy {

  @Output() onSavingRecallEvent = new EventEmitter();
  @Input() saveEvent: Observable<void>;
  @Input() recallObject: any = null;
  @Input() isEditMode: boolean = false;
  @Input() isViewMode: boolean = false;
  @Input() isCloseMode: boolean = false;
  saveBtnSubscription: any;
  navLinks: any[];
  showDefinitionData = false;
  showInvolvedParties = false;
  showSubTasks = false;
  recallForm: FormGroup;
  recallTaskId = -1;
  recallStatusList: any[] = [];
  isStatusLoading = false;
  otherRecallLanguages: any[] = [];
  recallNameInput: ITextInput = {
    formControlName: 'recallName',
    placeHolder: 'Desc_Recall_Name',
    label: 'Desc_Recall_Name'
  };
  tabSections = {
    definitionData: 0,
    involvedparties: 1,
    subTasks: 2
  }

  // Items
  @ViewChild("itemLocationsDialog", { static: true }) itemLocationsDialog: NgbModal;
  itemLocationsList: any[] = [];
  expiryDate: DateTimeModel = new DateTimeModel();
  batchNo: string = '';
  filterBy = 1;
  filteredItemsList: any[] = [];
  addedItemsList: any[] = [];
  isCollapsedItems = false;
  itemsDropdownSettings = {
    singleSelection: false,
    idField: 'itemId',
    textField: 'itemCodeName',
    selectAllText: '',
    unSelectAllText: '',
    searchPlaceholderText: '',
    allowSearchFilter: true,
    itemsShowLimit: 1
  };
  itemsSharedTabel: ITableProperties = {
    pageSize: 100,
    showPaginator: false,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Item', key: 'itemCodeName', isSortable: true, width: '40%' },
      { title: 'Desc_ExpiryDate', key: 'expiryDate', isSortable: true, isExpiryDate: true, width: '20%' },
      { title: 'Desc_BatchNo', key: 'batchNo', isSortable: false, width: '20%' },
      { title: 'Desc_Serial_Number', key: 'serialNo', isSortable: false, width: '20%' },
    ]
  };
  itemsDataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  expiryDateProperties: IDateTimePickerProperties = {
    label: 'Desc_ExpiryDate',
    formControlName: 'expiryDate',
    isCalendarOnly: true

  };
  itemsFilter = {
    ByBatch: 1,
    BySerial: 2
  }

  //Parties
  allPartyList: any[] = [];
  isPartyLoading = false;
  notificationTypesList: any[] = [];
  emailTemplateList: any[] = [];
  notificationTemplateList: any[] = [];
  addedPartyList: any[] = [];
  invlolvedDataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  invlolvedSharedTabel: ITableProperties = {
    pageSize: 100,
    showPaginator: false,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Involved_Party', key: 'partyCodeName', isSortable: true, width: '40%' },
      // { title: 'Desc_Type', key: 'partyTypeName', isSortable: true, width: '25%' },
      { title: 'Desc_Notification_Type', key: 'notificationTypeName', isSortable: true, width: '30%' },
      { title: 'Desc_Notification_Template', key: 'notificationTemplateName', isSortable: true, width: '30%' },
    ]
  };
  selectedParty = null;
  selectedNotification = null;
  selectedTemplate = null;

  ControlTransactionDivisionLevel = false;

  recallMisMatchReasonList: any[] = [];
  isMisMatchReasonLoading = false;
  subTasksList: any[] = [];
  subTasksDataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  subTasksSharedTabel: ITableProperties = {
    pageSize: 100,
    showPaginator: true,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    isMultiSelection: false,
    rowOperations: [{
      operation: RowOperation.View,
      title: 'Desc_View_Items',
      icon: "fa fa-eye",
      color: '#12344d'
    }],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Operation_Type', key: 'recallOperationName', isSortable: true, width: '20%' },
      { title: 'Desc_Transaction_ID', key: 'transactionId', isSortable: true, width: '20%' },
      { title: 'Desc_Location', key: 'partyCodeName', isSortable: true, width: '20%' },
      { title: 'Desc_Executed_By', key: 'employeeCodeName', isSortable: true, width: '20%' },
    ]
  };

  subTaskItemsDataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  subTaskItemsSharedTabel: ITableProperties = {
    pageSize: 100,
    showPaginator: true,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    isMultiSelection: false,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Item', key: 'itemCodeName', isSortable: true, width: '16%' },
      { title: 'Desc_UOM', key: 'uom', isSortable: true, width: '10%' },
      { title: 'Desc_BatchNo', key: 'batchNo', isSortable: true, width: '15%' },
      { title: 'Desc_ExpiryDate', key: 'expiryDate', isSortable: true, width: '16%', isExpiryDate: true },
      { title: 'Desc_Recalled_QTY', key: 'approvedQuantity', isSortable: false, width: '17%', numberOfStockDigits: true },
      { title: 'Desc_Actual_QTY', key: 'existingQty', isSortable: false, width: '16%', numberOfStockDigits: true, disableInputBasedOnProperty: true, disableInputPropertyName: 'hasSerialNumber' },
      { title: 'Desc_Notes', key: 'notes', isSortable: false, width: '10%' },
    ]
  };
  public canvasWidth = 240;
  chartOptions = {
    hasNeedle: true,
    needleColor: 'gray',
    needleUpdateSpeed: 1000,
    arcColors: ['red', 'orange', 'green'],
    arcDelimiters: [50, 75, 99.9],
    rangeLabel: ['0', '100'],
    needleStartValue: 0
  }
  @ViewChild("subTaskItemsDialog", { static: true }) subTaskItemsDialog: NgbModal;
  seletedSubTask;

  @ViewChild("childSerialsDialog", { static: true }) childSerialsDialog: NgbModal;
  OkResult: DialogResult = DialogResult.Ok;
  childsSerialForPakageOpened: any[] = [];
  selectedOpenedSerials = null;
  constructor(
    public coreSession: CoreSession,
    private translateService: TranslateService,
    private recallService: RecallService,
    private sessionData: SessionDataProvider,
    private masterDataService: MasterDataService,
    private calculationService: CalculationService,
    private datepipe: DatePipe,
    private config: NgbModalConfig,
    private modalService: NgbModal,
    private notificationService: NotificationService,
    private emailService: EmailService) {
  }
  ngOnInit() {
    this.subscribeSaveClick();
    this.prepareNavigation();
    if (this.ControlTransactionDivisionLevel) {
      this.subTasksSharedTabel.columns.push({ title: 'Desc_Division', key: 'divisionCodeName', isSortable: false, width: '20%' });
    }
    this.showEntrySection(this.tabSections.definitionData);
    this.itemsDropdownSettings.selectAllText = this.translateService.instant(ConstantMessages.SelectAllCaption);
    this.itemsDropdownSettings.unSelectAllText = this.translateService.instant(ConstantMessages.UnSelectAllCaption);
    this.itemsDropdownSettings.searchPlaceholderText = this.translateService.instant(ConstantMessages.SearchCaption);
    this.ControlTransactionDivisionLevel = this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true"
    if (this.isEditMode || this.isViewMode || this.isCloseMode) {
      this.coreSession.SetTitle(this.isEditMode ? 'Edit Recall Task' : this.isCloseMode ? 'Close Recall Task' : 'View Recall Task');
      this.recallTaskId = this.recallObject.recallTaskId;
      this.getRecallDataOnEdit();
    } else {
      this.coreSession.SetTitle('Add Recall Task');
      this.recallObject = {
        recallId: -1,
        recallName: null,
        recallCode: null,
        recallReason: null,
        recallStatusId: RecallTaskStatus.Pending,
        isNew: true,
        organizationId: this.coreSession.CurrentOperator.organizationId,
        itemsList: [],
        involvedPartyList: [],
        descriptionLanguages: [],
        recallMisMatchReasonId: null,
        notes: null
      }
    }
    if (this.isCloseMode || this.recallObject.recallStatusId === RecallTaskStatus.Closed) {
      this.fillRecallMisMatchReasonList();
      this.subTasksSharedTabel.columns.push({ title: 'Desc_Match_Ratio', key: 'chartOptions', isSortable: false, isChart: true, chartNeedleKey: 'identicalRatio', width: '20%' });
      if (this.isCloseMode) {
        this.showEntrySection(this.tabSections.subTasks);
      }
    }
    else {
      this.subTasksSharedTabel.columns.push({ title: 'Desc_Status', key: 'recallStatusName', isSortable: true, width: '20%' });
    }
    this.fillRecallStatusList();
    this.initRecallForm();
    if (!this.isViewMode) {
      this.fillAllPartyList();
      this.fillEmailTemplateList();
      this.fillSysNotificationTemplateList();
      this.itemsSharedTabel.rowOperations.push({
        operation: RowOperation.delete,
        title: 'Desc_Delete',
        icon: 'fa fa-close',
        color: '#f1685e',
      }, {
        operation: RowOperation.View,
        title: 'Desc_Item_Locations',
        icon: "fa fa-eye",
        color: '#12344d'
      });
      this.invlolvedSharedTabel.rowOperations.push({
        operation: RowOperation.delete,
        title: 'Desc_Delete',
        icon: 'fa fa-close',
        color: '#f1685e',
      });
    }
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
    if (this.modalService) {
      this.modalService.dismissAll();
    }
  }
  subscribeSaveClick() {
    if (this.isCloseMode) {
      this.saveBtnSubscription = this.saveEvent.subscribe(() => {
        this.onCloseRecallTask();
      });
    }
    else {
      this.saveBtnSubscription = this.saveEvent.subscribe(() => {
        this.onSaveRecall();
      });
    }
  }
  showEntrySection(selectedIndex: number) {
    this.clearSelection();
    if (selectedIndex == this.tabSections.definitionData) {
      this.showDefinitionData = true;
    } else if (selectedIndex == this.tabSections.involvedparties) {
      this.showInvolvedParties = true;
    }
    else if (selectedIndex == this.tabSections.subTasks) {
      this.showSubTasks = true;
    }
    this.navLinks[selectedIndex].isActive = true;
  }
  clearSelection() {
    this.showDefinitionData = false;
    this.showInvolvedParties = false;
    this.showSubTasks = false;
    this.navLinks.forEach(a => a.isActive = false);
  }
  prepareNavigation() {
    this.navLinks = [
      {
        label: 'Desc_Recall_Definition',
        index: this.tabSections.definitionData,
        isActive: false,
        hidden: false
      },
      {
        label: 'Desc_Involved_Parties',
        index: this.tabSections.involvedparties,
        isActive: false,
        hidden: false
      },
      {
        label: 'Desc_Sub_Tasks',
        index: this.tabSections.subTasks,
        isActive: false,
        hidden: !this.isViewMode,
      },
    ];
  }
  fillRecallStatusList() {
    this.isStatusLoading = true;
    this.recallStatusList = [];
    this.recallService.getRecallStatuses().subscribe(result => {
      this.recallStatusList = result;
      this.isStatusLoading = false;
    });
  }
  fillRecallMisMatchReasonList() {
    this.isMisMatchReasonLoading = true;
    this.recallMisMatchReasonList = [];
    this.recallService.getRecallMisMatchReasonList().subscribe(result => {
      this.recallMisMatchReasonList = result;
      this.isMisMatchReasonLoading = false;
    });
  }
  fillAllPartyList() {
    this.isPartyLoading = true;
    this.allPartyList = [];
    this.recallService.getAllPartyList().subscribe(result => {
      this.allPartyList = result;
      this.isPartyLoading = false;
    });
  }
  fillNotificationTypesList(selectedParty) {
    this.notificationTypesList = [];
    if (selectedParty) {
      if (selectedParty.partyTypeId === RecallPartyTypes.InternalParty && (selectedParty.partyId === RecallInternalPartyTypes.SalesPerson || selectedParty.partyId === RecallInternalPartyTypes.StoreKeeper)) {
        this.notificationTypesList.push(
          { notificationTypeName: this.translateService.instant('Desc_System_Notification'), notificationTypeId: RecallNotificationTypes.SystemNotification },
          { notificationTypeName: this.translateService.instant('Desc_Email'), notificationTypeId: RecallNotificationTypes.Email },
        );
      } else {
        var emailType = {
          notificationTypeName: this.translateService.instant('Desc_Email'), notificationTypeId: RecallNotificationTypes.Email
        };
        this.notificationTypesList.push(emailType);
        this.partyDataForm.get('notificationTypeId').setValue(RecallNotificationTypes.Email);
        this.onChangePartyNotification(emailType);
      }
    }
  }
  fillEmailTemplateList() {
    var filter = {
      emailTypeId: EmailTypes.Recall
    }
    this.emailService.getActiveEmailTemplatesList(filter).subscribe(result => {
      this.emailTemplateList = result;
    });
  }
  fillSysNotificationTemplateList() {
    var notificationTypeId = NotificationTypes.Recall.valueOf();
    this.notificationService.GetNotificationTemplateList(notificationTypeId).subscribe(result => {
      this.notificationTemplateList = result;
    });
  }
  initRecallForm() {
    this.recallForm = new FormGroup({
      'headerData': new FormGroup({
        recallName: new FormControl({ value: this.recallObject.recallName, disabled: this.isViewMode }, Validators.required),
        recallCode: new FormControl({ value: this.recallObject.recallCode, disabled: this.isViewMode }, Validators.required),
        recallStatusId: new FormControl({ value: this.recallObject.recallStatusId, disabled: true }, Validators.required),
        recallReason: new FormControl({ value: this.recallObject.recallReason, disabled: this.isViewMode }, Validators.required),
      }),
      'itemData': new FormGroup({
        filterBy: new FormControl(this.itemsFilter.ByBatch, Validators.required),
        itemId: new FormControl(null, Validators.required),
        itemsList: new FormControl([], Validators.required),
        batchNo: new FormControl(null, Validators.required),
        expiryDate: new FormControl(null, Validators.required),
        serialNo: new FormControl(null, Validators.required),
      }),
      'partyData': new FormGroup({
        partyId: new FormControl(null, Validators.required),
        notificationTypeId: new FormControl(null, Validators.required),
        notificationTemplateId: new FormControl(null, Validators.required),
      }),
      'closeData': new FormGroup({
        recallMisMatchReasonId: new FormControl({ value: this.recallObject.recallMisMatchReasonId === -1 ? null : this.recallObject.recallMisMatchReasonId, disabled: this.isViewMode && !this.isCloseMode }),
        notes: new FormControl({ value: this.recallObject.notes, disabled: this.isViewMode && !this.isCloseMode }),
      })
    });
  }
  getRecallDataOnEdit() {
    this.coreSession.ModalLoading.Show();
    var filter = {
      recallId: this.recallObject.recallId,
      recallStatusId: this.recallObject.recallStatusId
    }
    this.recallService.getRecallData(filter).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.recallObject.descriptionLanguages = response.data.descriptionLanguages;
          this.addedPartyList = response.data.involvedPartyList;
          this.addedItemsList = response.data.itemsList;
          this.subTasksList = response.data.subTasksList;
          this.fillItemsList();
          this.fillInvolvedPartyList();
          this.fillSubTasksList();
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  get headerDataForm() {
    return this.recallForm.get('headerData');
  }
  get itemDataForm() {
    return this.recallForm.get('itemData');
  }
  get partyDataForm() {
    return this.recallForm.get('partyData');
  }
  get closeDataForm() {
    return this.recallForm.get('closeData');
  }
  onOtherLanguagesTextSubmitted(event) {
    this.otherRecallLanguages = event;
  }
  onExpiryDateSelected(date) {
    this.batchNo = null;
    this.expiryDate = date;
    let batchNoFormat = this.sessionData.getConfigurationValue(ConstantConfigurations.BatchNumberFormat);
    if (batchNoFormat && batchNoFormat != '' && batchNoFormat != "No Format" && batchNoFormat != "NoFormat") {
      this.batchNo = this.datepipe.transform(this.expiryDate.date, batchNoFormat);
    }
    else {
      this.batchNo = this.datepipe.transform(this.expiryDate.date, this.coreSession.BatchNoFormat);
    }
    this.itemDataForm.get('batchNo').setValue(this.batchNo);
  }


  onRowOperationForItems(event: any) {
    switch (event.operation) {
      case RowOperation.delete:
        this.deleteItem(event.object);
        break;
      case RowOperation.View:
        this.viewItemLocations(event.object);
        break;
    }
  }
  viewItemLocations(item) {
    if (item.recallInternalOperationList && item.recallInternalOperationList.length > 0) {
      this.showItemLocationsnDialog(item);
    }
  }
  showItemLocationsnDialog(item) {
    var MWList = [];
    var vehiclesList = [];
    var customersList = [];
    this.itemLocationsList = [];
    MWList = item.recallInternalOperationList.filter(x => x.recallOperation === RecallInternalOperationTypes.TransferOut);
    vehiclesList = item.recallInternalOperationList.filter(x => x.recallOperation === RecallInternalOperationTypes.VehicleOffload);
    customersList = item.recallInternalOperationList.filter(x => x.recallOperation === RecallInternalOperationTypes.ReturnOrder);

    if (MWList && MWList.length > 0) {
      this.itemLocationsList.push({ recallOperation: RecallInternalOperationTypes.TransferOut, caption: 'Desc_Offloading_Warehouses', isCollapsed: true, partyList: MWList });
    }
    if (vehiclesList && vehiclesList.length > 0) {
      this.itemLocationsList.push({ recallOperation: RecallInternalOperationTypes.VehicleOffload, caption: 'Desc_Offloading_Vehicles', isCollapsed: true, partyList: vehiclesList });
    }
    if (customersList && customersList.length > 0) {
      this.itemLocationsList.push({ recallOperation: RecallInternalOperationTypes.ReturnOrder, caption: 'Desc_Return_From_Customers', isCollapsed: true, partyList: customersList });
    }
    this.modalService.open(this.itemLocationsDialog, { centered: true, size: 'lg' }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
        }
      });
  }
  onFindItemByBatch() {
    this.filteredItemsList = [];
    this.itemDataForm.get('itemId').reset();
    this.itemDataForm.get('itemsList').reset();
    this.itemDataForm.get('itemsList').setValue(null, { emitEvent: false });
    this.itemDataForm.get('itemsList').updateValueAndValidity();

    this.itemDataForm.get('batchNo').markAllAsTouched();
    this.itemDataForm.get('expiryDate').markAllAsTouched();
    if (this.itemDataForm.get('batchNo').invalid || this.itemDataForm.get('expiryDate').invalid) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return false;
    }
    this.getFilteredItems(true);
  }
  onAddItemByBatch() {
    this.itemDataForm.get('batchNo').markAllAsTouched();
    this.itemDataForm.get('expiryDate').markAllAsTouched();
    this.itemDataForm.get('itemsList').markAllAsTouched();
    if (this.itemDataForm.get('batchNo').invalid || this.itemDataForm.get('expiryDate').invalid || this.itemDataForm.get('itemsList').invalid) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return false;
    }
    var notValidItems = [];
    var selectedItems = this.itemDataForm.get('itemsList').value;
    if (selectedItems && selectedItems.length > 0 && this.filteredItemsList && this.filteredItemsList.length > 0) {
      var isAllValid = true;
      for (var i = 0; i < selectedItems.length; i++) {
        var index = this.filteredItemsList.findIndex(item => item.itemId === selectedItems[i].itemId);
        if (index >= 0) {
          var selectedItem = this.filteredItemsList[index];
          var indexInAddedItems = this.addedItemsList.findIndex(x => x.itemId === selectedItem.itemId && x.batchNo === selectedItem.batchNo && x.expiryDate === selectedItem.expiryDate);
          if (indexInAddedItems >= 0) {
            notValidItems.push(selectedItem);
            isAllValid = false;
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgItemAlreadyAdded) + ' : ' + selectedItem.itemCodeName);
          } else {
            this.addedItemsList.push(selectedItem);
            this.filteredItemsList.splice(index, 1);
          }
        }
      }
      if (isAllValid) {
        this.refillItems();
      } else {
        var clonedItems = cloneDeep(this.filteredItemsList);
        this.filteredItemsList = [];
        this.filteredItemsList = clonedItems;
        this.itemDataForm.get('itemsList').setValue(notValidItems);
        this.fillItemsList();
      }
    }
  }
  onFindItemBySerial() {
    this.filteredItemsList = [];
    this.itemDataForm.get('itemId').reset();
    this.itemDataForm.get('itemsList').reset();
    this.itemDataForm.get('itemsList').setValue(null, { emitEvent: false });
    this.itemDataForm.get('itemsList').updateValueAndValidity();

    this.itemDataForm.get('serialNo').markAllAsTouched();
    if (this.itemDataForm.get('serialNo').invalid) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return false;
    }
    this.getFilteredItems(false);
  }
  onAddItemBySerial() {
    this.itemDataForm.get('serialNo').markAllAsTouched();
    this.itemDataForm.get('itemId').markAllAsTouched();
    if (this.itemDataForm.get('serialNo').invalid || this.itemDataForm.get('itemId').invalid) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return false;
    }
    var selectedItemId = this.itemDataForm.get('itemId').value;
    var selectedSerialNo = this.itemDataForm.get('serialNo').value;
    if (selectedItemId && selectedSerialNo) {
      var index = this.filteredItemsList.findIndex(item => item.itemId === selectedItemId && item.serialNo === selectedSerialNo);
      if (index >= 0) {
        var selectedItem = this.filteredItemsList[index];
        var indexInAddedItems = -1;
        indexInAddedItems = this.addedItemsList.findIndex(x => x.itemId === selectedItem.itemId && x.batchNo === selectedItem.batchNo && x.expiryDate === selectedItem.expiryDate && (!x.serialNo || x.serialNo === '' || x.serialNo === selectedItem.serialNo));
        if (indexInAddedItems >= 0) {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgItemAlreadyAdded) + ' : ' + selectedItem.itemCodeName);
          return;
        }
        // Check if parent added before
        indexInAddedItems = -1;
        if (selectedItem.parentsSerialsList && selectedItem.parentsSerialsList.length > 0) {
          for (var parentSerial of selectedItem.parentsSerialsList) {
            indexInAddedItems = this.addedItemsList.findIndex(addedItem => addedItem.serialNo && addedItem.serialNo.toLowerCase() === parentSerial.toString().toLowerCase());
            if (indexInAddedItems >= 0) {
              this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.Desc_One_Or_More_Sub_Serials_Not_Valid) + ' : ' + selectedItem.itemCodeName + " : " + this.addedItemsList[indexInAddedItems].serialNo);
              return;
            }
          }
        }
        // check if child added before
        indexInAddedItems = -1;
        if (selectedItem.childsSerialsList && selectedItem.childsSerialsList.length > 0) {
          for (var childsSerial of selectedItem.childsSerialsList) {
            indexInAddedItems = this.addedItemsList.findIndex(addedItem => addedItem.serialNo && addedItem.serialNo.toLowerCase() === childsSerial.toString().toLowerCase());
            if (indexInAddedItems >= 0) {
              this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.Desc_One_Or_More_Sub_Serials_Not_Valid) + ' : ' + selectedItem.itemCodeName + " : " + this.addedItemsList[indexInAddedItems].serialNo);
              return;
            }
          }
        }
        if (selectedItem.serialStatusId === SerialStatus.PackageOpened) {
          this.showChildSerialsForPackageOpenedDialog(selectedItem);
        } else {
          this.addedItemsList.push(selectedItem);
          this.refillItems();
        }
      }
    }
  }
  showChildSerialsForPackageOpenedDialog(item) {
    this.selectedOpenedSerials = item.serialNo;
    this.childsSerialForPakageOpened = item.nonPackageOpenedChildSerials;
    this.modalService.open(this.childSerialsDialog, { centered: true }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
          var listOfChilds = []
          for (var childSerial of item.nonPackageOpenedChildSerials) {
            this.addedItemsList.push(childSerial);
          }
          this.refillItems();
        }
      });
  }
  getFilteredItems(filterByBatch: boolean) {
    this.coreSession.ModalLoading.Show();
    var filter = {
      serialNo: this.itemDataForm.get('serialNo').value,
      batchNo: this.itemDataForm.get('batchNo').value,
      expiryDateModel: this.expiryDate,
      filterByBatch: filterByBatch,
      filterBySerial: !filterByBatch
    }
    this.recallService.getFilteredItems(filter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        if (response.data.length > 0) {
          this.filteredItemsList = response.data;
          if (!filterByBatch) {
            this.itemDataForm.get('itemId').setValue(this.filteredItemsList[0].itemId);
          }
        } else {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), response.message);
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  refillItems() {
    this.fillItemsList();
    this.clearItemsEntry();
  }
  deleteItem(item) {
    var index = this.addedItemsList.findIndex(x => x.itemId === item.itemId && x.batchNo === item.batchNo && x.expiryDate === item.expiryDate && x.serialNo === item.serialNo);
    if (index >= 0) {
      this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
        (res: DialogResult) => {
          if (res === DialogResult.Yes) {
            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
            this.addedItemsList.splice(index, 1);
            this.fillItemsList();
          }
        });
    }
  }
  fillItemsList() {
    this.itemsDataSource = {
      totalItems: this.addedItemsList.length,
      data: this.addedItemsList
    }
  }
  clearItemsEntry() {
    this.filteredItemsList = [];
    this.itemDataForm.get('batchNo').reset();
    this.itemDataForm.get('expiryDate').reset();
    this.itemDataForm.get('serialNo').reset();
    this.itemDataForm.get('itemId').reset();
    this.itemDataForm.get('itemsList').reset();
    this.itemDataForm.get('itemsList').setValue(null, { emitEvent: false });
    this.itemDataForm.get('itemsList').updateValueAndValidity();
  }
  onChangeFilterBy() {
    this.clearItemsEntry();
  }
  onRowOperationForParties(event: any) {
    switch (event.operation) {
      case RowOperation.delete:
        this.deleteParty(event.object);
        break;
    }
  }
  onChangePartySelection(party) {
    this.selectedParty = party;
    this.selectedNotification = null;
    this.selectedTemplate = null;

    this.partyDataForm.get('notificationTypeId').reset();
    this.partyDataForm.get('notificationTemplateId').reset();
    this.fillNotificationTypesList(this.selectedParty);
  }
  onChangePartyNotification(notification) {
    this.selectedNotification = notification;
    this.selectedTemplate = null;

    this.partyDataForm.get('notificationTemplateId').reset();
  }
  onChangePartyTemplate(template) {
    this.selectedTemplate = template;
  }
  addParty() {
    this.partyDataForm.markAllAsTouched();
    if (this.partyDataForm.invalid || !this.selectedParty || !this.selectedNotification || !this.selectedTemplate) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return false;
    }
    var newParty = {
      partyId: this.selectedParty.partyId,
      partyCodeName: this.selectedParty.partyCodeName,
      partyTypeId: this.selectedParty.partyTypeId,
      partyTypeName: this.selectedParty.partyTypeName,

      notificationTypeId: this.selectedNotification.notificationTypeId,
      notificationTypeName: this.selectedNotification.notificationTypeName,

      notificationTemplateId: this.selectedTemplate.templateId,
      notificationTemplateName: this.selectedTemplate.description,
    };

    var index = this.addedPartyList.findIndex(x => x.partyId === newParty.partyId && x.partyTypeId === newParty.partyTypeId && x.notificationTypeId === newParty.notificationTypeId);
    if (index >= 0) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgPartyAlreadyAdded));
      return false;
    } else {
      this.addedPartyList.push(newParty);
      this.fillInvolvedPartyList();
      this.clearPartyEntry();
    }
  }
  deleteParty(party) {
    var index = this.addedPartyList.findIndex(x => x.partyId === party.partyId && x.partyTypeId === party.partyTypeId && x.notificationTypeId === party.notificationTypeId);
    if (index >= 0) {
      this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
        (res: DialogResult) => {
          if (res === DialogResult.Yes) {
            this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
            this.addedPartyList.splice(index, 1);
            this.fillInvolvedPartyList();
          }
        });
    }
  }
  clearPartyEntry() {
    this.selectedParty = null;
    this.selectedNotification = null;
    this.selectedTemplate = null;
    this.partyDataForm.reset();
  }
  fillInvolvedPartyList() {
    this.invlolvedDataSource = {
      totalItems: this.addedPartyList.length,
      data: this.addedPartyList
    }
  }

  onRowOperationForSubTask(event: any) {
    switch (event.operation) {
      case RowOperation.View:
        this.showSubTaskItems(event.object);
        break;
    }
  }
  fillSubTasksList() {
    if (this.isCloseMode || this.recallObject.recallStatusId === RecallTaskStatus.Closed) {
      this.recallObject.actualValue = 0;
      this.recallObject.targetValue = 0;
      for (let subTask of this.subTasksList) {
        subTask.actualValue = 0;
        subTask.targetValue = 0;
        for (let item of subTask.addedItems) {
          subTask.actualValue += item.existingQty;
          subTask.targetValue += item.approvedQuantity;
        }
        this.recallObject.actualValue += subTask.actualValue;
        this.recallObject.targetValue += subTask.targetValue;

        if (subTask.actualValue <= subTask.targetValue) {
          subTask.identicalRatio = subTask.actualValue / subTask.targetValue * 100;
        } else {
          subTask.identicalRatio = 100;
        }
        subTask.identicalRatio = this.calculationService.getNumberOnDigitFormat(subTask.identicalRatio);
        var subTaskOption = cloneDeep(this.chartOptions);
        subTask.chartOptions = subTaskOption;
      }
      if (this.recallObject.actualValue <= this.recallObject.targetValue) {
        this.recallObject.identicalRatio = this.recallObject.actualValue / this.recallObject.targetValue * 100;
      } else {
        this.recallObject.identicalRatio = 100;
      }
      this.recallObject.identicalRatio = this.calculationService.getNumberOnDigitFormat(this.recallObject.identicalRatio);
      this.recallObject.chartOptions = this.chartOptions;
    }

    this.subTasksDataSource = {
      totalItems: this.subTasksList.length,
      data: this.subTasksList
    }
  }
  showSubTaskItems(subTask) {
    this.seletedSubTask = subTask;
    this.subTaskItemsDataSource = {
      totalItems: this.seletedSubTask.addedItems.length,
      data: this.seletedSubTask.addedItems
    }
    this.modalService.open(this.subTaskItemsDialog, { centered: true, size: 'lg' }).result.then(
      (result) => {

      });
  }
  fillRecallObject() {
    this.recallObject.recallName = this.headerDataForm.get('recallName').value;
    this.recallObject.recallCode = this.headerDataForm.get('recallCode').value;
    this.recallObject.recallReason = this.headerDataForm.get('recallReason').value;
    this.recallObject.recallStatusId = this.headerDataForm.get('recallStatusId').value;
    this.recallObject.itemsList = this.addedItemsList;
    this.recallObject.involvedPartyList = this.addedPartyList;
    this.recallObject.isNew = !this.isEditMode && !this.isViewMode;
    this.recallObject.notes = null;
    this.recallObject.recallMisMatchReasonId = -1;
    let descriptions = this.otherRecallLanguages;
    if (descriptions.findIndex(x => x.languageId === this.coreSession.selectedLanguageId) < 0) {
      descriptions.push({
        languageId: this.coreSession.selectedLanguageId,
        description: this.recallObject.recallName
      });
    }
    this.recallObject.descriptionLanguages = descriptions;
  }
  onSaveRecall() {
    if (this.headerDataForm.invalid) {
      this.coreSession.markFormGroupTouched(this.headerDataForm as FormGroup);
      this.showEntrySection(this.tabSections.definitionData);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    if (!this.addedItemsList || this.addedItemsList.length === 0) {
      this.coreSession.markFormGroupTouched(this.headerDataForm as FormGroup);
      this.showEntrySection(this.tabSections.definitionData);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectItem));
      return;
    }
    if (!this.addedPartyList || this.addedPartyList.length === 0) {
      this.coreSession.markFormGroupTouched(this.headerDataForm as FormGroup);
      this.showEntrySection(this.tabSections.involvedparties);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgLinkRecallToParty));
      return;
    }
    this.fillRecallObject();

    let codeFilter = {
      tableName: 'RecallTask',
      columnName: 'RecallCode',
      insertedValue: this.recallObject.recallCode,
      excludedString: ' And RecallId != ' + this.recallObject.recallId,
    }
    // Validate Code
    if (this.recallObject != null) {
      this.coreSession.ModalLoading.Show();
      this.masterDataService.checkIfUniqueValue(codeFilter).subscribe(response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          if (!response.data) {
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgUniqueCode));
          }
          else {
            this.saveRecallToDB();
          }
        }
        else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      }, (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
    }
  }
  saveRecallToDB() {
    this.coreSession.ModalLoading.Show();
    this.recallService.saveRecallTask(this.recallObject).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSavingRecallEvent.emit();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  onCloseRecallTask() {
    if (this.isCloseMode && this.recallObject.identicalRatio != 100) {
      this.closeDataForm.get('recallMisMatchReasonId').setValidators(Validators.required);
    } else {
      this.closeDataForm.get('recallMisMatchReasonId').setValidators(Validators.nullValidator);
    }
    this.closeDataForm.get('recallMisMatchReasonId').updateValueAndValidity();
    if (this.closeDataForm.invalid) {
      this.coreSession.markFormGroupTouched(this.closeDataForm as FormGroup);
      this.showEntrySection(this.tabSections.subTasks);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    this.recallObject.recallMisMatchReasonId = this.closeDataForm.get('recallMisMatchReasonId').value ? this.closeDataForm.get('recallMisMatchReasonId').value : -1;
    this.recallObject.notes = this.closeDataForm.get('notes').value ? this.closeDataForm.get('notes').value : null;

    this.coreSession.ModalLoading.Show();
    this.recallService.closeRecallTask(this.recallObject).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSavingRecallEvent.emit();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
}
