import { Component, OnInit, ViewChild, OnDestroy } from '@angular/core';
import { Subject } from 'rxjs';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../core/core.session';
import { MenuActions } from '../../../shared/models/enums/menu-actions-enum';
import { ConstantURLs } from '../../../shared/models/constants/constant-URL';
import { RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { RecallService } from '../recall.service';
import { IDateTimePickerProperties } from '../../../shared/models/dateTimeModel/date-time-properties.interface';
import { RecallTaskStatus } from '../../../shared/models/enums/recall-task-status.enum';
import { NgbModal, NgbModalConfig } from '@ng-bootstrap/ng-bootstrap';
import { RecallInternalPartyTypes, RecallInternalOperationTypes } from '../../../shared/models/enums/recall-internal-party-types.enum';

@Component({
  selector: 'app-recall-list',
  templateUrl: './recall-list.component.html',
  styleUrls: ['./recall-list.component.css']
})
export class RecallListComponent implements OnInit, OnDestroy {

  @ViewChild("approveConfirmationDialog", { static: true }) approveConfirmationDialog: NgbModal;
  OkResult: DialogResult = DialogResult.Ok;
  showRecallEntrySlider = false;
  saveSubject: Subject<void> = new Subject<void>();
  recallFilterForm: FormGroup;
  isEditMode = false;
  isViewMode = false;
  isCloseMode = false;
  recallObject: any = null;
  recallStatusList: any[] = [];
  isStatusLoading = false;
  dateRange: DateTimeModel[] = [];

  TabelData: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    openFilterDirectly: true,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Recall_Name', key: 'recallName', isSortable: true, width: '24%' },
      { title: 'Desc_Recall_Code', key: 'recallCode', isSortable: true, width: '23%' },
      { title: 'Desc_Organization', key: 'organizationCodeName', isSortable: true, width: '23%' },
      { title: 'Desc_Status', key: 'recallStatusName', isSortable: true, width: '15%' },
      { title: 'Desc_Date', key: 'createdDate', isSortable: true, isDate: true, width: '15%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  recallFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    dateRange: DateTimeModel,
    recallStatusId: -1
  };
  recallDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Date',
    formControlName: 'dateRange',
    isCalendarOnly: true,
    isRange: true
  };

  operationsToApprove: any[] = [];

  constructor(
    private coreSession: CoreSession,
    private recallService: RecallService,
    private translateService: TranslateService,
    private config: NgbModalConfig,
    private modalService: NgbModal,) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Recalls');
    this.TabelData.showNewBtn = this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Add);
    this.fillRecallStatusList();
    this.initFilterForm();
    this.populateRecallList();
  }
  ngOnDestroy() {
    if (this.modalService) {
      this.modalService.dismissAll();
    }
  }
  addActions() {
    this.TabelData.rowOperations = [];
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Approve) &&
      this.recallFilterForm.get('recallStatusId').value === RecallTaskStatus.Pending) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.approve,
          title: 'Desc_Order_Approve',
          icon: 'fa fa-check',
          color: '#28a745'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Cancel) &&
      this.recallFilterForm.get('recallStatusId').value === RecallTaskStatus.Pending) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.delete,
          title: 'Desc_Cancel',
          icon: 'fa fa-close',
          color: '#f1685e'
        });
    }
    // if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Execute) &&
    //   this.recallFilterForm.get('recallStatusId').value === RecallTaskStatus.Approved) {
    //   this.TabelData.rowOperations.push(
    //     {
    //       operation: RowOperation.Execute,
    //       title: 'Desc_Vehilce_Execute',
    //       icon: 'fa fa-cogs',
    //       // fa fa-wrench
    //       color: "#12344d"
    //     });
    // }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Close) &&
      this.recallFilterForm.get('recallStatusId').value === RecallTaskStatus.Approved) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.Close,
          title: 'Desc_Close',
          icon: 'fa fa-circle-o-notch',
          color: "#12344d"
        });
    }

    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Edit) &&
      this.recallFilterForm.get('recallStatusId').value === RecallTaskStatus.Pending) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.edit,
          title: "Desc_Edit",
          icon: "fa fa-pencil",
          color: "#12344d",
          showWhenKeyValueFalse: true,
          showHideOptionPerLine: true,
          controlKeyName: 'inactive'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.View)) {
      this.TabelData.rowOperations.push(
        {
          operation: RowOperation.View,
          title: 'Desc_View',
          icon: 'fa fa-eye',
          color: '#12344d',
          showWhenKeyValueFalse: true,
          showHideOptionPerLine: true,
          controlKeyName: 'inactive'
        });
    }
  }
  fillRecallStatusList() {
    this.isStatusLoading = true;
    this.recallStatusList = [];
    this.recallService.getRecallStatuses().subscribe(result => {
      this.recallStatusList = result;
      var index = this.recallStatusList.findIndex(x => x.recallStatusId === RecallTaskStatus.Executed);
      if (index >= 0) {
        this.recallStatusList.splice(index, 1);
      }
      this.isStatusLoading = false;
    });
  }
  populateRecallList() {
    this.recallFilterForm.markAsTouched();
    if (this.recallFilterForm.get('recallStatusId').invalid) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectStatus));
      return;
    }
    this.recallFilter.recallStatusId = this.recallFilterForm.get('recallStatusId').value;
    this.recallFilter.dateRange = this.dateRange;

    this.coreSession.ModalLoading.Show();
    this.recallService.getRecallListSharedTable(this.recallFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.addActions();
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data,
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  initFilterForm() {
    this.recallFilterForm = new FormGroup({
      recallStatusId: new FormControl(RecallTaskStatus.Pending, Validators.required),
      dateRange: new FormControl([]),
    });
  }
  onChangeSelectedDate(event: any) {
    this.dateRange = event;
  }
  onTableFilterChanged(query: any) {
    this.recallFilter.customListFilter = query;
    this.populateRecallList();
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.edit:
        var isEditAvailable = this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.Edit);
        if (event.object.recallStatusId === RecallTaskStatus.Pending && isEditAvailable) {
          this.openSliderToEditRecall(event.object, true, false, false);
        }
        break;
      case RowOperation.delete:
        if (event.object.recallStatusId === RecallTaskStatus.Pending) {
          this.updateRecallStatus(event.object.recallId, RecallTaskStatus.Canceled);
        }
        break;
      case RowOperation.approve:
        if (event.object.recallStatusId === RecallTaskStatus.Pending) {
          this.prepareRecallApproveConfirmation(event.object.recallId);
        }
        break;
      case RowOperation.Close:
        this.checkIfAllRecallSubTasksExecuted(event.object);
        break;
      case RowOperation.View:
        var isViewAvailable = this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsListURL, MenuActions.View);
        if (isViewAvailable) {
          this.openSliderToEditRecall(event.object, false, true, false);
        }
        break;
    }
  }
  checkIfAllRecallSubTasksExecuted(recallTask) {
    var filter = {
      recallId: recallTask.recallId,
    }
    this.coreSession.ModalLoading.Show();
    this.recallService.checkIfAllRecallSubTasksExecuted(filter).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.openSliderToEditRecall(recallTask, false, true, true);
        } else {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  onFindButtonClicked() {
    this.recallFilter.customListFilter.page = 0;
    this.populateRecallList();
  }
  onResetFilter() {
    this.dateRange = [];
    this.recallFilterForm.reset();
  }
  openSliderToAddNewRecall() {
    this.isEditMode = false;
    this.isViewMode = false;
    this.isCloseMode = false;
    this.recallObject = null;
    this.showRecallEntrySlider = true;
  }
  openSliderToEditRecall(recallObj, isEditMode, isViewMode, isCloseMode) {
    this.isEditMode = isEditMode;
    this.isViewMode = isViewMode;
    this.isCloseMode = isCloseMode;
    this.recallObject = recallObj;
    this.showRecallEntrySlider = true;
  }
  closeRecallEntrySlider() {
    this.isEditMode = false;
    this.isViewMode = false;
    this.isCloseMode = false;
    this.recallObject = null;
    this.showRecallEntrySlider = false;
    this.coreSession.SetTitle('Recalls');
  }
  updateRecallStatus(recallId, newStatus) {
    var msg = '';
    if (newStatus === RecallTaskStatus.Canceled) {
      msg = ConstantMessages.MsgCancelConfirmation
    }

    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(msg), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.coreSession.ModalLoading.Show();
          var filter = {
            recallId: recallId,
            recallStatusId: newStatus
          }
          this.recallService.updateRecallTaskStatus(filter).subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
              this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
              this.populateRecallList();
            } else {
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
            }
          },
            (error: HttpErrorResponse) => {
              this.coreSession.ModalLoading.Hide();
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            });
        }
      });
  }
  prepareRecallApproveConfirmation(recallId) {
    // 1. get confirmation data from API
    // 2. show dialog with data
    // 3. Process with Approve process
    this.coreSession.ModalLoading.Show();
    var filter = {
      recallId: recallId,
    }
    this.recallService.prepareApproveConfirmation(filter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        if (response.data && response.data.length > 0) {
          this.showApproveConfirmationDialog(response.data, recallId);
        }
        else {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgNoDataFound));
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  showApproveConfirmationDialog(operationsList, recallId) {
    var MWList = [];
    var vehiclesList = [];
    var customersList = [];
    this.operationsToApprove = [];
    // operationsToApprove containes the following properties :
    // recallOperation
    // caption
    // isCollapsed
    // partyList

    MWList = operationsList.filter(x => x.recallOperation === RecallInternalOperationTypes.TransferOut);
    vehiclesList = operationsList.filter(x => x.recallOperation === RecallInternalOperationTypes.VehicleOffload);
    customersList = operationsList.filter(x => x.recallOperation === RecallInternalOperationTypes.ReturnOrder);

    if (MWList && MWList.length > 0) {
      this.operationsToApprove.push({ recallOperation: RecallInternalOperationTypes.TransferOut, caption: 'Desc_Offloading_Warehouses', isCollapsed: true, partyList: MWList });
    }
    if (vehiclesList && vehiclesList.length > 0) {
      this.operationsToApprove.push({ recallOperation: RecallInternalOperationTypes.VehicleOffload, caption: 'Desc_Offloading_Vehicles', isCollapsed: true, partyList: vehiclesList });
    }
    if (customersList && customersList.length > 0) {
      this.operationsToApprove.push({ recallOperation: RecallInternalOperationTypes.ReturnOrder, caption: 'Desc_Return_From_Customers', isCollapsed: true, partyList: customersList });
    }

    this.modalService.open(this.approveConfirmationDialog, { centered: true }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
          this.approveRecallTask(recallId);
        }
      });
  }
  approveRecallTask(recallId) {
    this.coreSession.ModalLoading.Show();
    var filter = {
      recallId: recallId,
    }
    this.recallService.approveRecallTask(filter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.populateRecallList();
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgUpdatedSuccessfuly));
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  onSaveRecallClick() {
    this.saveSubject.next()
  }
  afterSaveRecall() {
    this.closeRecallEntrySlider();
    this.populateRecallList();
  }
}

