import { Component, OnInit, Output, Input, EventEmitter, ViewChild } from '@angular/core';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { FormProcessMode } from '../../../shared/models/enums/form-process-mode.enum';
import { IDateTimePickerProperties } from '../../../shared/models/dateTimeModel/date-time-properties.interface';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { CoreSession } from '../../../core/core.session';
import { RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { Observable, Subject } from 'rxjs';
import { TranslateService } from '@ngx-translate/core';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { WarehouseTransactionStatus } from '../../../shared/models/enums/warehouse-transaction-status.enum';
import { RecallService } from '../recall.service';
import { ItemPackModel } from '../../../shared/models/Item/item.model';
import { SerialStatus } from '../../../shared/models/enums/SerialStatus';
import { SerialsPropertiesInterface } from '../../../shared/components/Serials/serials-properties.interface';
import { NgbModal, NgbModalConfig } from '@ng-bootstrap/ng-bootstrap';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import * as cloneDeep from "lodash/cloneDeep";
import { RecallSubTasksService } from '../recall-sub-tasks.service';

@Component({
  selector: 'app-recall-sub-task-execution',
  templateUrl: './recall-sub-task-execution.component.html',
  styleUrls: ['./recall-sub-task-execution.component.css']
})
export class RecallSubTaskExecutionComponent implements OnInit {

  @ViewChild("availableSerialsListDialog", { static: true }) availableSerialsListDialog: NgbModal;
  @Input() formProcessMode: FormProcessMode;
  @Input() entryScreenTitle: null;
  @Input() WHTransaction = null;
  @Input() saveEvent: Observable<void>;
  @Output('onSaveComplete') onSaveComplete = new EventEmitter();
  saveBtnSubscription: any;

  operationForm: FormGroup = new FormGroup({});
  transactionDate: DateTimeModel = new DateTimeModel();
  transactionDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Transaction_Date',
    formControlName: 'transactionDate',
    isCalendarOnly: true
  };


  itemsList: any[] = [];
  itemsSharedTabel: ITableProperties = {
    pageSize: 100,
    showPaginator: false,
    showSearch: true,
    isOnline: false,
    showSearchBtn: false,
    isMultiSelection: false,
    // showOptionPerLine: true,
    // showOptionOnKey: 'hasSerialNumber',
    rowOperations: [{
      operation: RowOperation.Adjust,
      title: 'Desc_Add_Serials',
      icon: 'fa fa-barcode',
      color: "#02b875",
      showHideOptionPerLine: true,
      showWhenKeyValueTrue: true,
      controlKeyName: 'hasSerialNumber'
    },
    {
      operation: RowOperation.View,
      title: 'Desc_View_Available_Serials',
      icon: "fa fa-eye",
      color: '#12344d',
      showHideOptionPerLine: true,
      showWhenKeyValueTrue: true,
      controlKeyName: 'recalledBySerial'
    },
    {
      operation: RowOperation.delete,
      title: 'Desc_Clear',
      icon: "fa fa-trash",
      color: '#f1685e'
    }],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Item', key: 'itemCodeName', isSortable: true, width: '20%' },
      { title: 'Desc_UOM', key: 'uom', isSortable: true, width: '8%' },
      { title: 'Desc_BatchNo', key: 'batchNo', isSortable: false, width: '12%' },
      { title: 'Desc_ExpiryDate', key: 'expiryDate', isSortable: false, width: '12%', isExpiryDate: true },
      // { title: 'Desc_Recalled_QTY', key: 'approvedQuantity', isSortable: false, width: '12%', numberOfStockDigits: true },
      { title: 'Desc_Stock_QTY_Recall', key: 'requiredQty', isSortable: false, width: '14%', numberOfStockDigits: true },
      { title: 'Desc_Actual_QTY', key: 'existingQty', isSortable: false, width: '14%', numberOfStockDigits: true, isEditable: true, disableInputBasedOnProperty: true, disableInputPropertyName: 'hasSerialNumber' },
      { title: 'Desc_Notes', key: 'notes', isSortable: false, width: '20%', isEditable: true },
    ]
  };
  itemsDataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  showSerialsSlider = false;
  saveSerialsSubject: Subject<void> = new Subject<void>();
  serialsProperties: SerialsPropertiesInterface;
  addedSerialsForAllPacks = [];
  availableSerialsForItemRecalledBySerial = [];
  constructor(private sessionData: SessionDataProvider,
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private recallSubTasksService: RecallSubTasksService,
    private config: NgbModalConfig,
    private modalService: NgbModal) { }

  ngOnInit() {
    this.subscribeSaveClick();
    this.coreSession.hideSubRecallExecutionEntrySliderFooter = false;
    this.coreSession.SetTitle(this.translateService.instant(this.entryScreenTitle));
    this.transactionDate = this.WHTransaction.transactionDateModel;
    this.transactionDateProperties.dateValue = this.transactionDate;
    this.initForm();
    this.itemsList = this.WHTransaction.addedItems;
    this.itemsDataSource = {
      totalItems: this.itemsList.length,
      data: this.itemsList
    };
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }
  subscribeSaveClick() {
    this.saveBtnSubscription = this.saveEvent.subscribe(() => {
      this.onSaveWHOperation();
    });
  }
  initForm() {
    this.operationForm = new FormGroup({
      transactionId: new FormControl({ value: this.WHTransaction.transactionId, disabled: true }),
      warehouseText: new FormControl({ value: this.WHTransaction.warehouseNameCode, disabled: true }),
      refWarehouseId: new FormControl({ value: this.WHTransaction.refWarehouseId == -1 ? null : this.WHTransaction.refWarehouseId, disabled: this.WHTransaction.refWarehouseId != -1 }, Validators.required),
      employeeText: new FormControl({ value: this.WHTransaction.employeeCodeName, disabled: true }),
      transactionDate: new FormControl({ value: this.transactionDate, disabled: true }),
      divisionText: new FormControl({ value: this.WHTransaction.DivisionName, disabled: true }),
      divisionId: new FormControl({ value: this.WHTransaction.divisionId, disabled: true }),
      notes: new FormControl({ value: this.WHTransaction.notes, disabled: false }),
    })
  }
  get refWarehouseInput() {
    return this.operationForm.get('refWarehouseId');
  }
  isDivisionLevel() {
    return (this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true"
      && this.sessionData.getConfigurationValue(ConstantConfigurations.ShowDivisionSelectionOnWHTransaction).toLowerCase() === "true")
  }
  onTransactionDateSelected(date) {
    this.transactionDate = date;
  }
  onRowOperation(event: any) {
    switch (event.operation) {
      case RowOperation.Adjust: // scann serials
        if (event.object.hasSerialNumber) {
          if (event.object.recalledBySerial && (event.object.availableSerialsForItemRecalledBySerial == null || event.object.availableSerialsForItemRecalledBySerial.length === 0)) {
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.Desc_No_Available_Serials));
            return;
          }
          this.openSliderToAddSerials(event.object);
        }
        break;
      case RowOperation.View: // View serials defined by user
        if (event.object.hasSerialNumber && event.object.recalledBySerial) {
          if (event.object.availableSerialsForItemRecalledBySerial == null || event.object.availableSerialsForItemRecalledBySerial.length === 0) {
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.Desc_No_Available_Serials));
            return;
          } else {
            this.availableSerialsForItemRecalledBySerial = event.object.availableSerialsForItemRecalledBySerial;
            this.viewDefinedSerialsListDialog();
          }
        }
        break;
      case RowOperation.delete:
        this.onClearItem(event.object);
        break;
    }
  }
  onClearItem(selectedPack: ItemPackModel) {
    selectedPack.notes = '';
    if (selectedPack.existingQty > 0) {
      this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgDeleteConfirmation), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
        (res: DialogResult) => {
          if (res === DialogResult.Yes) {
            var index = this.getPackIndex(selectedPack);
            if (index >= 0) {
              this.removePackSerialsFromAllSerialsList(selectedPack);
              selectedPack.existingQty = 0; // clear qty inserted by user not orginal stock (required) qty
              selectedPack.addedSerials = [];
            }
          }
        });
    }
  }
  removePackSerialsFromAllSerialsList(selectedPack) {
    // remove old scanned serials for this pack batch
    this.addedSerialsForAllPacks = this.addedSerialsForAllPacks.filter(x =>
      x.itemId != selectedPack.itemId ||
      x.packId != selectedPack.packId ||
      x.batchNo != selectedPack.batchNo ||
      x.expiryDateModel.date != selectedPack.expiryDateModel.date ||
      x.expiryDateModel.day != selectedPack.expiryDateModel.day ||
      x.expiryDateModel.month != selectedPack.expiryDateModel.month ||
      x.expiryDateModel.year != selectedPack.expiryDateModel.year);
  }

  //#region [Serials]
  fillSerialProperties(selectedPack: ItemPackModel) {
    selectedPack.recallId = this.WHTransaction.recallId;
    this.serialsProperties = {
      itemFormProcessMode: this.formProcessMode,
      packObj: cloneDeep(selectedPack),
      packObjects: [],
      warehouseId: this.formProcessMode === FormProcessMode.OffLoad ? this.WHTransaction.refWarehouseId : this.WHTransaction.warehouseId,
      vehicleId: this.formProcessMode === FormProcessMode.OffLoad ? this.WHTransaction.warehouseId : -1,
      transactionId: this.WHTransaction.transactionId,
      newSerialStatus: this.formProcessMode === FormProcessMode.OffLoad ? SerialStatus.OffloadedToWarehouse : SerialStatus.InTranset,
      requiredQuantity: selectedPack.requiredQty,
      quantityMustMatch: false,
      transactionRecallId: this.WHTransaction.recallId
    };
  }
  openSliderToAddSerials(selectedPack: ItemPackModel) {
    this.fillSerialProperties(selectedPack);
    this.showSerialsSlider = true;
    this.coreSession.hideSubRecallExecutionEntrySliderFooter = true;
  }
  onSaveSerialsClick() {
    this.saveSerialsSubject.next();
  }
  onAfterSaveSerials(selectedPack: ItemPackModel) {
    var index = this.getPackIndex(selectedPack);
    if (index >= 0) {
      this.removePackSerialsFromAllSerialsList(selectedPack);
      this.itemsList[index].existingQty = selectedPack.requiredQty; // set actual qty scanned by the user
      this.itemsList[index].addedSerials = selectedPack.addedSerials; // set added serial in scanned serials
      this.itemsList[index].addedSerials.forEach(serial => {
        this.addedSerialsForAllPacks.push(serial);
      });
    }
    this.closeSerialsSlider();
  }
  getPackIndex(selectedPack: ItemPackModel) {
    var index = this.itemsList.findIndex(x =>
      x.itemId === selectedPack.itemId &&
      x.packId === selectedPack.packId &&
      x.batchNo === selectedPack.batchNo &&
      x.expiryDateModel.date === selectedPack.expiryDateModel.date &&
      x.expiryDateModel.day === selectedPack.expiryDateModel.day &&
      x.expiryDateModel.month === selectedPack.expiryDateModel.month &&
      x.expiryDateModel.year === selectedPack.expiryDateModel.year
    );
    return index;
  }
  closeSerialsSlider() {
    this.showSerialsSlider = false;
    this.coreSession.hideSubRecallExecutionEntrySliderFooter = false;
  }

  viewDefinedSerialsListDialog() {
    this.modalService.open(this.availableSerialsListDialog, { centered: true }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
        }
      });
  }
  //#endregion

  //#region [Save]
  onSaveWHOperation() {
    if (this.operationForm.invalid) {
      this.coreSession.markFormGroupTouched(this.operationForm);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    this.WHTransaction.addedItems = this.itemsList;
    this.WHTransaction.whTransactionStatusId = WarehouseTransactionStatus.Executed;
    this.WHTransaction.refWarehouseId = this.refWarehouseInput.value;
    this.WHTransaction.isNewTransaction = false;

    this.WHTransaction.addedItems.forEach(x => {
      x.existingQty = +x.existingQty; // Convert input to number instead of string
      // x.recallScannedSerials = x.addedSerials; // set recallScannedSerials, added serials will handeled in back end
      // x.addedSerials = [];
    });
    this.WHTransaction.addedItems = this.WHTransaction.addedItems.filter(x => x.existingQty > 0 || x.requiredQty > 0 || x.approvedQuantity > 0)
    this.coreSession.ModalLoading.Show();
    this.recallSubTasksService.executeWHRecallSubTask(this.WHTransaction).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSaveComplete.emit();
      } else {
        if (response.data != null && response.data.length > 0) {
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
          // this.invalidSerials = response.data;
          // this.invalidSerialsDialog.showDialog().then(
          //   (result) => {

          //   });
        } else
          this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }
  //#endregion
}
