import { Component, OnInit, ViewChild } from '@angular/core';
import { Subject } from 'rxjs';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { TranslateService } from '@ngx-translate/core';
import { CoreSession } from '../../../core/core.session';
import { MenuActions } from '../../../shared/models/enums/menu-actions-enum';
import { ConstantURLs } from '../../../shared/models/constants/constant-URL';
import { RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { RecallService } from '../recall.service';
import { RecallTaskStatus } from '../../../shared/models/enums/recall-task-status.enum';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { FormProcessMode } from '../../../shared/models/enums/form-process-mode.enum';
import { RecallInternalOperationTypes } from '../../../shared/models/enums/recall-internal-party-types.enum';
import { IOperationSummary } from '../../../shared/models/operation-summary/operation-summary-interface';
import { OrderService } from '../../order/order.service';
import { OrderListFilter } from '../../../shared/models/shared-table/custom-list.interface';
import { DialogResult } from '../../../shared/models/enums/dialog-result.enum';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { DateTimeModel } from '../../../shared/models/dateTimeModel/date-time.model';
import { IDateTimePickerProperties } from '../../../shared/models/dateTimeModel/date-time-properties.interface';
import { DialogMode } from '../../../shared/models/enums/dialog-mode.enum';
import { OrderStatus } from '../../../shared/models/enums/order-status.enum';
import { RecallSubTasksService } from '../recall-sub-tasks.service';

@Component({
  selector: 'app-recall-sub-tasks-list',
  templateUrl: './recall-sub-tasks-list.component.html',
  styleUrls: ['./recall-sub-tasks-list.component.css']
})
export class RecallSubTasksListComponent implements OnInit {

  @ViewChild("dispatchOrderDialog", { static: true }) dispatchOrderDialog: NgbModal;
  OkResult: DialogResult = DialogResult.Ok;
  saveSubject: Subject<void> = new Subject<void>();
  recallFilterForm: FormGroup;
  recallStatusList: any[] = [];
  isStatusLoading = false;
  recallTaksList: any[] = [];
  isRecallTasksLoading = false;
  ControlTransactionDivisionLevel = false;

  subTasksSharedTabel: ITableProperties = {
    pageSize: this.coreSession.pageSize,
    showPaginator: true,
    isOnline: true,
    showSearch: true,
    showSearchBtn: true,
    showNewBtn: false,
    isMultiSelection: false,
    openFilterDirectly: true,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      { title: 'Desc_Recall_Task', key: 'recallCodeName', isSortable: true, width: '18%' },
      { title: 'Desc_Operation_Type', key: 'recallOperationName', isSortable: true, width: '15%' },
      { title: 'Desc_Transaction_ID', key: 'transactionId', isSortable: true, width: '18%' },
      // { title: 'Desc_Customer', key: 'customerCodeName', isSortable: true, width: '25%' },
      // { title: 'Desc_Warehouse', key: 'warehouseCodeName', isSortable: true, width: '12%' },
      { title: 'Desc_Location', key: 'partyCodeName', isSortable: true, width: '15%' },
      { title: 'Desc_Organization', key: 'organizationCodeName', isSortable: true, width: '15%' },
      { title: 'Desc_Status', key: 'recallStatusName', isSortable: true, width: '10%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  recallFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: this.coreSession.pageSize
    },
    recallId: -1,
    recallStatusId: RecallTaskStatus.Pending
  };
  entryScreenTitle = null;
  formProcessMode: FormProcessMode = FormProcessMode.OffLoad; // [OffLoad, TransferOut, Orders]
  WHTransaction = null;
  selectedOrder = null
  showWHExecutionSlider = false;
  showViewSummarySlider = false;
  operationSummaryData: IOperationSummary = {
    operationData: {},
    formProcessMode: FormProcessMode.OffLoad,
    formTitle: ''
  };

  assignOrdersForm: FormGroup;
  selectedSchdualeDateModel: DateTimeModel;
  schduleDateProperties: IDateTimePickerProperties = {
    label: "Desc_Schedule_Date",
    formControlName: "schduleDate",
    min: new Date()
  };
  selectedDriver: any = {
    vehicleId: -1
  };

  constructor(
    public coreSession: CoreSession,
    private recallSubTasksService: RecallSubTasksService,
    private translateService: TranslateService,
    private sessionData: SessionDataProvider,

    private orderService: OrderService,
    private modalService: NgbModal) {
  }
  ngOnInit() {
    this.coreSession.SetTitle('Recall Sub Tasks');
    this.ControlTransactionDivisionLevel = this.sessionData.getConfigurationValue(ConstantConfigurations.ControlTransactionOnDivisionLevel).toLowerCase() === "true"
    if (this.ControlTransactionDivisionLevel) {
      this.subTasksSharedTabel.columns.push({ title: 'Desc_Division', key: 'divisionCodeName', isSortable: false, width: '15%' });
    }
    this.fillRecallStatusList();
    this.fillRecallList();
    this.initFilterForm();
    this.initDispatchOrderForm();
    this.addActions();
    this.populateRecallSubTasksList();
  }
  ngOnDestroy() {
    if (this.modalService) {
      this.modalService.dismissAll();
    }
  }
  addActions() {
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsSubTasksListURL, MenuActions.View)) {
      this.subTasksSharedTabel.rowOperations.push(
        {
          operation: RowOperation.View,
          title: 'Desc_View',
          icon: 'fa fa-eye',
          color: "#02b875",
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsSubTasksListURL, MenuActions.Execute)) {
      this.subTasksSharedTabel.rowOperations.push(
        {
          operation: RowOperation.Execute,
          title: 'Desc_Vehilce_Execute',
          icon: 'fa fa-cogs',
          // fa fa-wrench
          color: "#12344d",
          showHideOptionPerLine: true,
          showWhenKeyValueFalse: true,
          controlKeyName: 'isOrder'
        });
    }
    if (this.coreSession.checkActivitiesAvailability(ConstantURLs.recallsSubTasksListURL, MenuActions.Dispatch)) {
      this.subTasksSharedTabel.rowOperations.push(
        {
          operation: RowOperation.Dispatch,
          title: 'Desc_Order_Dispatch',
          icon: 'fa fa-link',
          color: "#12344d",
          showHideOptionPerLine: true,
          showWhenKeyValueTrue: true,
          controlKeyName: 'isOrder'
        });
    }
  }
  fillRecallStatusList() {
    this.isStatusLoading = true;
    this.recallStatusList = [];
    this.recallStatusList.push(
      { recallStatusId: RecallTaskStatus.Pending, recallStatusName: this.translateService.instant('Desc_Pending') },
      { recallStatusId: RecallTaskStatus.Executed, recallStatusName: this.translateService.instant('Desc_Executed') }
    );
    this.isStatusLoading = false;

    // this.isStatusLoading = true;
    // this.recallStatusList = [];
    // this.recallService.getRecallStatuses().subscribe(result => {
    //   this.recallStatusList = result;
    //   this.isStatusLoading = false;
    // });
  }
  fillRecallList() {
    this.isRecallTasksLoading = true;
    this.recallTaksList = [];
    this.recallSubTasksService.getRecallTasksList().subscribe(result => {
      this.recallTaksList = result;
      this.isRecallTasksLoading = false;
    });
  }
  populateRecallSubTasksList() {

    this.recallFilter.recallStatusId = this.recallFilterForm.get('recallStatusId').value ? this.recallFilterForm.get('recallStatusId').value : -1;
    this.recallFilter.recallId = this.recallFilterForm.get('recallId').value ? this.recallFilterForm.get('recallId').value : -1;

    this.coreSession.ModalLoading.Show();
    this.recallSubTasksService.getRecallSubListSharedTable(this.recallFilter).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.dataSource = {
          totalItems: response.data.totalItems,
          data: response.data.data,
        }
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  initFilterForm() {
    this.recallFilterForm = new FormGroup({
      recallId: new FormControl(null),
      recallStatusId: new FormControl(RecallTaskStatus.Pending),
    });
  }
  initDispatchOrderForm() {
    this.assignOrdersForm = new FormGroup({
      employeeId: new FormControl(null, [Validators.required]),
      schduleDate: new FormControl(null, [Validators.required]),
    });
  }
  onTableFilterChanged(query: any) {
    this.recallFilter.customListFilter = query;
    this.populateRecallSubTasksList();
  }
  onRowOperation(event: any) {
    this.selectedOrder = null;
    this.WHTransaction = null;
    switch (event.operation) {
      case RowOperation.Execute:
        if (event.object.recallOperation === RecallInternalOperationTypes.TransferOut || event.object.recallOperation === RecallInternalOperationTypes.VehicleOffload) {
          if (event.object.recallSubTaskStatusId != RecallTaskStatus.Pending) {
            this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgSelectedTransactionIsExeceuted));
          } else {
            this.executeWHSubTask(event.object);
          }
        }
        break;
      case RowOperation.Dispatch:
        if (event.object.recallOperation === RecallInternalOperationTypes.ReturnOrder) {
          // Set customer, outlet, division, organization
          this.selectedOrder = event.object;
          // Set Order Id
          this.selectedOrder.orderId = event.object.transactionId;
          this.openDispatchOrderDialog();
        }
        break;
      case RowOperation.rowClick:
      case RowOperation.View:
        if (event.object.recallOperation === RecallInternalOperationTypes.TransferOut || event.object.recallOperation === RecallInternalOperationTypes.VehicleOffload) {
          this.viewWarehouseTransactionSummary(event.object);
        } else if (event.object.recallOperation === RecallInternalOperationTypes.ReturnOrder) {
          this.viewOrderSummary(event.object);
        }
        break;
    }
  }
  onFindButtonClicked() {
    this.recallFilter.customListFilter.page = 0;
    this.populateRecallSubTasksList();
  }
  onResetFilter() {
    this.recallFilterForm.reset();
  }
  executeWHSubTask(selectedSubTask) {
    this.coreSession.ModalLoading.Show();
    this.recallSubTasksService.getRecallSubTaskDataToExecute(selectedSubTask).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.WHTransaction = response.data;
          if (selectedSubTask.recallOperation === RecallInternalOperationTypes.VehicleOffload) {
            this.formProcessMode = FormProcessMode.OffLoad;
            this.entryScreenTitle = 'Desc_Execute_Offload_Request';
            this.showWHExecutionSlider = true;
          }
          if (selectedSubTask.recallOperation === RecallInternalOperationTypes.TransferOut) {
            this.formProcessMode = FormProcessMode.TransferOut;
            this.entryScreenTitle = 'Desc_Execute_Transfer_Out';
            this.showWHExecutionSlider = true;
          }
          if (selectedSubTask.recallOperation === RecallInternalOperationTypes.ReturnOrder) {
            this.formProcessMode = FormProcessMode.Orders;
            this.entryScreenTitle = 'Desc_Order';
          }
        } else {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
  viewWarehouseTransactionSummary(selectedSubTask) {
    this.coreSession.ModalLoading.Show();
  }

  viewOrderSummary(selectedSubTask) {
    this.coreSession.ModalLoading.Show();
    let orderFilter: OrderListFilter = {};
    orderFilter.orderId = selectedSubTask.transactionId;
    orderFilter.customerId = selectedSubTask.customerId;
    orderFilter.outletId = selectedSubTask.outletId;
    orderFilter.divisionId = selectedSubTask.divisionId;
    orderFilter.isViewMode = true;
    this.orderService.getOrder(orderFilter).subscribe(
      response => {
        this.coreSession.ModalLoading.Hide();
        if (response.status != null && response.status >= 0) {
          this.selectedOrder = response.data;
          this.operationSummaryData = {
            operationData: this.selectedOrder,
            formProcessMode: FormProcessMode.Orders,
            formTitle: 'Desc_Order_Summary'
          };
          this.showViewSummarySlider = true;
        } else {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), response.message);
        }
      },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }

  // #region [Dispatch Orders]
  onEmployeeDataChanged(employee) {
    this.selectedDriver = {};
    if (employee) {
      this.selectedDriver = employee;
    }
  }
  openDispatchOrderDialog() {
    this.assignOrdersForm.reset();
    this.modalService.open(this.dispatchOrderDialog, { centered: true }).result.then(
      (result) => {

      });
  }
  onDispatchButtonClicked() {
    this.dispatchOrder();
  }
  dispatchOrder() {
    this.coreSession.markFormGroupTouched(this.assignOrdersForm);
    if (this.assignOrdersForm.invalid) {
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant('Desc_Fill_Field_To_Dispatch'));
      return;
    }
    var msg = this.selectedOrder.orderStatusId === OrderStatus.Assigned ? ConstantMessages.Desc_Re_Assigne_Order_Confirmation : ConstantMessages.Desc_Assigne_Order_Confirmation;
    this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(msg), DialogMode.YesNo, this.translateService.instant(ConstantMessages.WarningCaption)).then(
      (res: DialogResult) => {
        if (res === DialogResult.Yes) {
          this.selectedOrder.scheduleDateModel = this.selectedSchdualeDateModel;
          this.selectedOrder.driverId = this.selectedDriver.employeeId;
          this.selectedOrder.vehicleId = this.selectedDriver.vehicleId;

          this.coreSession.ModalLoading.Show();
          this.recallSubTasksService.dispatchRecallOrder(this.selectedOrder).subscribe(
            response => {
              this.coreSession.ModalLoading.Hide();
              if (response.status != null && response.status >= 0) {
                this.populateRecallSubTasksList();
                this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.OrderDispatched));
                this.modalService.dismissAll();
              } else {
                this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), response.message);
                this.modalService.dismissAll();
              }
            },
            (error: HttpErrorResponse) => {
              this.coreSession.ModalLoading.Hide();
              this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
          );
        }
      });
  }
  onSchduleDateSelected(event: any) {
    this.selectedSchdualeDateModel = event;
  }
  //#endregion

  onSaveClick() {
    this.saveSubject.next();
  }
  onSaveComplete() {
    this.closeSliders();
    this.populateRecallSubTasksList();
  }
  closeSliders() {
    this.formProcessMode = FormProcessMode.OffLoad;
    this.WHTransaction = null;
    this.selectedOrder = null;
    this.showWHExecutionSlider = false;
    this.showViewSummarySlider = false;
    this.operationSummaryData = null;
    this.entryScreenTitle = null;
  }

}

