import { Component, OnInit, Input, EventEmitter, Output } from '@angular/core';
import { Observable } from 'rxjs';
import { FormGroup, FormControl, Validators } from '@angular/forms';
import { CoreSession } from '../../../../core/core.session';
import { TranslateService } from '@ngx-translate/core';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { EchoService } from '../../../echo.service';
import { HttpErrorResponse } from '@angular/common/http';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { DatePipe } from '@angular/common';

@Component({
  selector: 'app-cust-app-reply',
  templateUrl: './cust-app-reply.component.html',
  styleUrls: ['./cust-app-reply.component.css']
})
export class CustAppReplyComponent implements OnInit {
  @Input() events: Observable<void>;
  @Input() feedbackReply: any;
  @Input() isFooterDisabled: false;
  @Output() onSaveCompleted = new EventEmitter();
  myForm: FormGroup;
  saveBtnSubscription: any;
  dateFormat: string;
  constructor(private coreSession: CoreSession,
    private translateService: TranslateService,
    private echoService: EchoService, 
    private sessionDataProvider: SessionDataProvider,
    public datePipe: DatePipe) { }

  ngOnInit() {
    this.initForm();
    this.subscribeSaveClick();

  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
  }
  initForm() {
    this.myForm = new FormGroup({
      customerName: new FormControl({ value: this.feedbackReply.customerName, disabled: true }),
      customerCode: new FormControl({ value: this.feedbackReply.customerCode, disabled: true }),
      createdDate: new FormControl({ value: this.feedbackReply.createdDate, disabled: true }),
      replyDate: new FormControl({ value: this.datePipe.transform(this.feedbackReply.replyDate, this.sessionDataProvider.DateTimeFormat), disabled: true }),
      feedbackSubject: new FormControl({ value: this.feedbackReply.subject, disabled: true }),
      feedbackBody: new FormControl({ value: this.feedbackReply.message, disabled: true }),
      replyBody: new FormControl({value: this.feedbackReply.replyBody, disabled: this.isFooterDisabled}, Validators.required)
    });
  }
  subscribeSaveClick() {
    this.saveBtnSubscription = this.events.subscribe(() => {
      this.onSaveClicked();
    });
  }

  onSaveClicked() {
    if(this.myForm.invalid){
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
    }else {
      this.feedbackReply.replyBody = this.myForm.get('replyBody').value;
      this.saveFeedbackReply(this.feedbackReply);
    }
    
  }

  saveFeedbackReply(feedbackReply: any) {
    this.coreSession.ModalLoading.Show();
    this.echoService.saveFeedbackReply(feedbackReply).subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), response.message);
        this.onSaveCompleted.emit();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    },
      (error: HttpErrorResponse) => {
        this.coreSession.ModalLoading.Hide();
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    );
  }
}
