import { Component, EventEmitter, Input, OnDestroy, OnInit, Output } from '@angular/core';
import { FormControl, FormGroup } from '@angular/forms';
import { TranslateService } from '@ngx-translate/core';
import { Observable } from 'rxjs/Rx';
import { CoreSession } from '../../../core/core.session';
import { SessionDataProvider } from '../../../core/session-data-provider.service';
import { ConstantConfigurations } from '../../../shared/models/constants/constant-configuration';
import { ConstantMessages } from '../../../shared/models/constants/constant-message';
import { MultiSelectionOperation, RowOperation } from '../../../shared/models/enums/shared-table-operation.enum';
import { SharedTableResult } from '../../../shared/models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../shared/models/shared-table/table-properties.interface';
import { EchoService } from '../../echo.service';

@Component({
  selector: 'app-cust-app-items-entry',
  templateUrl: './cust-app-items-entry.component.html',
  styleUrls: ['./cust-app-items-entry.component.css']
})
export class CustAppItemsEntryComponent implements OnInit, OnDestroy {

  @Input() events: Observable<void>;
  @Output() onSaveCompleted = new EventEmitter();
  public itemFilterForm: FormGroup;
  itemsList: any[]= [];
  saveBtnSubscription: any;
  isCategoryCustomerAppMode : boolean = false;
  isBrandCustomerAppMode : boolean = false;
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };

  TabelProperties: ITableProperties = {
    pageSize: 25,
    showPaginator: true,
    showSearch: true,
    isOnline: false,
    isMultiSelection: true ,
    rowOperations: [],
    multiSelectionOperations: [],
    columns: [
      {
        title: 'Desc_Item_Name',
        key: 'itemName',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Item_Code',
        key: 'itemCode',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_UOM',
        key: 'uom',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Barcode',
        key: 'itemDefinition',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Category',
        key: 'category',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Division',
        key: 'divisionName',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Brand',
        key: 'brand',
        isSortable: true,
        width: '14%'
      },
      {
        title: 'Desc_Image',
        key: 'ImagePath',
        isSortable: false,
        isImageURL:true,
        width: '14%'
      }
    ]
  };

  constructor(private echoService: EchoService,
    public coreSession: CoreSession,
    private translateService: TranslateService,
    private sessionData:SessionDataProvider) { }

  ngOnInit() {
    this.isCategoryCustomerAppMode = this.sessionData.getConfigurationValue(ConstantConfigurations.CategoryBrandCustomerAppMode).toLowerCase().includes('1');

    this.isBrandCustomerAppMode = this.sessionData.getConfigurationValue(ConstantConfigurations.CategoryBrandCustomerAppMode).toLowerCase().includes('2');
    this.populateTable();
    this.initFilterForm();
    this.subscribeSaveClick();
  }

  initFilterForm() {
    this.itemFilterForm = new FormGroup({
        divisionId: new FormControl(),
        itemCategoryId: new FormControl(),
        brandId: new FormControl(),
        hasImage:new FormControl()
    });
  }

  populateTable() {
    this.echoService.getEchoItems().subscribe(response => {
      if (response.status != null && response.status >= 0) {
        this.dataSource = <SharedTableResult>response.data;
        this.itemsList = response.data.data;
        if(this.itemsList!= null && this.itemsList.length >0){
          this.itemsList.forEach((orginalItem)=>{
            if (orginalItem.hasImage) {
                  orginalItem.ImagePath =
                  document.getElementsByTagName("base")[0].href + "Resources/items-img/" + orginalItem.itemId + ".png"
                } else {
                  orginalItem.ImagePath =
                  document.getElementsByTagName("base")[0].href + "Resources/NoProduct-Echo.jpg"
                }})
        }
      } else {
        this.coreSession.showError(ConstantMessages.ErrorCaption, this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    },
      error => {
        this.coreSession.showError(ConstantMessages.ErrorCaption, this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }

  onClearFilters(){
      this.itemFilterForm.get("divisionId").setValue(null);
      this.itemFilterForm.get("itemCategoryId").setValue(null);
      this.itemFilterForm.get("brandId").setValue(null);
  }

 onFindFilter(){
  var hasImage = this.itemFilterForm.get("hasImage").value;
  let itemsList = this.itemsList;
  if(hasImage==true)
  itemsList = itemsList.filter(x => x.hasImage == hasImage);

  this.dataSource = {
    data: itemsList,
    totalItems: itemsList.length
  }
}

  onSearchClick(){
    let divisionId = this.itemFilterForm.get("divisionId").value;
    let itemCategoryId = this.itemFilterForm.get("itemCategoryId").value;
    let brandId = this.itemFilterForm.get("brandId").value;
    let itemsList = this.itemsList;
    if(divisionId && divisionId != -1 ){
      itemsList = itemsList.filter(x => x.divisionId == divisionId);
    }
    if(itemCategoryId && itemCategoryId != -1 ){
      itemsList = itemsList.filter(x => x.itemCategoryId == itemCategoryId);
    }
    if(brandId && brandId != -1 ){
      itemsList = itemsList.filter(x => x.brandId == brandId);
    }

    this.dataSource = {
      data: itemsList,
      totalItems: itemsList.length
    }
  }

  subscribeSaveClick() {
    this.saveBtnSubscription = this.events.subscribe(() => {
      this.onSaveClicked();
    });
  }
  ngOnDestroy() {
    this.saveBtnSubscription.unsubscribe();
 }
  onSaveClicked(){
    this.dataSource = {
      data: this.itemsList,
      totalItems: this.itemsList.length
    }
    let selectedPacksList = this.dataSource.data.filter(x => x.isChecked);
    this.echoService.saveEchoItems(selectedPacksList).subscribe(response => {
      if (response.status != null && response.status >= 0) {
        this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgSavedSuccessfuly));
        this.onSaveCompleted.emit();
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      }
    },
      error => {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
      });
  }

}
