import { Component, OnDestroy, OnInit } from '@angular/core';
import { CoreSession } from '../../../../core/core.session';
import { ConstantMessages } from '../../../../shared/models/constants/constant-message';
import { EchoService } from '../../../echo.service';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { ITableProperties } from '../../../../shared/models/shared-table/table-properties.interface';
import { SharedTableResult } from '../../../../shared/models/shared-table/shared-table-result.interface';
import { RowOperation } from '../../../../shared/models/enums/shared-table-operation.enum';
import { Subject } from 'rxjs';
import { ConstantURLs } from '../../../../shared/models/constants/constant-URL';
import { NavBarService } from '../../../../core/sidebar/sidebar.service';
import { MenuActions } from '../../../../shared/models/enums/menu-actions-enum';
import { DialogMode } from '../../../../shared/models/enums/dialog-mode.enum';
import { TranslateService } from '@ngx-translate/core';
import { DialogResult } from '../../../../shared/models/enums/dialog-result.enum';
import { HttpErrorResponse } from '@angular/common/http';
import { NavigationData } from '../../../../shared/models/navigation-data/navigation-data.model';

@Component({
    selector: 'app-organization-details',
    templateUrl: './organization-details-list.component.html',
    styleUrls: ['./organization-details-list.component.css']
})
export class OrganizationDetailsListComponent implements OnInit, OnDestroy {
    saveSubject: Subject<void> = new Subject<void>();
    organizationDetailsObject: any;
    showSlider = false;
    isShowFooter = true;
    selectedOrganizationId = -1;
    isEditMode = false;
    isViewMode = false;
    TableData: ITableProperties = {
        pageSize: 50,
        showPaginator: false,
        showSearch: true,
        isOnline: true,
        showSearchBtn: false,
        showNewBtn: false,
        rowOperations: [
        ],
        multiSelectionOperations: [],
        columns: [
            {
                title: 'Desc_Vendor',
                key: 'organizationCode',
                isSortable: true,
                width: '50%'
            }, {
                title: 'Desc_Email',
                key: 'email',
                isSortable: true,
                width: '50%'
            },
        ]
    };
    dataSource: SharedTableResult = {
        totalItems: 0,
        data: []
    };
    navLinks: any[];
    constructor(private echoService: EchoService,
        public coreSession: CoreSession,
        private sessionDataProvider: SessionDataProvider,
        private navService: NavBarService,
        private translateService: TranslateService) { }



    ngOnInit() {
        this.coreSession.SetTitle('Organization Details');
        this.populateOrganizations();
        this.prepareNavigation();
        this.setNavigation(this.navLinks[0]);
        this.TableData.showNewBtn = this.coreSession.checkActivitiesAvailability(
            ConstantURLs.organizationDetailsURL,
            MenuActions.Add
        );
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.organizationDetailsURL, MenuActions.Edit)) {
            this.TableData.rowOperations.push({
                operation: RowOperation.edit,
                title: "Desc_Edit",
                icon: "fa-regular fa-pen-to-square",
                color: "#12344d"
            });
        }
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.organizationDetailsURL, MenuActions.Delete)) {
            this.TableData.rowOperations.push({
                operation: RowOperation.delete,
                title: "Desc_Delete",
                icon: "fa-solid fa-trash",
                color: "#f1685e",
            });
        }
        if (this.coreSession.checkActivitiesAvailability(ConstantURLs.organizationDetailsURL, MenuActions.View)) {
            this.TableData.rowOperations.push(
                {
                    operation: RowOperation.View,
                    title: 'Desc_View',
                    icon: 'fa-regular fa-eye',
                    color: '#12344d'
                });
        }
    }

    setNavigation(tab) {
      let navigationList: NavigationData[] = [];
      navigationList.push(
        {
          Name: 'Desc_Settings',
          URL: ConstantURLs.settingsURL
        },
        {
          Name: tab.label,
          URL: undefined
        }
      );
      this.navService.setNavigation(navigationList);
    }
    populateOrganizations() {
        this.coreSession.ModalLoading.Show();
        this.echoService.getOrganization().subscribe(response => {
            this.coreSession.ModalLoading.Hide();
            if (response.status != null && response.status >= 0) {
                this.dataSource = {
                    data: response.data,
                    totalItems: response.data.length
                };
            } else {
                this.coreSession.showError(response.message, '');
            }
        },
            error => {
                this.coreSession.showError(ConstantMessages.ErrorHappened, '');
            });
    }
    prepareNavigation() {
        this.navLinks = [
            {
                label: 'Desc_Vendor_Details',
                index: 0,
                isActive: true,
            }
        ];
    }
    onRowOperation(event: any) {
        switch (event.operation) {
            case RowOperation.edit: {
                this.isEditMode = true;
                this.isViewMode = false;
                this.getOrganizationDetails(event.object);
                break;
            }
            case RowOperation.delete: {
                this.DeleteOrganizationDetails(event.object);
                this.isEditMode = false;
                break;
            }
            case RowOperation.View:
                this.isViewMode = true;
                this.getOrganizationDetails(event.object);
                break;
        }
    }

    getOrganizationDetails(object) {
        this.coreSession.ModalLoading.Show();
        this.echoService.getOrganizationDetails(object.organizationID).subscribe(
            response => {
                this.coreSession.ModalLoading.Hide();
                if (response.status != null && response.status >= 0) {
                    this.organizationDetailsObject = response.data;
                    this.isEditMode = true;
                    this.showSlider = true;
                } else {
                    this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(response.message));
                }
            },
            (error: HttpErrorResponse) => {
                this.coreSession.ModalLoading.Hide();
                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
            }
        );
    }
    DeleteOrganizationDetails(object) {
        this.coreSession.ModalDialog.ShowMessage(
            this.translateService.instant(ConstantMessages.MsgDeleteConfirmation),
            DialogMode.YesNo,
            this.translateService.instant(ConstantMessages.Delete)
        ).then((result: DialogResult) => {
            if (result === DialogResult.Yes) {
                this.coreSession.ModalLoading.Show();
                this.echoService.deleteOrganizationDetails(object.organizationID)
                    .subscribe(
                        response => {
                            this.coreSession.ModalLoading.Hide();
                            if (response.status != null && response.status >= 0) {
                                this.coreSession.showSuccess(this.translateService.instant(ConstantMessages.SuccessCaption), this.translateService.instant(ConstantMessages.MsgDeletedSuccessfully));
                                this.populateOrganizations();
                            } else {
                                this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(response.message));
                            }
                        },
                        error => {
                            this.coreSession.ModalLoading.Hide();
                            this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
                        }
                    );
            }
        });
    }
    closeSlider() {
        this.organizationDetailsObject = null;
        this.selectedOrganizationId = -1;
        this.showSlider = false;
        this.isEditMode = false;
        this.coreSession.SetTitle("Organization Details");
    }

    onSaveClick() {
        this.saveSubject.next();
    }
    onSaveCompleted(isClose) {
        if (isClose)
            this.closeSlider();
        this.populateOrganizations();
    }
    ngOnDestroy() {
        this.navService.setNavigation([]);
    }

    onNewBtnClicked() {
        this.organizationDetailsObject = null;
        this.selectedOrganizationId = -1;
        this.isEditMode = false;
        this.showSlider = true;
        this.isViewMode = false;
    }
}
