import { Component, OnInit, Input, OnDestroy, ViewChild, OnChanges, Output, EventEmitter } from '@angular/core';
import { FormGroup, FormControl } from '@angular/forms';
import { SharedTableResult } from '../../../models/shared-table/shared-table-result.interface';
import { ITableProperties } from '../../../models/shared-table/table-properties.interface';
import { CoreSession } from '../../../../core/core.session';
import { TranslateService } from '@ngx-translate/core';
import { ConstantMessages } from '../../../models/constants/constant-message';
import { HttpErrorResponse } from '@angular/common/http';
import { DialogResult } from '../../../models/enums/dialog-result.enum';
import { NgbModal, NgbModalConfig } from '@ng-bootstrap/ng-bootstrap';
import { SessionDataProvider } from '../../../../core/session-data-provider.service';
import { BrandService } from '../../../services/data-definition/product/brand.service';

@Component({
  selector: 'app-select-brands',
  templateUrl: './select-brands.component.html',
  styleUrls: ['./select-brands.component.css']
})
export class SelectBrandsComponent implements OnInit, OnChanges, OnDestroy {

  @ViewChild('selectBrandsDialog', { static: true }) selectBrandsDialog: any;
  OkResult: DialogResult = DialogResult.Ok;

  @Input() currentId = 'BrandsSelection';
  @Input() addedBrandsList: any[] = [];
  @Input() isBrandsList: boolean = false;
  @Output() saveEvent = new EventEmitter();

  query;
  public brandsFilterForm: FormGroup;
  showSelectBrandsDialog: boolean = false;

  sharedTabelData: ITableProperties = {
    pageSize: 100,
    showPaginator: true,
    isOnline: false,
    showSearch: true,
    // showSearchBtn: true,
    isMultiSelection: true,
    showEditButton: false,
    rowOperations: [],
    multiSelectionOperations: [],
    specificActionWhenCheckAll: true,
    specificActionWhenCheckRow: true,
    columns: [
      { title: 'Desc_Brands', key: 'brandName', isSortable: true, width: '50%' },
    ]
  };
  dataSource: SharedTableResult = {
    totalItems: 0,
    data: []
  };
  brandsFilter: any = {
    customListFilter: {
      searchFilter: '',
      page: 0,
      pageSize: 100
    },
  }

  constructor(
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private brandService: BrandService,
    private config: NgbModalConfig,
    private modalService: NgbModal,
    private sessionData: SessionDataProvider
  ) {
    this.config.backdrop = true;
    this.config.keyboard = true;
  }
  ngOnInit() {
    if (this.isBrandsList) {
      this.sharedTabelData.columns.push(
        { title: "Desc_Min_Count_Of_Images", key: "minimumNoImages", isEditable: true, numberOfStockDigits: true, isSortable: true, width: '25%' },
        { title: "Desc_Max_Count_Of_Images", key: "maximumNoImages", isEditable: true, numberOfStockDigits: true, isSortable: true, width: '25%' }
      )
    }
  }
  ngOnChanges() {
  }
  ngOnDestroy() {
    this.config.backdrop = true;
    this.config.keyboard = true;
  }
  populateBrands() {
    this.brandsFilter.customListFilter = this.query ? this.query : this.brandsFilter.customListFilter;
    this.coreSession.ModalLoading.Show();
    this.brandService.getAllBrands().subscribe(response => {
      this.coreSession.ModalLoading.Hide();
      if (response.status != null && response.status >= 0) {
        this.reflectAddedBeforeBrands(response);
      } else {
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), response.message);
      }
    }, (error: HttpErrorResponse) => {
      this.coreSession.ModalLoading.Hide();
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.ErrorHappened));
    });
  }
  reflectAddedBeforeBrands(dataSource) {
    if (dataSource && dataSource.data != null && dataSource.data.length > 0) {
      dataSource.data.forEach(newBrands => {
        var index = this.addedBrandsList.findIndex(addedBrands => addedBrands.brandId === newBrands.brandId);
        if (index >= 0) {
          newBrands.isChecked = true;
          newBrands.disableValuesUsed = false;
          newBrands.minimumNoImages = +this.addedBrandsList[index].minimumNoImages;
          newBrands.maximumNoImages = +this.addedBrandsList[index].maximumNoImages;
          newBrands = this.addedBrandsList[index];
        }
      })
    }
    this.dataSource = {
      totalItems: dataSource.totalItems,
      data: dataSource.data
    }
  }
  public closeSelectBrandsDialog() {
    this.modalService.dismissAll()
    this.showSelectBrandsDialog = false;
  }
  onFindBrands() {
    this.populateBrands();
  }
  onClearFilters() {
    this.brandsFilterForm.reset();
  }
  onCheckRowOperation(brands) {
    var index = this.addedBrandsList.findIndex(x => x.brandId === brands.brandId);
    if (brands.isChecked) {
      if (index < 0) {
        this.addedBrandsList.push(brands);
        brands.disableValuesUsed = false;
      }
    } else {
      if (index >= 0) {
        brands.disableValuesUsed = true;
        this.addedBrandsList.splice(index, 1);
      }
    }
  }
  onCheckAllOperation(checkAll: boolean) {
    this.dataSource.data.forEach(brands => {
      var index = this.addedBrandsList.findIndex(x => x.brandId === brands.brandId);
      if (index >= 0 && !checkAll) {
        brands.disableValuesUsed = true;
        this.addedBrandsList.splice(index, 1);
      } else if (index < 0 && checkAll) {
        brands.disableValuesUsed = false;
        this.addedBrandsList.push(brands);
      }
    })
  }
  onTableFilterChanged(query: any) {
    this.brandsFilter.customListFilter = query;
    this.query = query;
    this.populateBrands();
  }
  public showDialog() {
    this.sharedTabelData.columns = [
      { title: 'Desc_Brands', key: 'brandName', isSortable: true, width: '50%' }
    ];
    if (this.isBrandsList) {
      this.sharedTabelData.columns.push(
        { title: "Desc_Min_Count_Of_Images", key: "minimumNoImages", isEditable: true, numberOfStockDigits: true, isSortable: true, width: '25%' },
        { title: "Desc_Max_Count_Of_Images", key: "maximumNoImages", isEditable: true, numberOfStockDigits: true, isSortable: true, width: '25%' }
      )
    }
    this.coreSession.SetTitle('Select Brand');
    this.showSelectBrandsDialog = true;
    this.brandsFilter.customListFilter = {
      searchFilter: '',
      page: 0,
      pageSize: 100
    };
    this.populateBrands();
    return this.modalService.open(this.selectBrandsDialog, { centered: true, size: 'xl' }).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
          return this.addedBrandsList;
        } else {
          return -1;
        }
      });
  }
  onValueChanges(event: any) {
    let changedValue = event.object;
    let minCountImages = isNaN(changedValue.minimumNoImages) ? 0 : +changedValue.minimumNoImages;
    let maxCountImages = isNaN(changedValue.maximumNoImages) ? 0 : +changedValue.maximumNoImages;
    if (!this.addedBrandsList) { this.addedBrandsList = []; }
    let index = this.addedBrandsList.findIndex(a => a.brandId === changedValue.brandId);
    if (index > -1) {
      this.addedBrandsList[index].minimumNoImages = minCountImages;
      this.addedBrandsList[index].maximumNoImages = maxCountImages;
    } else {
      this.addedBrandsList[index].minimumNoImages = minCountImages;
      this.addedBrandsList[index].maximumNoImages = maxCountImages;
    }
  }
  onSaveClick() {
    this.validateBrands()
  }
  validateBrands() {
    let isValid = true;
    this.addedBrandsList.forEach(brand => {
      brand.maximumNoImages = +brand.maximumNoImages;
      brand.minimumNoImages = +brand.minimumNoImages;
      if (brand.maximumNoImages < brand.minimumNoImages && brand.isChecked == true) {
        let errorMsg = this.translateService.instant(ConstantMessages.MsgMinimumLessThanMaximumNoImages);
        this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), errorMsg);
        isValid = false;
      }
    });
    if (isValid)
      this.saveEvent.emit(this.addedBrandsList);
  }
}
