import { CalculationService } from './../../../../services/calculations.service';
import {
  Component,
  OnInit,
  Input,
  OnChanges,
  Output,
  EventEmitter,
  OnDestroy
} from "@angular/core";
import { TranslateService } from "@ngx-translate/core";
import { Observable, } from 'rxjs';
import { CoreSession } from "../../../../../core/core.session";
import { SessionDataProvider } from "../../../../../core/session-data-provider.service";
import { ConstantMessages } from '../../../../models/constants/constant-message';
import { ConstantConfigurations } from '../../../../models/constants/constant-configuration';
import { LimitItemsToWarehouseTypes } from '../../../../models/enums/limit-items-to-warehouse-types.enum';
import { DialogResult } from '../../../../models/enums/dialog-result.enum';
import { DialogMode } from '../../../../models/enums/dialog-mode.enum';

@Component({
  selector: "app-take-bundle",
  templateUrl: "./take-bundle.component.html",
  styleUrls: ["./take-bundle.component.css"]
})
export class TakeBundleComponent implements OnInit, OnChanges, OnDestroy {
  @Output()
  takeBundles: EventEmitter<any> = new EventEmitter();
  @Output()
  cancelPromotion = new EventEmitter();
  @Input() bundles: any[] = []; // Get Current Level Promotion
  promoWithItemSelection: any[] = [];  // Get Current Item Level Promotion
  currentLevelPromoWithItemSelection: any[] = []; // Get All Item Promotion Levels
  lastPromotion: boolean = false; // To check If there is a next level or last level
  currentLevelId = 1;
  nextLevelId = 1;
  @Input() nextEvent: Observable<void>;
  nextSubscription: any;
  limitOrdersToWarehouseStockMessageType: string;
  limitItemsToMainWarehouseStock: boolean = false;
  isValidQty: boolean = false;

  constructor(
    public coreSession: CoreSession,
    private translateService: TranslateService,
    private sessionData: SessionDataProvider,
    private calculationService: CalculationService
  ) { }

  ngOnInit() {
    this.subscribeNextClick();
    this.limitOrdersToWarehouseStockMessageType = this.sessionData.getConfigurationValue(ConstantConfigurations.LimitOrdersToWarehouseStockMessageType).toLowerCase();
    this.limitItemsToMainWarehouseStock = this.sessionData.getConfigurationValue(ConstantConfigurations.LimitItemsToMainWarehouseStock).toLowerCase().includes(LimitItemsToWarehouseTypes.Order.valueOf() + "");
  }

  ngOnChanges() {
  }
  ngOnDestroy() {
    this.nextSubscription.unsubscribe();
  }
  subscribeNextClick() {
    this.nextSubscription = this.nextEvent.subscribe(() => {
      // this.onNextClicked();
    });
  }
  onNextClicked() {
    if (this.bundles.findIndex(x => x.originalRepeatCount > 0) < 0) {
      let errorMsg = this.translateService.instant(ConstantMessages.MsgEnterQuantity);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), errorMsg);
      return;
    }
    if (this.limitOrdersToWarehouseStockMessageType.includes("2") && this.bundles.findIndex(x => x.bundleAvailableQty < x.originalRepeatCount && x.originalRepeatCount > 0) >= 0) {
      let errorMsg = this.translateService.instant(ConstantMessages.RequiredMoreThanAvailable);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), errorMsg);
      return;
    }
    if (this.bundles.findIndex(x => (x.bundleLimit - x.usedQuantity) < x.originalRepeatCount && x.originalRepeatCount > 0 && x.bundleLimit > 0) >= 0) {
      let errorMsg = this.translateService.instant(ConstantMessages.RequiredMoreThanBundleLimit);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), errorMsg);
      return;
    }
    this.fillPromotionNeedItemSelection();
    this.currentLevelPromoWithItemSelection = this.promoWithItemSelection.filter(x => x.levelId == this.currentLevelId);
    let promotions = this.bundles;
    let lastPromotion = this.lastPromotion;
    let currentLevelId = this.currentLevelId;
    let nextLevelId = this.nextLevelId;
    this.takeBundles.emit({ promotions, lastPromotion, currentLevelId, nextLevelId });
  }
  onCancelPromotionClicked() {
    this.cancelPromotion.emit();
  }

  fillPromotionNeedItemSelection() {
    let bundles = this.bundles.filter(x => x.originalRepeatCount > 0)
    for (var promo of bundles) {
      promo.originalRepeatCount = +promo.originalRepeatCount; // convert to number
      promo.newRepeatCount = promo.originalRepeatCount;
      promo.promotionOutputs.forEach(output => {
        output.isSelected = true;
      });
      promo.isSelected = true;
    }
  }
  onChangeQty(currentBundle) {
    var itemIds = currentBundle.bundleItems.map(b => b.itemId);
    var bundleToChange = this.bundles.filter(x => x.bundleItems.find(item => itemIds.includes(item.itemId)));

    bundleToChange.forEach(bundle => {
      bundle.bundleItems.forEach(bundleItem => {
        let otherBundles = this.bundles.filter(x => x.promotionId != bundle.promotionId && x.bundleItems.find(y => y.itemId == bundleItem.itemId));
        let used = 0;
        otherBundles.forEach(otherBundle => {
          otherBundle.bundleItems.filter(x => x.itemId === bundleItem.itemId).forEach(item => {
            if (item.itemId == bundleItem.itemId)
              used += otherBundle.originalRepeatCount * item.quantity * item.packQuantity;
          });
        })
        let qtyforSameItem = 0;
        let packQty = 0;
        let qty = 0;
        if (bundle.bundleItems.findIndex(x => x.packId != bundleItem.packId && x.itemId == bundleItem.itemId) > -1) {
          packQty = bundle.bundleItems.find(x => x.packId != bundleItem.packId && bundleItem.itemId == x.itemId).packQuantity;
          qty = bundle.bundleItems.find(x => x.packId != bundleItem.packId && bundleItem.itemId == x.itemId).quantity;
          qtyforSameItem += qty * packQty;
        }
        bundleItem.availableQtyInPcs = (bundleItem.originalAvailableQty - used) / ((bundleItem.quantity * bundleItem.packQuantity) + qtyforSameItem);
      });
    });
    bundleToChange.forEach(bundle => {
      bundle.bundleAvailableQty = Math.min(...bundle.bundleItems.map(a => a.availableQtyInPcs));
      bundle.bundleAvailableQty = Math.floor(bundle.bundleAvailableQty);
    });
  }
  checkQty(bundle) {
    this.isValidQty = true;
    if (this.limitItemsToMainWarehouseStock) {
      if (bundle.bundleAvailableQty >= +bundle.originalRepeatCount) {
        this.isValidQty = true;
        if (bundle.bundleLimit > 0) {
          this.checkBundleLimitValidation(bundle);
        }
      } else {
        if (this.limitOrdersToWarehouseStockMessageType.includes("1") && bundle.bundleAvailableQty < +bundle.originalRepeatCount) {
          this.coreSession.ModalDialog.ShowMessage(this.translateService.instant(ConstantMessages.MsgExceededAllowedQuantity).replace('{0}', this.calculationService.getNumberOnStockDigitFormat(bundle.bundleAvailableQty).toString()), DialogMode.YesNo).then(
            (res: DialogResult) => {
              if (res === DialogResult.Yes) {
                this.onChangeQty(bundle);
              } else {
                bundle.originalRepeatCount = 0;
                this.isValidQty = false;
                this.onChangeQty(bundle);
              }
            });
        } else if (this.limitOrdersToWarehouseStockMessageType.includes("2") && bundle.bundleAvailableQty < +bundle.originalRepeatCount) {
          let errorMsg = this.translateService.instant(ConstantMessages.RequiredMoreThanAvailable);
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), errorMsg);
          bundle.originalRepeatCount = 0;
          this.onChangeQty(bundle);
        }
      }
    } else if (bundle.bundleLimit > 0) {
      this.checkBundleLimitValidation(bundle);
    }
    else {
      this.isValidQty = true;
    }
    return this.isValidQty;
  }
  checkBundleLimitValidation(bundle) {
    var maxLimit = bundle.bundleLimit - bundle.usedQuantity;
    if (maxLimit < +bundle.originalRepeatCount) {
      this.isValidQty = false;
      let warningMsg = this.translateService.instant(ConstantMessages.RequiredMoreThanBundleLimit);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), warningMsg);
      bundle.originalRepeatCount = 0;
      this.onChangeQty(bundle);
    } else {
      this.onChangeQty(bundle);
    }
  }
}

