import { Component, OnInit, Input, ViewChild, OnChanges, Output, EventEmitter } from '@angular/core';
import { Observable } from 'rxjs';
import { DialogResult } from '../../models/enums/dialog-result.enum';
import { FormGroup, Validators, FormControl } from '@angular/forms';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { CoreSession } from '../../../core/core.session';
import { TranslateService } from '@ngx-translate/core';
import { ConstantMessages } from '../../models/constants/constant-message';
import { DateTimeModel } from '../../models/dateTimeModel/date-time.model';
import { IDateTimePickerProperties } from '../../models/dateTimeModel/date-time-properties.interface';
import { EmailTemplateModel } from '../../models/email/email.model';
import { HttpErrorResponse } from '@angular/common/http';
import { EmailService } from '../../../content/admin-settings/email/email.service';

@Component({
  selector: 'app-service-setup-dialog',
  templateUrl: './service-setup-dialog.component.html',
  styleUrls: ['./service-setup-dialog.component.css']
})
export class ServiceSetupDialogComponent implements OnInit, OnChanges {

  //#region [DECLARATIONS]

  @Input() events: Observable<void>;
  @Input() serviceSetupObj: any = {};
  @Output() emailTemp = new EventEmitter<any>();
  emailTempObj: EmailTemplateModel;
  @ViewChild('content', { static: true }) content: any;
  OkResult: DialogResult = DialogResult.Ok;
  form: FormGroup;
  emailConfiguartionForm: FormGroup;
  showEmailTemp : boolean = false;
  timeToRunDate: DateTimeModel;
  timeToRunDateProperties: IDateTimePickerProperties = {
    label: 'Desc_Time_To_Run',
    formControlName: 'timeToRunDate',
  };
  hideMinutesTosleep: boolean = false;
  //#endregion

  //#region [CONSTRUCTOR]
  constructor(
    private modalService: NgbModal,
    private coreSession: CoreSession,
    private translateService: TranslateService,
    private emailService: EmailService,
  ) { }

  ngOnInit() {
    this.emailTempObj = new EmailTemplateModel();
    this.initForm();
  }
  ngOnChanges(){
    if (this.serviceSetupObj && this.serviceSetupObj.serviceId > 0){
      this.fillFormInEditMode();
      this.onCertainTimeChange();
    }
  }
  //#endregion


  //#region [EVENTS]  

  onSaveClicked(modal) {
    this.fillServiceSetupObject();
    if (this.form.invalid) {
      this.coreSession.markFormGroupTouched(this.form as FormGroup);
      this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
      return;
    }
    if (this.serviceSetupObj.isCertainTimeRuning && (!this.timeToRunControl || (this.timeToRunControl.value == null))) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgSelectDate));
      return;
    }
    if (!this.serviceSetupObj.isCertainTimeRuning && (!this.numberOfMinutesControl || (this.numberOfMinutesControl.value == null))) {
      this.coreSession.showError(this.translateService.instant(ConstantMessages.ErrorCaption), this.translateService.instant(ConstantMessages.MsgFillNumberOfMin));
      return;
    }
    if(this.showEmailTemp){
      if (this.emailConfiguartionForm.invalid) {
        if (this.email.invalid) {
          this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgInsertValidEmail));
          return;
        }
        this.coreSession.markFormGroupTouched(this.emailConfiguartionForm as FormGroup);
        this.coreSession.showWarrning(this.translateService.instant(ConstantMessages.WarningCaption), this.translateService.instant(ConstantMessages.MsgFillMandatory));
        return;
      }
      this.emailTemp.emit(this.emailTempObj);
    }
    this.saveData(modal);
  }

  getReportScheduleTemplateDetail() {
    this.coreSession.ModalLoading.Show();
    this.emailService
      .getReportScheduleTemplateDetail().subscribe(
        response => {
          this.coreSession.ModalLoading.Hide();
          if (response.status != null && response.status >= 0) {
            this.emailTempObj = response.data;
            this.fillFormInEditMode();
          } else {
            this.coreSession.showError(
              "",
              this.translateService.instant(response.message)
            );
          }
        },
        (error: HttpErrorResponse) => {
          this.coreSession.ModalLoading.Hide();
          this.coreSession.showError(ConstantMessages.ErrorHappened, "");
        }
      );
  }

  onDateSelected(date: any) {
    if (date)
      this.timeToRunDate = date;
  }

  onCertainTimeChange() {
    if(this.serviceSetupObj.serviceTitle === "ReportScheduleService"){
      this.getReportScheduleTemplateDetail();
      this.hideMinutesTosleep = true;
      this.showEmailTemp = true;
      this.serviceSetupObj.isCertainTimeRuning = true;
      this.certainTimeControl.setValue(this.serviceSetupObj.isCertainTimeRuning);
      this.numberOfMinutesControl.setValue(null);
      this.numberOfMinutesControl.disable();
      this.timeToRunControl.enable();
    }else {
      this.hideMinutesTosleep = false;
      this.showEmailTemp = false;
      if (this.certainTimeControl.value) {
        this.numberOfMinutesControl.setValue(null);
        this.numberOfMinutesControl.disable();
        this.timeToRunControl.enable();
      }
      else {
        this.timeToRunControl.setValue(null);
        this.timeToRunControl.disable();
        this.numberOfMinutesControl.enable();
      }
    }
  }

  //#endregion

  //#region [METHODS]
  saveData(modal) {
    modal.close(DialogResult.Ok);

  }
  fillServiceSetupObject() {
    this.emailTempObj = new EmailTemplateModel();
    this.serviceSetupObj.isCertainTimeRuning = this.certainTimeControl ? this.certainTimeControl.value : false;
    //this.serviceSetupObj.timeToRun = this.timeToRunControl ? this.timeToRunControl.value : null;
    this.serviceSetupObj.numberOfMinutesToSleep = + (this.numberOfMinutesControl ? this.numberOfMinutesControl.value : 0);
    this.serviceSetupObj.timeToRunModel = this.timeToRunDate;
    this.serviceSetupObj.isWebService = this.isWebService.value;

    this.emailTempObj.serverAddress = this.address.value;
    this.emailTempObj.email = this.email.value;
    this.emailTempObj.emailPort = +this.port.value;
    this.emailTempObj.password = this.password.value;
    if (this.enableSSL.value == undefined) this.emailTempObj.enableSSL = false;
    else this.emailTempObj.enableSSL = this.enableSSL.value;
  }
  initForm() {
    this.form = new FormGroup({
      certainTime: new FormControl(false),
      isWebService: new FormControl(false),

      timeToRunDate: new FormControl(
        {
          value: this.timeToRunDate,
          disabled: false
        }
      ),
      numberOfMinutes: new FormControl(null)
    });

    this.emailConfiguartionForm = new FormGroup({
      address: new FormControl(this.emailTempObj.serverAddress, Validators.required),
      port: new FormControl(this.emailTempObj.emailPort, Validators.required),
      email: new FormControl(this.emailTempObj.email, [Validators.required, Validators.email, Validators.pattern('^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,4}$')]),
      psw: new FormControl(this.emailTempObj.password, Validators.required),
      enableSSL: new FormControl(this.emailTempObj.enableSSL),
    });
  }
  fillFormInEditMode() {
    this.certainTimeControl.setValue(this.serviceSetupObj.isCertainTimeRuning);
    this.numberOfMinutesControl.setValue(!this.serviceSetupObj.isCertainTimeRuning ? this.serviceSetupObj.numberOfMinutesToSleep : null);
    this.timeToRunControl.setValue(this.serviceSetupObj.isCertainTimeRuning ? this.serviceSetupObj.timeToRun : null);
    this.isWebService.setValue(this.serviceSetupObj.isWebService);
    if (this.showEmailTemp && this.emailTempObj !== null) {
      this.address.setValue(this.emailTempObj.serverAddress !== null ? this.emailTempObj.serverAddress : null);
      this.email.setValue(this.emailTempObj.email !== null ? this.emailTempObj.email : null);
      this.port.setValue(this.emailTempObj.emailPort !== null ? this.emailTempObj.emailPort : null);
      this.password.setValue(this.emailTempObj.password !== null ? this.emailTempObj.password : null);
      this.enableSSL.setValue(this.emailTempObj.enableSSL !== null ? this.emailTempObj.enableSSL : null);
    }
    // disable 
    if (!this.serviceSetupObj.isCertainTimeRuning) {
      this.timeToRunControl.setValue(null);
      this.timeToRunControl.disable();
      this.numberOfMinutesControl.enable();
    }
    else {
      this.timeToRunControl.enable();
      this.numberOfMinutesControl.setValue(null);
      this.numberOfMinutesControl.disable();
    }
  }


  public showDialog(size?: string) {
      this.onCertainTimeChange();
    return this.modalService.open(this.content, { centered: true , size : size}).result.then(
      (result) => {
        if (result === DialogResult.Ok) {
          return this.serviceSetupObj;
        } else {
          return -1;
        }
      });
  }
  //#endregion

  //#region [PROPERTIES]

  get certainTimeControl() {
    return this.form.get('certainTime');
  }
  get timeToRunControl() {
    return this.form.get('timeToRunDate');
  }
  get numberOfMinutesControl() {
    return this.form.get('numberOfMinutes');
  }
  get isWebService() {
    return this.form.get('isWebService');
  }
  get address() {
    return this.emailConfiguartionForm.get("address");
  }
  get port() {
    return this.emailConfiguartionForm.get("port");
  }
  get email() {
    return this.emailConfiguartionForm.get("email");
  }
  get password() {
    return this.emailConfiguartionForm.get("psw");
  }
  get enableSSL() {
    return this.emailConfiguartionForm.get("enableSSL");
  }
  //#endregion
}
