﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.Model.Customer;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;
using static EchoCustomerOutletModel;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class CustomerController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly CustomerService _customerService;
        private readonly MasterDataService _masterDataService;
        private readonly IHostingEnvironment _hostingEnvironment;
        private readonly IWebHostEnvironment _webHostEnvironment;

        public CustomerController(IRequestRepository requestRepository, IWebHostEnvironment webHostEnvironment, ILicensing licensing, IHostingEnvironment hostingEnvironment, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _requestRepository = requestRepository;
            _customerService = new CustomerService(_requestRepository);
            _masterDataService = new MasterDataService(requestRepository);
            _hostingEnvironment = hostingEnvironment;
            _webHostEnvironment = webHostEnvironment;
        }

        #region GET

        [HttpPost]
        [ActionName("GetCustomerSubChannels")]
        public JsonResult GetCustomerSubChannels([FromBody] GeneralFilter filter)
        {
            try
            {
                List<CustomerSubChannelModel> subChannelsList = new List<CustomerSubChannelModel>();
                var result = _customerService.GetCustomerSubChannels(filter, ref subChannelsList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(subChannelsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.NoMoreRows)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetCustomers")]
        public JsonResult GetCustomers([FromBody] OrderListFilter filter)
        {
            try
            {
                //OrderListFilter orderListFilter = new OrderListFilter();
                //if (filter != null)
                //    orderListFilter = JsonConvert.DeserializeObject<OrderListFilter>(filter);
                var custList = new SharedTableResult<CustomerModel>();
                var result = _customerService.GetCustomerList(filter, ref custList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(custList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.NoMoreRows)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerLazyLoading")]
        public JsonResult GetCustomerLazyLoading([FromBody] GeneralFilter filter)
        {
            try
            {
                var custList = new SharedTableResult<CustomerModel>();

                var result = _customerService.GetCustomerLazyLoading(filter, ref custList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(custList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.NoMoreRows)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetOutletsLazyLoading")]
        public JsonResult GetOutletsLazyLoading([FromBody] CustomListFilters filter)
        {
            try
            {
                var outList = new SharedTableResult<CustomerOutletModel>();
                int groupId = -1;
                int territoryId = -1;
                int classId = -1;
                bool IsWitGetTerritories = false;
                if (filter != null)
                {
                    groupId = filter.GroupId;
                    territoryId = filter.TerritoryId;
                    classId = filter.ClassId;
                    IsWitGetTerritories = filter.IsWitGetTerritories;
                }
                var result = _customerService.GetOutletsLazyLoading(filter, ref outList, groupId, territoryId, classId, IsWitGetTerritories);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(outList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.NoMoreRows)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllCustomers")]
        public JsonResult GetAllCustomers()
        {
            try
            {
                var custList = new List<CustomerModel>();
                var result = _customerService.GetAllCustomersList(ref custList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(custList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("IsValidToRemoveAccount")]
        public JsonResult IsValidToRemoveAccount([FromBody] GeneralFilter filter)
        {
            try
            {
                int accountId = -1;
                if (filter != null)
                {
                    accountId = filter.AccountId;
                }
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                bool isUsed = false;
                GlobalErrors result = _customerService.IsValidToDeleteAccount(accountId, ref isUsed);
                if (result == GlobalErrors.Success)
                {
                    if (isUsed)
                    {
                        return JSonResultResponse.GetSuccessrJSon(false, ResourcesManager.TranslateKey(MessagesConstants.MessageCantDeleteUsedAccount, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerData")]
        public JsonResult GetCustomerData([FromBody] GeneralFilter filter)
        {
            try
            {
                CustomerModel customerObj = new CustomerModel();
                int customerId = -1;
                bool showInActive = false;
                if (filter != null)
                {
                    customerId = filter.CustomerId;
                    showInActive = filter.ShowInActive;
                }
                customerObj.CustomerId = customerId;
                List<string> attachedFiles = new List<string>();
                GlobalErrors result = _customerService.GetCustomerData(customerObj.CustomerId, showInActive, ref customerObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetOutletData")]
        public JsonResult GetOutletData([FromBody] GeneralFilter filter)
        {
            try
            {
                CustomerOutletModel outletObj = new CustomerOutletModel();
                int customerId = -1;
                int outletId = -1;
                bool hasPrivilegeOnPaymentTermDivision = false;
                bool hasPrivilegeOnOnHoldDivision = false;
                if (filter != null)
                {
                    customerId = filter.CustomerId;
                    outletId = filter.OutletId;
                    hasPrivilegeOnPaymentTermDivision = filter.hasPrivilegeOnPaymentTermDivision;
                    hasPrivilegeOnOnHoldDivision = filter.hasPrivilegeOnOnHoldDivision;
                }
                outletObj.CustomerId = customerId;
                outletObj.OutletId = outletId;
                outletObj.HasPrivilegeOnPaymentTermDivision = hasPrivilegeOnPaymentTermDivision;
                outletObj.HasPrivilegeOnOnHoldDivision = hasPrivilegeOnOnHoldDivision;
                GlobalErrors result = _customerService.GetOutletData(outletObj.CustomerId, outletObj.OutletId, hasPrivilegeOnPaymentTermDivision, hasPrivilegeOnOnHoldDivision, ref outletObj);
                // get attached files
                if (result == GlobalErrors.Success)
                {
                    List<string> attachementNames = new List<string>();
                    FileManagerModel model = new FileManagerModel();
                    // For Attachements :
                    string custOut = outletObj.CustomerId + "_" + outletObj.OutletId;
                    string rootPath = string.Format(@"{0}", LocalUtilities.CustomerAppServicePath);
                    string folderName = "Echo\\identification\\" + custOut;
                    var userAttachementPath = Path.Combine(rootPath, folderName);
                    DirectoryInfo Attachementdir = new DirectoryInfo(userAttachementPath);
                    if (Attachementdir.Exists)
                    {
                        FileInfo[] files = Attachementdir.GetFiles();
                        model.Files = files;
                        if (files != null && files.Length > 0)
                        {
                            foreach (FileInfo file in files)
                            {
                                attachementNames.Add(file.Name);
                            }
                        }
                    }
                    if (attachementNames != null && attachementNames.Count > 0)
                    {
                        outletObj.AttachedFiles = attachementNames;
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(outletObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("IsValidToDeleteCustomer")]
        public JsonResult IsValidToDeleteCustomer([FromBody] GeneralFilter filter)
        {
            try
            {
                CustomerModel customerObj = new CustomerModel();
                int customerId = -1;
                if (filter != null)
                {
                    customerId = filter.CustomerId;
                }
                customerObj.CustomerId = customerId;
                List<string> tables = new List<string>();
                GlobalErrors result = _customerService.IsValidToDeleteCustomer(customerObj.CustomerId, ref customerObj, ref tables);
                if (result != GlobalErrors.Success)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                }
                else
                {
                    if (customerObj.CustomerValidations.IsDefaultCustomer)
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Delete_Default_Customer, _requestRepository.LanguageId));
                    else if (customerObj.CustomerValidations.IsIncludedInTables)
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Delete_Customer_Has_Data_In_Tables, _requestRepository.LanguageId) + string.Join(",", tables.Select(m => m).ToArray()));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }

                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("isValidToDeleteCustomerOutlet")]
        public JsonResult isValidToDeleteCustomerOutlet([FromBody] GeneralFilter filter)
        {
            try
            {
                CustomerOutletModel customerObj = new CustomerOutletModel();
                int customerId = -1;
                int outletId = -1;
                if (filter != null)
                {
                    customerId = filter.CustomerId;
                    outletId = filter.OutletId;
                }
                customerObj.CustomerId = customerId;
                customerObj.OutletId = outletId;
                List<string> tables = new List<string>();
                GlobalErrors result = _customerService.IsValidToDeleteCustomerOutlet(customerObj.CustomerId, customerObj.OutletId, ref customerObj, ref tables);
                if (result != GlobalErrors.Success)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                }
                else
                {
                    if (customerObj.CustomerValidations.IsDefaultCustomer)
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Delete_Default_Customer, _requestRepository.LanguageId));
                    else if (customerObj.CustomerValidations.IsIncludedInTables)
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Delete_Customer_Has_Data_In_Tables, _requestRepository.LanguageId) + string.Join(",", tables.Select(m => m).ToArray()));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }

                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerStatus")]
        public JsonResult GetCustomerStatus()
        {
            try
            {
                string token = this.Request.Headers["Authorization"].ToString();
                List<CustomerStatusModel> customerStatusObj = new List<CustomerStatusModel>();
                GlobalErrors result = _customerService.GetCustomerStatus(ref customerStatusObj, 1);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerStatusObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerTypes")]
        public JsonResult GetCustomerTypes()
        {
            try
            {
                string token = this.Request.Headers["Authorization"].ToString();
                List<CustomerTypeModel> customerTypesObj = new List<CustomerTypeModel>();
                GlobalErrors result = _customerService.GetCustomerTypes(ref customerTypesObj, 1);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerTypesObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerClasses")]
        public JsonResult GetCustomerClasses()
        {
            try
            {
                List<CustomerClassModel> customerClassObj = new List<CustomerClassModel>();
                GlobalErrors result = _customerService.GetCustomerClasses(ref customerClassObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerClassObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerSubClasses")]
        public JsonResult GetCustomerSubClasses()
        {
            try
            {
                List<CustomerSubClassModel> customerSubClassObj = new List<CustomerSubClassModel>();
                GlobalErrors result = _customerService.GetCustomerSubClasses(ref customerSubClassObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerSubClassObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetPaymentTerms")]
        public JsonResult GetPaymentTerms()
        {
            try
            {
                List<PaymentTermModel> paymentTermObj = new List<PaymentTermModel>();
                GlobalErrors result = _customerService.GetPaymentTerms(ref paymentTermObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(paymentTermObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetPaymentTermTypes")]
        public JsonResult GetPaymentTermTypes()
        {
            try
            {
                List<PaymentTermTypeModel> paymentTermObj = new List<PaymentTermTypeModel>();
                GlobalErrors result = _customerService.GetPaymentTermTypes(ref paymentTermObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(paymentTermObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetPeriodTypes")]
        public JsonResult GetPeriodTypes()
        {
            try
            {
                List<PeriodTypeModel> periodTypeObj = new List<PeriodTypeModel>();
                GlobalErrors result = _customerService.GetPeriodTypes(ref periodTypeObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(periodTypeObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllCustomerGroups")]
        public JsonResult GetAllCustomerGroups([FromBody] GeneralFilter filter)
        {
            try
            {
                List<CustomerGroupModel> customerGroupObj = new List<CustomerGroupModel>();
                int channelId = -1;
                int subChannelId = -1;
                string organizationIds = string.Empty;
                if (filter != null)
                {
                    channelId = filter.ChannelId;
                    subChannelId = filter.SubChannelId;
                    organizationIds = filter.OrganizationIds;
                }
                GlobalErrors result = _customerService.GetAllCustomerGroups(channelId, subChannelId, organizationIds, ref customerGroupObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerGroupObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllChannels")]
        public JsonResult GetAllChannels()
        {
            try
            {
                List<CustomerChannelModel> channelList = new List<CustomerChannelModel>();
                GlobalErrors result = _customerService.GetAllChannels(ref channelList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(channelList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllSubChannels")]
        public JsonResult GetAllSubChannels([FromBody] GeneralFilter filter)
        {
            try
            {
                List<CustomerSubChannelModel> subChannelList = new List<CustomerSubChannelModel>();
                int channelId = -1;
                if (filter != null)
                {
                    channelId = filter.ChannelId;
                }
                GlobalErrors result = _customerService.GetAllSubChannels(channelId, ref subChannelList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(subChannelList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllCustomerChannels")]
        public JsonResult GetAllCustomerChannels([FromBody] GeneralFilter filter)
        {
            try
            {
                List<CustomerChannelModel> customerGroupObj = new List<CustomerChannelModel>();
                int groupId = -1;

                if (filter != null)
                {
                    groupId = filter.groupId;
                }
                GlobalErrors result = _customerService.GetAllCustomerChannels(ref customerGroupObj, groupId.ToString());
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerGroupObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }


        [HttpPost]
        [ActionName("IsCustomerHasUnpaidTransactions")]
        public JsonResult IsCustomerHasUnpaidTransactions([FromBody] GeneralFilter filter)
        {
            try
            {
                int customerId = -1;
                int outletId = -1;
                if (filter != null)
                {
                    customerId = filter.CustomerId;
                    outletId = filter.OutletId;
                }
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                bool isHasUnpaidTransactions = false;
                GlobalErrors result = _customerService.IsCustomerHasUnpaidTransactions(customerId, outletId, ref isHasUnpaidTransactions);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(isHasUnpaidTransactions, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CallCustOutLocationProcedure")]
        public JsonResult CallCustOutLocationProcedure([FromBody] List<CustomerModel> customer)
        {
            try
            {
                GlobalErrors result = GlobalErrors.NotInitialized;
                result = _customerService.CallInvoiceLocationProcedure(customer);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        //[HttpGet]
        //[ActionName("GetNewCustomerList")]
        //public JsonResult GetNewCustomerList(string filter)
        //{
        //    try
        //    {
        //        SharedTableResult<CustomerOutletModel> newCustomers = new SharedTableResult<CustomerOutletModel>();
        //        GeneralFilter generalFilter = JsonConvert.DeserializeObject<GeneralFilter>(filter);
        //        GlobalErrors result = _customerService.GetNewCustomerList(generalFilter, ref newCustomers);


        //        if (result == GlobalErrors.Success)
        //        {
        //            if (newCustomers != null && newCustomers.Data != null && newCustomers.Data.ToList().Count > 0)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(newCustomers, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(newCustomers, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //            }
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //[HttpGet]
        //[ActionName("getCustomerAttachements")]
        //public JsonResult getCustomerAttachements(string customerId)
        //{
        //    List<string> attachementNames = new List<string>();
        //    try
        //    {
        //        // get attached files
        //        {
        //            FileManagerModel model = new FileManagerModel();
        //            // For Attachements :
        //            string custOut = customerId + "_" + "1";
        //            string rootPath = string.Format(@"{0}", LocalUtilities.CustomerAppServicePath);
        //            string folderName = "Echo\\identification\\" + custOut;
        //            var userAttachementPath = Path.Combine(rootPath, folderName);
        //            DirectoryInfo Attachementdir = new DirectoryInfo(userAttachementPath);
        //            if (Attachementdir.Exists)
        //            {
        //                FileInfo[] files = Attachementdir.GetFiles();
        //                model.Files = files;
        //                if (files != null && files.Length > 0)
        //                {
        //                    foreach (FileInfo file in files)
        //                    {
        //                        attachementNames.Add(file.Name);
        //                    }
        //                }
        //            }
        //        }

        //        if (attachementNames != null && attachementNames.Count > 0)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(attachementNames, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //        }

        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}

        [HttpPost]
        [ActionName("GetAllEchoCustomers")]
        public JsonResult GetAllEchoCustomers()
        {
            try
            {
                SharedTableResult<EchoCustomersModel> allEchoCustomers = new SharedTableResult<EchoCustomersModel>();
                GlobalErrors result = _customerService.GetAllEchoCustomers(ref allEchoCustomers);
                if (result == GlobalErrors.Success)
                {
                    if (allEchoCustomers != null && allEchoCustomers.Data != null && allEchoCustomers.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(allEchoCustomers, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(allEchoCustomers, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllEchoCustomerChannels")]
        public JsonResult GetAllEchoCustomerChannels()
        {
            try
            {
                List<CustomerChannelModel> customerGroupObj = new List<CustomerChannelModel>();
                GlobalErrors result = _customerService.GetAllEchoCustomerChannels(ref customerGroupObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerGroupObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetAllEchoCustomerGroups")]
        public JsonResult GetAllEchoCustomerGroups()
        {
            try
            {
                List<CustomerGroupModel> customerGroupObj = new List<CustomerGroupModel>();
                GlobalErrors result = _customerService.GetAllEchoCustomerGroups(ref customerGroupObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerGroupObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("CheckIfContactDataExists")]
        public JsonResult CheckIfContactDataExists([FromBody] CustomerOutletContactModel customerOutletContact)
        {
            try
            {
                string errorMessage = string.Empty;
                GlobalErrors result = _customerService.CheckIfContactDataExists(customerOutletContact, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    if (string.IsNullOrEmpty(errorMessage))
                    {
                        return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon(errorMessage);
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetCustomerOperatorsList")]
        public JsonResult GetCustomerOperatorsList([FromBody] GeneralFilter filter)
        {
            try
            {
                string outletCode = string.Empty;
                if (filter != null)
                {
                    outletCode = filter.OutletCode;
                }
                List<CustomerOutletModel> operatorsList = new List<CustomerOutletModel>();
                GlobalErrors result = _customerService.GetCustomerOperatorsList(outletCode, ref operatorsList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(operatorsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("DownloadFile")]
        public async Task<FileStream> DownloadFile(string id, string fileName)
        {
            string operationFileName = "";
            try
            {
                if (string.IsNullOrEmpty(fileName))
                {
                    return null;
                }
                string rootPath = string.Format(@"{0}", LocalUtilities.CustomerAppServicePath);
                string folderName = "Echo\\identification\\" + id;
                var path = Path.Combine(rootPath, folderName, fileName);

                return new FileStream(path, FileMode.Open, FileAccess.Read);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        [HttpPost]
        [ActionName("GetOutletImages")]
        public JsonResult GetOutletImages([FromBody] GeneralFilter filter)
        {
            try
            {
                string CustomerOutletId = string.Empty;
                if (filter != null && !string.IsNullOrEmpty(filter.CustomerOutletId))
                {
                    CustomerOutletId = filter.CustomerOutletId;
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                List<string> fileNames = new List<string>();
                string fileName = string.Empty;
                string targetFolderPath = _webHostEnvironment.WebRootPath + "\\OutletImages";
                string targetFile = targetFolderPath + "\\" + CustomerOutletId;
                string sourceFilePath = LocalUtilities.OutletImagesSourcePath;
                string sourceFilePathForEcho = LocalUtilities.SonicWebPath + "\\" + CustomerOutletId;
                if (!Directory.Exists(targetFolderPath))
                {
                    Directory.CreateDirectory(targetFolderPath);
                }
                var isFileExistInEcho = Directory.Exists(sourceFilePathForEcho) ? Directory.GetFiles(sourceFilePathForEcho).ToList() : new List<string>();
                var isFileExistInAndroid = Directory.Exists(sourceFilePath) ? Directory.GetFiles(sourceFilePath, CustomerOutletId + "*.*").ToList() : new List<string>();

                if (isFileExistInEcho != null && isFileExistInEcho.Count > 0)
                {
                    _customerService.GetImagesToRootFolder(targetFolderPath, sourceFilePathForEcho, CustomerOutletId);
                }
                if (isFileExistInAndroid != null && isFileExistInAndroid.Count > 0)
                {
                    _customerService.GetImagesToRootFolder(targetFolderPath, string.Empty, CustomerOutletId + "_");
                }
                List<string> filePaths = Directory.GetFiles(targetFolderPath, CustomerOutletId + "_" + "*.*").ToList();
                if (filePaths != null && filePaths.Count > 0)
                {
                    List<string> ImageExtensions = new List<string> { ".JPG", ".JPEG", ".JPE", ".BMP", ".GIF", ".PNG" };
                    foreach (string file in filePaths)
                    {
                        fileName = file.Substring(file.LastIndexOf("\\") + 1);
                        if (ImageExtensions.Contains(Path.GetExtension(fileName).ToUpperInvariant()))
                        {
                            fileName = file.Substring(file.LastIndexOf("\\") + 1);
                            if (!fileNames.Contains("wwwroot/OutletImages/" + fileName))
                            {
                                fileNames.Add("wwwroot/OutletImages/" + fileName);
                            }
                        }
                    }
                    return JSonResultResponse.GetSuccessrJSon(fileNames, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        //#region [Price Protection]

        //[HttpPost]
        //[ActionName("GetPriceProtectionList")]
        //public JsonResult GetPriceProtectionList([FromBody] PriceProtectionFilterModel priceProtectionFilter)
        //{
        //    try
        //    {
        //        SharedTableResult<PriceProtectionModel> priceProtectionList = null;
        //        GlobalErrors result = GlobalErrors.NotInitialized;
        //        if (priceProtectionFilter != null)
        //        {
        //            priceProtectionList = new SharedTableResult<PriceProtectionModel>();
        //            result = _customerService.GetPriceProtection(priceProtectionFilter, ref priceProtectionList);
        //        }
        //        if (result == GlobalErrors.Success)
        //        {
        //            if (priceProtectionList != null && priceProtectionList.Data != null && priceProtectionList.Data.ToList().Count > 0)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(priceProtectionList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(priceProtectionList, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //            }
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //[HttpPost]
        //[ActionName("GetPriceProtectionData")]
        //public JsonResult GetPriceProtectionData([FromBody] GeneralFilter filter)
        //{
        //    try
        //    {
        //        string requestId = string.Empty;
        //        if(filter != null)
        //        {
        //            requestId = filter.RequestId;
        //        }
        //        List<PriceProtectionDetailModel> priceProtectionDetail = null;
        //        GlobalErrors result = GlobalErrors.NotInitialized;
        //        if (!string.IsNullOrEmpty(requestId))
        //        {
        //            result = _customerService.GetPriceProtectionData(requestId, ref priceProtectionDetail);
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //        if (result == GlobalErrors.Success)
        //        {
        //            if (priceProtectionDetail != null && priceProtectionDetail != null && priceProtectionDetail.Count > 0)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(priceProtectionDetail, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(priceProtectionDetail, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //            }
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }

        //}
        //#endregion

        #endregion

        #region POST
        
        [HttpPost]
        [ActionName("GetCustomerOutlets")]
        public JsonResult GetCustomerOutlets([FromBody] GeneralFilter filter)
        {
            try
            {
                var custOutList = new List<CustomerOutletModel>();
                var result = _customerService.GetCustomerOutlets(filter, ref custOutList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(custOutList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetCustomerCommnet")]
        public JsonResult GetCustomerComment([FromBody] int customerId)
        {
            try
            {
                var commnetList = new List<CustomerComment>();
                var result = _customerService.GetCustomerComment(customerId, ref commnetList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(commnetList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("addCustomerCommnet")]
        public JsonResult addCustomerComment([FromBody] CustomerComment CustomerComment)
        {
            try
            {
                var commnetList = new List<CustomerComment>();
                var result = _customerService.addCustomerComment(CustomerComment.CustomerID, CustomerComment.Notes);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(commnetList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("DeleteCustomerData")]
        public JsonResult DeleteCustomerData(int customerId)
        {
            try
            {
                if (customerId == -1)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.DeleteCustomer(customerId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertNewCustomer")]
        public JsonResult InsertNewCustomer([FromBody] CustomerModel customerObj)
        {
            try
            {
                if (customerObj == null) 
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                string errorMessage = string.Empty;
                
                //bool isValid = true;
                //GlobalErrors result = _customerService.ValidateCustomerToSave(customerObj, ref isValid, ref errorMessage);




                GlobalErrors result = _customerService.InsertNewCustomer(customerObj, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_While_Saving : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #region [Outlet Attachments]
        [HttpPost]
        [ActionName("InsertOutletAttachments")]
        public JsonResult InsertOutletAttachments(Collection<IFormFile> attachedFiles, string customerID, string outletID)
        {
            bool isAttachedExist = attachedFiles != null && attachedFiles.Count >= 0;
            try
            {
                if (isAttachedExist)
                    if (!String.IsNullOrEmpty(LocalUtilities.CustomerAttachments) && !String.IsNullOrEmpty(LocalUtilities.CustomerAttachmentsDummy))
                    {
                        if (!Directory.Exists(LocalUtilities.CustomerAttachments))
                        {
                            Directory.CreateDirectory(LocalUtilities.CustomerAttachments);
                        }
                        string dummyCustomerFolderPath = Path.Combine(LocalUtilities.CustomerAttachmentsDummy, _requestRepository.SessionId + "\\" + customerID);
                        string customerAttachmentsBasePath = dummyCustomerFolderPath.Substring(dummyCustomerFolderPath.IndexOf("wwwroot"));// + "/" + customerObj.CustomerId.ToString();
                        customerAttachmentsBasePath = customerAttachmentsBasePath.Replace('\\', '/');
                        string dummyOutletFolderPath = Path.Combine(dummyCustomerFolderPath, outletID);
                        if (!Directory.Exists(LocalUtilities.CustomerAttachments))
                            Directory.CreateDirectory(LocalUtilities.CustomerAttachments);
                        if (!Directory.Exists(LocalUtilities.CustomerAttachmentsDummy + "\\" + _requestRepository.SessionId))
                            Directory.CreateDirectory(LocalUtilities.CustomerAttachmentsDummy + "\\" + _requestRepository.SessionId);
                        var customersattachmentFolders = Directory.GetDirectories(LocalUtilities.CustomerAttachmentsDummy + "\\" + _requestRepository.SessionId).Where(x => !(x.Contains("\\" + customerID)));
                        foreach (string attachmentFolder in customersattachmentFolders)
                        {
                            Directory.Delete(attachmentFolder, true);

                        }
                        if (!Directory.Exists(dummyCustomerFolderPath))
                        {
                            Directory.CreateDirectory(dummyCustomerFolderPath);
                        }
                        if (!Directory.Exists(dummyOutletFolderPath))
                        {
                            Directory.CreateDirectory(dummyOutletFolderPath);
                        }

                        // Delete old attachment
                        if (Directory.Exists(dummyOutletFolderPath))
                        {
                            Directory.Delete(dummyOutletFolderPath, true);
                        }
                        if (isAttachedExist && !Directory.Exists(dummyOutletFolderPath))
                        {
                            Directory.CreateDirectory(dummyOutletFolderPath);
                        }
                        for (int i = 0; i < attachedFiles.Count; i++)
                        {
                            var file = attachedFiles[i];
                            if (file.Length > 0)
                            {
                                string fileName = file.FileName;
                                string filePath = Path.Combine(dummyOutletFolderPath, fileName);
                                using (var stream = new FileStream(filePath, FileMode.Create))
                                {
                                    file.CopyTo(stream);
                                }
                            }
                        }
                        return JSonResultResponse.GetSuccessrJSon(customerAttachmentsBasePath, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                    }
                return JSonResultResponse.GetSuccessrJSon("", ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("DownloadOutletAttachments")]
        public FileStream DownloadOutletAttachments(string filePath, string fileName)
        {
            try
            {
                if (string.IsNullOrEmpty(fileName))
                {
                    return null;
                }
                string folderName = LocalUtilities.CustomerAttachmentsDummy + "\\" + _requestRepository.SessionId + "\\" + filePath;
                var path = Path.Combine(folderName, fileName);

                return new FileStream(path, FileMode.Open, FileAccess.Read);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        #endregion
        [HttpPost]
        [ActionName("UpdateCustomer")]
        public JsonResult UpdateCustomer([FromBody] CustomerModel customerObj)
        {
            try
            {
                if (customerObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                string errorMessage = string.Empty;
                GlobalErrors result = _customerService.UpdateCustomer(customerObj, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_While_Saving : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        //[HttpGet]
        //[ActionName("UpdateInactiveCustomer")]
        //public JsonResult UpdateInactiveCustomer(int customerId, bool isActive)
        //{
        //    try
        //    {
        //        if (!ModelState.IsValid)
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
        //        }
        //        GlobalErrors result = _customerService.UpdateInactiveCustomer(customerId, isActive);
        //        if (result == GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        [HttpPost]
        [ActionName("InsertNewCustomerOutlet")]
        public JsonResult InsertNewCustomerOutlet([FromBody] CustomerOutletModel outletObj)
        {
            try
            {
                if (outletObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.InsertNewCustomerOutlet(outletObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        //[HttpPost]
        //[ActionName("UpdateOutletActiveStatus")]
        //public JsonResult UpdateOutletActiveStatus([FromBody] GeneralFilter filter)
        //{
      
        //        try
        //        {
        //            int outletId = -1;
        //            bool isActive = false;
        //            if (filter != null)
        //            {
        //            outletId = filter.OutletId;
        //                isActive = filter.IncludeInactive;
        //            }
        //            if (!ModelState.IsValid)
        //            {
        //                return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
        //            }
        //            GlobalErrors result = _customerService.UpdateOutletActiveStatus(outletId, isActive);
        //            if (result == GlobalErrors.Success)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //            }
        //        }
        //        catch (Exception ex)
        //        {
        //            ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //            return JSonResultResponse.GetErrorJSon();
        //        }
            
        //}

        [HttpPost]
        [ActionName("UpdateCustomerOutlet")]
        public JsonResult UpdateCustomerOutlet([FromBody] CustomerOutletModel outletObj)
        {
            try
            {
                if (outletObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.UpdateCustomerOutlet(outletObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertNewCustomerGroup")]
        public JsonResult InsertNewCustomerGroup([FromBody] CustomerGroupModel groupObj)
        {
            try
            {
                if (groupObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.InsertNewCustomerGroup(ref groupObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(groupObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateCustomerGroup")]
        public JsonResult UpdateCustomerGroup([FromBody] CustomerGroupModel groupObj)
        {
            try
            {
                if (groupObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.UpdateCustomerGroup(groupObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertNewPaymentTerm")]
        public JsonResult InsertNewPaymentTerm([FromBody] PaymentTermModel paymentTermObj)
        {
            try
            {
                if (paymentTermObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.InsertNewPaymentTerm(paymentTermObj);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(paymentTermObj, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertNewCustomerClass")]
        public JsonResult InsertNewCustomerClass([FromBody] CustomerClassModel customerClass)
        {
            try
            {
                if (customerClass == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.InsertNewCustomerClass(customerClass);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerClass, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertNewCustomerSubClass")]
        public JsonResult InsertNewCustomerSubClass([FromBody] CustomerSubClassModel customerSubClass)
        {
            try
            {
                if (customerSubClass == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                GlobalErrors result = _customerService.InsertNewCustomerSubClass(customerSubClass);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customerSubClass, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetDivisionsCredits")]
        public JsonResult GetDivisionsCredits([FromBody] GeneralFilter filter)
        {
            try
            {
                List<DivisionsCredits> divisionsCredits = new List<DivisionsCredits>();
                int customerTemplateId = -1;
                if(filter != null)
                {
                    customerTemplateId = filter.TemplateId;
                }
                GlobalErrors result = _customerService.GetDivisionsCredits(ref divisionsCredits, customerTemplateId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(divisionsCredits, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        
        [HttpPost]
        [ActionName("InitializeCustomerTemplateForm")]
        public JsonResult InitializeCustomerTemplateForm([FromBody] FieldsObject fields)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<CustomerTypeModel> customerTypes = new List<CustomerTypeModel>();
                result = _customerService.GetCustomerTypes(ref customerTypes, fields.languageId);

                List<ComboBoxModel> priceList = new List<ComboBoxModel>();
                result = _customerService.GetNewTemplatePriceList(ref priceList);

                List<ComboBoxModel> promotionGroup = new List<ComboBoxModel>();
                result = _customerService.GetNewTemplatePromotionGroups(ref promotionGroup);

                List<ComboBoxModel> discountGroup = new List<ComboBoxModel>();
                result = _customerService.GetNewTemplateDiscountGroups(ref discountGroup);

                List<ComboBoxModel> generalFields = new List<ComboBoxModel>();
                result = _customerService.getGeneralFields(ref generalFields);

                List<ComboBoxModel> detailsFields = new List<ComboBoxModel>();
                result = _customerService.getDetailsFields(ref detailsFields, fields);

                List<ComboBoxModel> addressFields = new List<ComboBoxModel>();
                result = _customerService.getAddressFields(ref addressFields, fields);

                List<ComboBoxModel> imagesFields = new List<ComboBoxModel>();
                result = _customerService.getImagesFields(ref imagesFields, fields);

                List<ComboBoxModel> defaultGroup = new List<ComboBoxModel>();
                result = _customerService.GetGroups(ref defaultGroup);

                List<ComboBoxModel> addedDisplayFields = new List<ComboBoxModel>();
                result = _customerService.getAddedDisplayFields(ref addedDisplayFields, fields);

                List<PaymentTermModel> paymentTerms = new List<PaymentTermModel>();
                result = _customerService.GetPaymentTerms(ref paymentTerms);

                List<ComboBoxModel> generalAddedFields = new List<ComboBoxModel>();
                result = _customerService.getAddedFields(ref generalAddedFields, 1, fields);

                List<ComboBoxModel> detailsAddedFields = new List<ComboBoxModel>();
                result = _customerService.getAddedFields(ref detailsAddedFields, 2, fields);

                List<ComboBoxModel> addressAddedFields = new List<ComboBoxModel>();
                result = _customerService.getAddedFields(ref addressAddedFields, 3, fields);

                List<ComboBoxModel> displayAddedFields = new List<ComboBoxModel>();
                result = _customerService.getAddedFields(ref displayAddedFields, 4, fields);

                List<ComboBoxModel> imagesAddedFields = new List<ComboBoxModel>();
                result = _customerService.getAddedFields(ref imagesAddedFields, 5, fields);

                List<ComboBoxModel> generalAddedMandatoryFields = new List<ComboBoxModel>();
                result = _customerService.getAddedMandatoryFields(ref generalAddedMandatoryFields, 1, fields);

                List<ComboBoxModel> detailsAddedMandatoryFields = new List<ComboBoxModel>();
                result = _customerService.getAddedMandatoryFields(ref detailsAddedMandatoryFields, 2, fields);

                List<ComboBoxModel> addressAddedMandatoryFields = new List<ComboBoxModel>();
                result = _customerService.getAddedMandatoryFields(ref addressAddedMandatoryFields, 3, fields);

                List<ComboBoxModel> displayAddedMandatoryFields = new List<ComboBoxModel>();
                result = _customerService.getAddedMandatoryFields(ref displayAddedMandatoryFields, 4, fields);

                List<ComboBoxModel> imagesAddedMandatoryFields = new List<ComboBoxModel>();
                result = _customerService.getAddedMandatoryFields(ref imagesAddedMandatoryFields, 5, fields);

                var data = new
                {
                    customerTypes = customerTypes,
                    defaultGroup = defaultGroup,
                    priceList = priceList,
                    promotionGroup = promotionGroup,
                    discountGroup = discountGroup,
                    generalFields = generalFields,
                    detailsFields = detailsFields,
                    addressFields = addressFields,
                    imagesFields = imagesFields,
                    addedDisplayFields = addedDisplayFields,
                    generalAddedFields = generalAddedFields,
                    addressAddedFields = addressAddedFields,
                    displayAddedFields = displayAddedFields,
                    detailsAddedFields = detailsAddedFields,
                    imagesAddedFields = imagesAddedFields,
                    generalAddedMandatoryFields = generalAddedMandatoryFields,
                    addressAddedMandatoryFields = addressAddedMandatoryFields,
                    displayAddedMandatoryFields = displayAddedMandatoryFields,
                    detailsAddedMandatoryFields = detailsAddedMandatoryFields,
                    imagesAddedMandatoryFields = imagesAddedMandatoryFields,
                    paymentTerms = paymentTerms,
                };

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
       
        [HttpPost]
        [ActionName("ValidateOutletToSave")]
        public JsonResult ValidateOutletToSave([FromBody] CustomerOutletModel outlet)
        {
            try
            {
                bool uniqueCode = true;
                bool uniqueBarcode = true;
                bool uniqueMachineSerialNumber = true;
                bool uniqueMachineIMEINumber = true;
                bool isValid = true;
                bool uniquePhone = true;
                bool uniqueEmail = true;
                bool uniqueOperatorName = true;
                GlobalErrors result = _customerService.ValidateOutletToSave(outlet, ref uniqueCode, ref uniqueBarcode, ref uniqueMachineSerialNumber, ref uniqueMachineIMEINumber, ref uniquePhone, ref uniqueEmail, ref uniqueOperatorName, ref isValid);

                var data = new
                {
                    uniqueCode = uniqueCode,
                    uniqueBarcode = uniqueBarcode,
                    uniqueMachineSerialNumber = uniqueMachineSerialNumber,
                    uniqueMachineIMEINumber = uniqueMachineIMEINumber,
                    uniquePhone = uniquePhone,
                    uniqueEmail = uniqueEmail,
                    uniqueOperatorName = uniqueOperatorName,
                    isValid = isValid
                };
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        //[HttpPost]
        //[ActionName("UpdateCustomerCreditRelease")]
        //public JsonResult UpdateCustomerCreditRelease([FromBody] CreditReleaseModel creditReleaseObj)
        //{
        //    try
        //    {
        //        GlobalErrors result = _customerService.UpdateCustomerCreditRelease(creditReleaseObj);
        //        if (result == GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        [HttpPost]
        [ActionName("GetCustomerGroupsListSharedTable")]
        public JsonResult GetCustomerGroupsListSharedTable([FromBody] GeneralFilter customerGroupsFilter)
        {
            try
            {
                SharedTableResult<CustomerGroupModel> customerGroupsList = new SharedTableResult<CustomerGroupModel>();
                GlobalErrors result = _customerService.GetCustomerGroupsListSharedTable(customerGroupsFilter, ref customerGroupsList);
                if (result == GlobalErrors.Success)
                {
                    if (customerGroupsList != null && customerGroupsList.Data != null && customerGroupsList.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(customerGroupsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(customerGroupsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetOutletsListSharedTable")]
        public JsonResult GetOutletsListSharedTable([FromBody] CustomerOutletFilter customerOutletFilter)
        {
            try
            {
                SharedTableResult<SimpleCustomerModel> outletsList = new SharedTableResult<SimpleCustomerModel>();
                GlobalErrors result = _customerService.GetOutletsListSharedTable(customerOutletFilter, ref outletsList);
                if (result == GlobalErrors.Success)
                {
                    if (outletsList != null && outletsList.Data != null && outletsList.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(outletsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(outletsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomersListSharedTable")]
        public JsonResult GetCustomersListSharedTable([FromBody] CustomerOutletFilter customerOutletFilter)
        {
            try
            {
                SharedTableResult<SimpleCustomerModel> customersList = new SharedTableResult<SimpleCustomerModel>();
                GlobalErrors result = _customerService.GetCustomersListSharedTable(customerOutletFilter, ref customersList);
                if (result == GlobalErrors.Success)
                {
                    if (customersList != null && customersList.Data != null && customersList.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        //[HttpGet]
        //[ActionName("updateCustomerOrg")]
        //public JsonResult updateCustomerOrg(int customerId, int outletId, int organizationId)
        //{
        //    try
        //    {
        //        GlobalErrors result = _customerService.updateCustomerOrg(customerId, outletId, organizationId);
        //        if (result == GlobalErrors.Success || result == GlobalErrors.SuccessWithZeroRowAffected)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //[HttpPost]
        //[ActionName("MatchNewCustomer")]
        //public JsonResult MatchNewCustomer([FromBody] CustomerOutletModel outletObj)
        //{
        //    try
        //    {
        //        if (outletObj == null) return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        GlobalErrors result = _customerService.MatchNewCustomer(outletObj);
        //        if (result == GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(outletObj.CustomerId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}

        [HttpPost]
        [ActionName("CheckCustomerGroupAchievementPromotions")]
        public JsonResult CheckCustomerGroupAchievementPromotions([FromBody] List<int> groupIdsList)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
                List<Promotion> groupPromotions = new List<Promotion>();
                result = _customerService.CheckCustomerGroupAchievementPromotions(groupIdsList, ref groupPromotions);
                
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(groupPromotions, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        #region [CustomerMapping]
        [HttpGet]
        [ActionName("GetCustomerMappingList2")]
        public JsonResult GetCustomerMappingList2(CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid || filter == null)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetCustomerMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerMappingList")]
        public JsonResult GetCustomerMappingList([FromBody] CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid || filter == null)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetCustomerMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetRouteCustomerMappingList")]
        public JsonResult GetRouteCustomerMappingList([FromBody] CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetRouteCustomerMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetRouteHistoryMappingList")]
        public JsonResult GetRouteHistoryMappingList([FromBody] CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetRouteHistoryMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetTerritoryCustomersMappingList")]
        public JsonResult GetTerritoryCustomersMappingList([FromBody] CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetTerritoryCustomersMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetEmployeeCustomersMappingList")]
        public JsonResult GetEmployeeCustomersMappingList([FromBody] CustomerMappingFilter filter)
        {
            try
            {
                if (!ModelState.IsValid || filter == null)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                var customersList = new List<CustomerOutletModel>();
                var result = _customerService.GetEmployeeCustomersMappingList(filter, ref customersList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomerOutletMappingDetails")]
        public JsonResult GetCustomerOutletMappingDetails([FromBody] CustomerMappingFilter filter)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!ModelState.IsValid || filter == null)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                CustomerOutletModel customersOutletDetailsObject = new CustomerOutletModel();
                result = _customerService.GetCustomerOutletMappingDetails(filter, ref customersOutletDetailsObject);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(customersOutletDetailsObject, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CheckMapsAvailability")]
        public JsonResult CheckMapsAvailability()
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                bool isConnected = _customerService.PingHost(false);
                return JSonResultResponse.GetSuccessrJSon(isConnected, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, 1));

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        #endregion

        [HttpPost]
        [ActionName("GetPayerAccountList")]
        public JsonResult GetPayerAccountList([FromBody]int currencyId)
        {
            try
            {
                List<PayerAccountModel> payerAccounts = new List<PayerAccountModel>();

                GlobalErrors result = _customerService.GetPayerAccount(currencyId,ref payerAccounts);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(payerAccounts, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetCustomersPayerOutlets")]
        public JsonResult GetCustomersPayerOutlets([FromBody] GeneralFilter filter)
        {
            try
            {
                List<PayerAccountModel> custOutlets = new List<PayerAccountModel>();

                GlobalErrors result = _customerService.GetCustomersPayerOutlets(filter, ref custOutlets);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(custOutlets, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetAllSalesDistributionChannels")]
        public JsonResult GetAllSalesDistributionChannels([FromBody] GeneralFilter filter)
        {
            try
            {
                List<SalesDistributionChannelModel> channels = new List<SalesDistributionChannelModel>();
                GlobalErrors result = _customerService.GetSalesDistributionChannels(ref channels,filter);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(channels, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CheckCustomerOrganizationIsExist")]
        public JsonResult CheckCustomerOrganizationIsExist([FromBody] GeneralFilter filter)
        {
            try
            {
                bool isCustomerLinktoOrganization = false;
                GlobalErrors result = _customerService.CheckCustomerOrganizationIsExist(ref isCustomerLinktoOrganization,filter);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(isCustomerLinktoOrganization, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("CheckCustomerOutletDivisionIsExist")]
        public JsonResult CheckCustomerOutletDivisionIsExist()
        {
            try
            {
                bool isTableHasRecords = false;
                GlobalErrors result = _customerService.CheckCustomerOutletDivisionIsExist(ref isTableHasRecords);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(isTableHasRecords, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetMaxNewCustomerCodeSequence")]
        public JsonResult GetMaxNewCustomerCodeSequence()
        {
            try
            {
                string documentId = string.Empty;
                string responseMessage = MessagesConstants.Desc_No_Document_Sequence;
                documentId = _customerService.GetMaxCustomerCodeSequence();

                if (!string.IsNullOrEmpty(documentId))
                {
                    return JSonResultResponse.GetSuccessrJSon(documentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(responseMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetMaxNewOutletCodeSequence")]
        public JsonResult GetMaxNewOutletCodeSequence()
        {
            try
            {
                string documentId = string.Empty;
                string responseMessage = MessagesConstants.Desc_No_Document_Sequence;
                documentId = _customerService.GetMaxNewOutletCodeSequence();

                if (!string.IsNullOrEmpty(documentId))
                {
                    return JSonResultResponse.GetSuccessrJSon(documentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(responseMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        //#region [Price Protection]
        //[HttpPost]
        //[ActionName("UpdatePriceProtection")]
        //public JsonResult UpdatePriceProtection([FromBody] List<PriceProtectionDetailModel> PriceProtectionList)
        //{
        //    try
        //    {
        //        if (PriceProtectionList != null && PriceProtectionList.Count > 0)
        //        {
        //            var result = _customerService.UpdatePriceProtection(PriceProtectionList);

        //            if (result == GlobalErrors.Success)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //            }
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //[HttpPost]
        //[ActionName("UpdateStatusForPriceProtection")]
        //public JsonResult UpdateStatusForPriceProtection([FromBody] PriceProtectionModel request, int statusId)
        //{
        //    try
        //    {
        //        if (request != null && statusId > 0)
        //        {
        //            string message = string.Empty;
        //            var result = _customerService.UpdateStatusForPriceProtection(request, statusId, ref message);

        //            if (result == GlobalErrors.Success)
        //            {
        //                return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(message, _requestRepository.LanguageId));
        //            }
        //            else
        //            {
        //                return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(message) ? MessagesConstants.Desc_Error_While_Saving : message, _requestRepository.LanguageId));
        //            }
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //#endregion
        #endregion
        [HttpPost]
        [ActionName("GetCustomeLastTrans")]
        public JsonResult GetCustomeLastTrans([FromBody] GeneralFilter filter)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
                TransactionModel lastInvoice = new TransactionModel();
                result = _customerService.GetCustomerLastInvoice(filter.CustomerId, filter.OutletId, ref lastInvoice);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(lastInvoice, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}