﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.SignalR;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.Model.Customer;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class DiscountController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly CustomerService _customerService;
        private readonly MasterDataService _masterDataService;
        private readonly PricesAndOffersService _pricesAndOffersService;
        private readonly IHostingEnvironment _hostingEnvironment;
        private readonly IWebHostEnvironment _webHostEnvironment;

        public DiscountController(IHubContext<NotifyHub, ITypedHubClient> hubContext, IRequestRepository requestRepository, IWebHostEnvironment webHostEnvironment, ILicensing licensing, IHostingEnvironment hostingEnvironment, IActiveTokens activeTokens) : base(requestRepository, licensing, activeTokens)
        {
            _requestRepository = requestRepository;
            _customerService = new CustomerService(_requestRepository);
            _masterDataService = new MasterDataService(requestRepository);
            _pricesAndOffersService = new PricesAndOffersService(hubContext, requestRepository,_hostingEnvironment);
            _hostingEnvironment = hostingEnvironment;
            _webHostEnvironment = webHostEnvironment;
        }

        #region GET
        [HttpPost]
        [ActionName("GetDiscounts")]
        public JsonResult GetDiscounts([FromBody] GeneralFilter filter)
        {
            try
            {
                SharedTableResult<DiscountModel> discounts = new SharedTableResult<DiscountModel>();
                GlobalErrors result = _pricesAndOffersService.GetDiscounts(filter, ref discounts);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(discounts, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetDiscountData")]
        public JsonResult GetDiscountData([FromBody] DiscountFilter filter)
        {
            try
            {
                int discountId = -1;
                bool isClone = false;
                if(filter != null)
                {
                    discountId = filter.DiscountId;
                    isClone = filter.IsClone;
                }
                List<CustomerChannelModel> appliedChannels = new List<CustomerChannelModel>();
                List<CustomerGroupModel> appliedGroups = new List<CustomerGroupModel>();
                List<SimpleCustomerModel> appliedCustomers = new List<SimpleCustomerModel>();
                List<SalesDistributionChannelModel> appliedSDCList = new List<SalesDistributionChannelModel>();
                List<TextLanguage> descriptions = new List<TextLanguage>();
                List<string> attachedFiles = new List<string>();
                GlobalErrors result = _pricesAndOffersService.GetDiscountDescriptions(discountId, isClone, ref descriptions);
                if (result == GlobalErrors.Success)
                    result = _pricesAndOffersService.GetDiscountAppliedChannels(discountId, ref appliedChannels);
                if (result == GlobalErrors.Success)
                    result = _pricesAndOffersService.GetDiscountAppliedGroups(discountId, ref appliedGroups);
                if (result == GlobalErrors.Success)
                    result = _pricesAndOffersService.GetDiscountAppliedCustomers(discountId, ref appliedCustomers);
                if (result == GlobalErrors.Success)
                    result = _pricesAndOffersService.GetDiscountAppliedSDCs(discountId, ref appliedSDCList);
                if (result == GlobalErrors.Success)
                {
                    List<string> attachementNames = new List<string>();
                    FileManagerModel model = new FileManagerModel();
                    // For Attachements :
                    string folderName = Path.Combine("Resources\\attachements\\discounts", discountId + "");
                    string rootPath = _hostingEnvironment.ContentRootPath;
                    var userAttachementPath = Path.Combine(rootPath, folderName);
                    DirectoryInfo Attachementdir = new DirectoryInfo(userAttachementPath);
                    if (Attachementdir.Exists)
                    {
                        FileInfo[] files = Attachementdir.GetFiles();
                        model.Files = files;
                        if (files != null && files.Length > 0)
                        {
                            foreach (FileInfo file in files)
                            {
                                attachementNames.Add(file.Name);
                            }
                        }
                    }
                    if (attachementNames != null && attachementNames.Count > 0)
                    {
                        attachedFiles = attachementNames;
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        languageDescriptions = descriptions,
                        appliedChannels = appliedChannels,
                        appliedGroups = appliedGroups,
                        appliedCustomers = appliedCustomers,
                        sdcsList = appliedSDCList,
                        attachedFiles = attachedFiles
                    };
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetDiscountTypes")]
        public JsonResult GetDiscountTypes()
        {
            try
            {
                List<DiscountTypesModel> discountTypes = new List<DiscountTypesModel>();

                GlobalErrors result = _pricesAndOffersService.GetDiscountTypes(ref discountTypes);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(discountTypes, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetSalesOrderTypes")]
        public JsonResult GetSalesOrderTypes()
        {
            try
            {
                List<DiscountModel> salesOrderTypes = new List<DiscountModel>();

                GlobalErrors result = _pricesAndOffersService.GetSalesOrderTypes(ref salesOrderTypes);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(salesOrderTypes, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        #endregion

        #region INSERT
        [HttpPost]
        [ActionName("InsertDiscount")]
        public JsonResult InsertDiscount(Collection<IFormFile> attachedFiles, [FromForm] string discountString)
        {
            try
            {
                string excludedString = string.Empty;
                bool uniqueCode = true;
                DiscountModel discountObj = JsonConvert.DeserializeObject<DiscountModel>(discountString);
                if (discountObj.DiscountId != -1)
                {
                    excludedString = string.Format(" And( (DiscountID != {0}) )", discountObj.DiscountId);
                }
                GlobalErrors result = _masterDataService.CheckIfUniqueValue(CoreDataBaseConstants.TableNames.Discount, CoreDataBaseConstants.QueryColumnsNames.DiscountCode, discountObj.DiscountCode, excludedString, ref uniqueCode);
                if (!uniqueCode)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Discount_Code, _requestRepository.LanguageId) + " " + ResourcesManager.TranslateKey(MessagesConstants.Desc_Msg_Must_Be_Unique, _requestRepository.LanguageId));
                }

                discountObj.StartDate = new DateTime(discountObj.StartDateModel.Year, discountObj.StartDateModel.Month, discountObj.StartDateModel.Day);
                discountObj.EndDate = new DateTime(discountObj.EndDateModel.Year, discountObj.EndDateModel.Month, discountObj.EndDateModel.Day);

                result = _pricesAndOffersService.SaveDiscount(discountObj);

                #region[save Attachements]
                bool createNewAttachementDirectory = attachedFiles != null && attachedFiles.Count > 0;
                // Create Directory in Sonic 
                string rootPath = _hostingEnvironment.ContentRootPath;
                string folderName = "Resources\\attachements\\discounts\\";
                string thisDiscountName = Path.Combine(folderName, discountObj.DiscountId.ToString());
                string attachfullPathSonic = Path.Combine(rootPath, thisDiscountName);


                //// Delete old save from sonic
                //if (Directory.Exists(attachfullPathSonic))
                //{
                //    Directory.Delete(attachfullPathSonic, true);
                //}
                if (createNewAttachementDirectory && !Directory.Exists(attachfullPathSonic))
                {
                    Directory.CreateDirectory(attachfullPathSonic);
                }
                for (int i = 0; i < discountObj.DeletedFiles.Count; i++)
                { // delete only deleted files , and another file existed .
                    string fileName = discountObj.DeletedFiles[i];
                    string deletepath = Path.Combine(attachfullPathSonic, fileName);
                    string[] fileList = System.IO.Directory.GetFiles(attachfullPathSonic, fileName);
                    foreach (string file in fileList)
                    {
                        //System.Diagnostics.Debug.WriteLine(file + "will be deleted");
                        System.IO.File.Delete(file);
                    }
                }
                for (int i = 0; i < attachedFiles.Count; i++)
                {
                    var file = attachedFiles[i];
                    if (file.Length > 0)
                    {
                        //if (file.Name.ToString().ToLower() == "attachedfiles")
                        {
                            string fileName = file.FileName;
                            string filePath = Path.Combine(attachfullPathSonic, fileName);
                            using (var stream = new FileStream(filePath, FileMode.Create))
                            {
                                file.CopyTo(stream);
                            }
                        }
                    }
                }
                #endregion
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        DiscountId = discountObj.DiscountId,
                        Status = discountObj.Status
                    };
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region UPDATE
        [HttpPost]
        [ActionName("UpdateDiscount")]
        public JsonResult UpdateDiscount(Collection<IFormFile> attachedFiles, [FromForm] string discountString)
        {
            try
            {
                string excludedString = string.Empty;
                bool uniqueCode = true;
                DiscountModel discountObj = JsonConvert.DeserializeObject<DiscountModel>(discountString);
                if (discountObj.DiscountId != -1)
                {
                    excludedString = string.Format(" And( (DiscountID != {0}) )", discountObj.DiscountId);
                }
                GlobalErrors result = _masterDataService.CheckIfUniqueValue(CoreDataBaseConstants.TableNames.Discount, CoreDataBaseConstants.QueryColumnsNames.DiscountCode, discountObj.DiscountCode, excludedString, ref uniqueCode);
                if (!uniqueCode)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Discount_Code, _requestRepository.LanguageId) + " " + ResourcesManager.TranslateKey(MessagesConstants.Desc_Msg_Must_Be_Unique, _requestRepository.LanguageId));
                }

                discountObj.StartDate = new DateTime(discountObj.StartDateModel.Year, discountObj.StartDateModel.Month, discountObj.StartDateModel.Day);
                discountObj.EndDate = new DateTime(discountObj.EndDateModel.Year, discountObj.EndDateModel.Month, discountObj.EndDateModel.Day);

                result = _pricesAndOffersService.SaveDiscount(discountObj);

                #region[save Attachements]
                bool createNewAttachementDirectory = attachedFiles != null && attachedFiles.Count > 0;
                // Create Directory in Sonic 
                string rootPath = _hostingEnvironment.ContentRootPath;
                string folderName = "Resources\\attachements\\discounts\\";
                string thisDiscountName = Path.Combine(folderName, discountObj.DiscountId.ToString());
                string attachfullPathSonic = Path.Combine(rootPath, thisDiscountName);


                //// Delete old save from sonic
                //if (Directory.Exists(attachfullPathSonic))
                //{
                //    Directory.Delete(attachfullPathSonic, true);
                //}
                if (createNewAttachementDirectory && !Directory.Exists(attachfullPathSonic))
                {
                    Directory.CreateDirectory(attachfullPathSonic);
                }
                for (int i = 0; i < discountObj.DeletedFiles.Count; i++)
                { // delete only deleted files , and another file existed .
                    string fileName = discountObj.DeletedFiles[i];
                    string deletepath = Path.Combine(attachfullPathSonic, fileName);
                    string[] fileList = System.IO.Directory.GetFiles(attachfullPathSonic, fileName);
                    foreach (string file in fileList)
                    {
                        //System.Diagnostics.Debug.WriteLine(file + "will be deleted");
                        System.IO.File.Delete(file);
                    }
                }
                for (int i = 0; i < attachedFiles.Count; i++)
                {
                    var file = attachedFiles[i];
                    if (file.Length > 0)
                    {
                        //if (file.Name.ToString().ToLower() == "attachedfiles")
                        {
                            string fileName = file.FileName;
                            string filePath = Path.Combine(attachfullPathSonic, fileName);
                            using (var stream = new FileStream(filePath, FileMode.Create))
                            {
                                file.CopyTo(stream);
                            }
                        }
                    }
                }
                #endregion
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        DiscountId = discountObj.DiscountId,
                        Status = discountObj.Status
                    };
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("UpdateDiscountActiveStatus")]
        public JsonResult UpdateDiscountActiveStatus([FromBody] DiscountFilter filter)
        {
            try
            {
                int discountId = -1;
                bool isActive = false;
                if(filter != null)
                {
                    discountId = filter.DiscountId;
                    isActive = filter.IsActive;
                }
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                GlobalErrors result = _pricesAndOffersService.UpdateDiscountActiveStatus(discountId, isActive);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region DELETE
        [HttpPost]
        [ActionName("DeleteDiscount")]
        public JsonResult DeleteDiscount([FromBody] int discountId)
        {
            try
            {
                GlobalErrors result = _pricesAndOffersService.DeleteDiscount(discountId);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion
    }
}