﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.Model.Customer;
using Sonic.Web.PL.Helper;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class DivisionController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly CustomerService _customerService;
        private readonly MasterDataService _masterDataService;
        private readonly IHostingEnvironment _hostingEnvironment;
        private readonly IWebHostEnvironment _webHostEnvironment;
        private readonly ItemService _itemService;
        private readonly EmployeeService _employeeService;
        private readonly LoginService _loginService;
        private readonly IActiveTokens _IactiveToken;


        public DivisionController(IRequestRepository requestRepository, IWebHostEnvironment webHostEnvironment, ILicensing licensing, IHostingEnvironment hostingEnvironment, IActiveTokens activeTokens) : base(requestRepository, licensing, activeTokens)
        {
            _requestRepository = requestRepository;
            _IactiveToken = activeTokens;
            _customerService = new CustomerService(_requestRepository);
            _masterDataService = new MasterDataService(requestRepository);
            _hostingEnvironment = hostingEnvironment;
            _webHostEnvironment = webHostEnvironment;
            _itemService = new ItemService(_requestRepository);
            _employeeService = new EmployeeService(_requestRepository);
            _loginService = new LoginService(_requestRepository, activeTokens);
        }

        #region GET
        [HttpPost]
        [ActionName("AllowToDeleteDivision")]
        public JsonResult AllowToDeleteDivision([FromBody] int divisionId)
        {
            try
            {
                string response = string.Empty;
                var result = _itemService.AllowToDeleteDivision(divisionId, ref response);
                if (result != GlobalErrors.Success)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(response, _requestRepository.LanguageId));
                }
                else
                    return JSonResultResponse.GetSuccessrJSon(response, response);
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in Getting Division");
            }
        }

        [HttpPost]
        [ActionName("GetDivisionData")]
        public JsonResult GetDivisionData([FromBody] int divisionId)
        {
            try
            {
                var data = new DivisionModel();
                string message = "Success Getting Division";
                var result = _itemService.GetDivisionData(divisionId, ref data);
                if (result != GlobalErrors.Success)
                {
                    message = "Error in Getting Division";
                    return JSonResultResponse.GetErrorJSon(message);
                }
                else
                    return JSonResultResponse.GetSuccessrJSon(data, message);
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in Getting Division");
            }
        }

        [HttpPost]
        [ActionName("GetAllDivisions")]
        public JsonResult GetAllDivisions([FromBody] DivisionFilter filter)
        {
            try
            {
                var data = new List<DivisionModel>();
                string message = "Success Getting Divisions";
                GlobalErrors result = _itemService.GetAllDivisions(filter, ref data);
                if (result != GlobalErrors.Success)
                {
                    message = "Error in Getting Divisions";
                    return JSonResultResponse.GetErrorJSon(message);
                }
                else
                    return JSonResultResponse.GetSuccessrJSon(data, message);
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in Getting Divisions");
            }
        }

        [HttpPost]
        [ActionName("CheckDivisionCodeExist")]
        public JsonResult CheckDivisionCodeExist([FromBody] DivisionModel divisionCode)
        {
            try
            {
                bool exist = false;
                var result = _itemService.CheckCodeIsExist("Division", "DivisionCode", divisionCode.DivisionCode, "DivisionId", divisionCode.DivisionId, ref exist);
                var data = new
                {
                    isExist = exist,
                };

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, "");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon();
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetDivisions")]
        public JsonResult GetDivisions([FromBody] GeneralFilter filter)
        {
            try
            {
                List<DivisionModel> divisionList = new List<DivisionModel>();

                var result = _itemService.GetDivisions(filter, ref divisionList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(divisionList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetDivisionsUnderOrgAccess")]
        public JsonResult GetDivisionsUnderOrgAccess([FromBody] DivisionModel employeeDivisons)
        {
            try
            {
                List<DivisionModel> divisionsList = new List<DivisionModel>();
                GlobalErrors result = _itemService.GetDivisionsUnderOrgAccess(ref divisionsList, employeeDivisons.EmployeeAccess , employeeDivisons.IsFromStaffEmployeeOrgAccess);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(divisionsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("CheckIfDivisionHasImage")]

        public JsonResult CheckIfDivisionHasImage(int divisionId)
        {
            try
            {
                string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                string resourcesFullPath = Path.Combine(_hostingEnvironment.ContentRootPath, LocalUtilities.DivisionImageFolderPath, divisionId + ".png");

                if (System.IO.File.Exists(resourcesFullPath))
                {
                    return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));

                }
                else
                {
                    return JSonResultResponse.GetSuccessrJSon(false, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }


        #endregion

        #region UPDATE
        [HttpPost]
        [ActionName("UpdateDivision")]
        public JsonResult UpdateDivision([FromForm] string division, [FromForm] Collection<IFormFile> image)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");

                DivisionModel divisionModel = JsonConvert.DeserializeObject<DivisionModel>(division);

                string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                var result = _itemService.EditDivision(divisionModel, image, hostingEnvironment);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(division, "transaction Saved Successfuly");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in saving division");
            }
        }

        #endregion

        #region INSERT
        [HttpPost]
        [ActionName("InsertDivision")]

        public JsonResult InsertDivision([FromForm] string division, [FromForm] Collection<IFormFile> image)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                }

                DivisionModel divisionModel = JsonConvert.DeserializeObject<DivisionModel>(division);

                if (divisionModel != null)
                {
                    string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                    var result = _itemService.SaveDivision(divisionModel, image, hostingEnvironment);

                    if (result == GlobalErrors.Success)
                    {
                        result = _employeeService.FillOperatorRelatedData(false, true, false);
                        if (result != GlobalErrors.Success) return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                        //string httpToken = this.Request.Headers["Authorization"].ToString();
                        //string tokenString = WorkExtension.GenerateToken(_requestRepository.CurrentOperator, HttpContext.Session.Id, divisionModel.ExpirationDate);
                        //if (!string.IsNullOrEmpty(httpToken) && !string.IsNullOrEmpty(tokenString))
                        //{
                        //    _IactiveToken.DeleteActiveTokens(httpToken);
                        //    _IactiveToken.AddActiveTokens(tokenString);
                        //}
                        if (!string.IsNullOrEmpty(division))
                        {
                            var data = new
                            {
                                division = division,
                                currentOperator = _requestRepository.CurrentOperator
                            };
                            return JSonResultResponse.GetSuccessrJSon(data, "transaction Saved Successfuly");
                        }
                        else
                        {
                            return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                        }
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                    }
                }

            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in saving division");
            }
            return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
        }
        #endregion

        #region DELETE
        [HttpPost]
        [ActionName("DeleteDivision")]
        public JsonResult DeleteDivision([FromBody] DivisionModel division)
        {
            try
            {
                string message = "Success Getting Division";
                var result = _itemService.DeleteDivision(division.DivisionId);
                if (result == GlobalErrors.Success)
                {
                    DateTime expirationDate = Convert.ToDateTime(division.ExpirationDate);
                    result = _employeeService.FillOperatorRelatedData(false, true, false);
                    if (result != GlobalErrors.Success) return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
                    string httpToken = this.Request.Headers["Authorization"].ToString();
                    string tokenString = WorkExtension.GenerateToken(_requestRepository.CurrentOperator, HttpContext.Session.Id, expirationDate);
                    if (!string.IsNullOrEmpty(httpToken) && !string.IsNullOrEmpty(tokenString))
                    {
                        _IactiveToken.DeleteActiveTokens(httpToken);
                        _IactiveToken.AddActiveTokens(tokenString);
                    }
                    if (!string.IsNullOrEmpty(tokenString))
                    {
                        var data = new
                        {
                            tokenString = tokenString,
                            currentOperator = _requestRepository.CurrentOperator
                        };
                        return JSonResultResponse.GetSuccessrJSon(data, message);
                    }
                    else
                    {
                        message = "Error in Getting Division";
                        return JSonResultResponse.GetErrorJSon(message);
                    }
                }
                else
                {
                    message = "Error in Getting Division";
                    return JSonResultResponse.GetErrorJSon(message);
                }


            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon("Error in Getting Division");
            }
        }
        #endregion
    }
}