﻿using DinkToPdf.Contracts;
using DinkToPdf;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Razor.Templating.Core;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Data;
using System.Linq;
using ClosedXML.Excel;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class DynamicReportController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly DynamicReportService _dynamicReportService;
        private IConverter _converter;
        private readonly SharedMethod _sharedMethod;
        public DynamicReportController(IConverter converter,IRequestRepository requestRepository, ILicensing licensing, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _requestRepository = requestRepository;
            _converter = converter;
            _dynamicReportService = new DynamicReportService(_requestRepository , _converter);
            _sharedMethod = new SharedMethod(_requestRepository);
        }

        [HttpPost]
        [ActionName("GetReportFilters")]
        public JsonResult GetReportFilters([FromBody] int reportId)
        {
            try
            {
                List <ReportFilterModel> data = new List<ReportFilterModel>();
                GlobalErrors result =_dynamicReportService.GetReportFilters(reportId, ref data);
                if (result != GlobalErrors.Success)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                else
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetChildFilterData")]
        public JsonResult GetChildFilterData([FromBody] ReportFilterModel chidlControl)
        {
            try
            {
                List<ReportFilterDataSourceModel> data = new List<ReportFilterDataSourceModel>();
                GlobalErrors result = _dynamicReportService.GetChildFilterData(chidlControl, ref data);
                if (result != GlobalErrors.Success)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                else
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("ShowReportData")]
        public JsonResult ShowReportData([FromBody] DynamicReportModel reportt)
        {
            try
            {
                string uniqueIdentifier = string.Empty;
                GlobalErrors result = _dynamicReportService.ShowReportData(ref uniqueIdentifier, reportt);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(uniqueIdentifier, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("MarkReportAsFavorite")]
        public JsonResult AddReportToFavorite([FromBody] int reportId)
        {
            try
            {
                GlobalErrors result = _dynamicReportService.MarkReportAsFavorite(reportId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("UnMarkReportAsFavorite")]
        public JsonResult UnMarkReportAsFavorite([FromBody] int reportId)
        {
            try
            {
                GlobalErrors result = _dynamicReportService.UnMarkReportAsFavorite(reportId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("ExportReportFromGrid")]
        public IActionResult ExportReportFromGrid([FromBody] DynamicReportModel reportModel)
        {
            var result = GlobalErrors.NotInitialized;
            double topMargin = 10;
            string path = string.Empty;
            string htmlContent = string.Empty;
            string headerPath = string.Empty;
            var transaction = new TransactionHistory();

            try
            {
                result = _dynamicReportService.PrepareReportQuery(ref reportModel);

                if (result  == GlobalErrors.Success)
                {
                    reportModel.ReportFilters.RemoveAll(x => x.Disabled && (x.IsDateFilter || x.IsDateTimeFilter));
                    if (reportModel.ReportFilters != null && reportModel.ReportFilters.Count > 0)
                    {
                        for (int index = 0; index < reportModel.ReportFilters.Count; index++)
                        {
                            var filter = reportModel.ReportFilters[index];
                            bool isEven = index % 2 == 0;
                            reportModel.FiltersData.Add(new KeyValueModel(isEven, filter.ParameterDescription, filter.SelectedText));
                        }
                    }
                    result = _dynamicReportService.ExportReport(ref path, ref reportModel);
                }
                if (result == GlobalErrors.Success && reportModel != null && reportModel.ReportData != null && reportModel.ReportData.Count > 0)
                {
                    string pageFooter = ResourcesManager.TranslateKey(CaptionsConstants.Desc_Report_Page_Footer, _requestRepository.LanguageId);
                    string ofString = ResourcesManager.TranslateKey(CaptionsConstants.Desc_Of_String, _requestRepository.LanguageId);
                    string footer = string.Format("{0} [page] {1} [toPage]", pageFooter, ofString);
                    RazorTemplateEngine.Initialize();
                    htmlContent = RazorTemplateEngine.RenderAsync(path, reportModel).Result;

                    if (_requestRepository.Configurations.PrintHeaderPerPage)
                    {
                        double totalLines = 0;
                        result = _sharedMethod.PrepareCustomHeader(ref htmlContent, ref headerPath);

                        if (reportModel != null)
                        {
                            var detail = reportModel;
                            object value = null;
                            if (((IDictionary<string, object>)detail).ContainsKey("HeaderLines"))
                            {
                                ((IDictionary<string, object>)detail).TryGetValue("HeaderLines", out value);
                            }
                            if (value != null && !string.IsNullOrEmpty(value.ToString()))
                            {
                                totalLines = double.Parse(value.ToString());
                            }
                        }
                        double singleLineMargin = 7;

                        topMargin = (singleLineMargin * (totalLines));
                    }

                    if (result != GlobalErrors.Success) return View();

                    var globalSettings = new GlobalSettings
                    {
                        ColorMode = ColorMode.Color,
                        Orientation = Orientation.Portrait,
                        PaperSize = PaperKind.A4,
                        DocumentTitle = reportModel.ReportName,
                        Margins = new MarginSettings { Top = topMargin, Left = 3, Right = 3 },
                        DPI = 500
                    };

                    var objectSettings = new ObjectSettings
                    {
                        PagesCount = true,
                        HtmlContent = htmlContent,
                        WebSettings = { DefaultEncoding = "utf-8", UserStyleSheet = Path.Combine(Directory.GetCurrentDirectory(), "PDFUtility", "ReportsStyle.css") },
                        HeaderSettings = { FontName = "Arial", FontSize = 9, HtmUrl = headerPath },
                        FooterSettings = { FontName = "Arial", FontSize = 9, Left = DateTime.Now.ToString(), Right = footer, Spacing = 1.8 }
                    };
                    var pdf = new HtmlToPdfDocument()
                    {
                        GlobalSettings = globalSettings,
                        Objects = { objectSettings }
                    };
                    return File(_converter.Convert(pdf), "application/pdf");
                }
                else
                {
                    return View();
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return View();
            }
        }
        [HttpPost]
        [ActionName("ExportExcelFromGrid")]
        public IActionResult ExportExcelFromGrid([FromBody] DynamicReportModel reportModel)
        {
            var result = GlobalErrors.NotInitialized;
            double topMargin = 10;
            string path = string.Empty;
            string htmlContent = string.Empty;
            string headerPath = string.Empty;
            var transaction = new TransactionHistory();

            try
            {
                result = _dynamicReportService.PrepareReportQuery(ref reportModel);

                if (result == GlobalErrors.Success)
                {
                    reportModel.ReportFilters.RemoveAll(x => x.Disabled && (x.IsDateFilter || x.IsDateTimeFilter));
                    if (reportModel.ReportFilters != null && reportModel.ReportFilters.Count > 0)
                    {
                        for (int index = 0; index < reportModel.ReportFilters.Count; index++)
                        {
                            var filter = reportModel.ReportFilters[index];
                            bool isEven = index % 2 == 0;
                            reportModel.FiltersData.Add(new KeyValueModel(isEven, filter.ParameterDescription, filter.SelectedText));
                        }
                    }
                    result = _dynamicReportService.ExportExcel(ref reportModel);
                }
                if (result == GlobalErrors.Success && reportModel != null && reportModel.ReportData != null && reportModel.ReportData.Count > 0)
                {
                    XLWorkbook workbook = new XLWorkbook();
                    IXLWorksheet worksheet = workbook.Worksheets.Add("Sheet1");

                    // Header Information
                    // Report name
                    int reportNameRow = 2;
                    int reportNameColumn = 6;
                    worksheet.Range(reportNameRow, reportNameColumn, reportNameRow, reportNameColumn + 3).Merge().SetValue(reportModel.ReportName);
                    worksheet.Range(reportNameRow, reportNameColumn, reportNameRow, reportNameColumn + 3).Style.Font.Bold = true;
                    worksheet.Range(reportNameRow, reportNameColumn, reportNameRow, reportNameColumn + 3).Style.Font.FontSize = 16; // Set the font size to 14
                    worksheet.Range(reportNameRow, reportNameColumn, reportNameRow, reportNameColumn + 3).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                    if (reportModel.FiltersData.Count > 0)
                    {
                        int headerRow = 4;
                        int headerColumn = 3;
                        int headerMergeEndColumn = headerColumn + 1; // Update this based on your requirements for merging cells

                        for (int i = 0; i < reportModel.FiltersData.Count; i += 2)
                        {
                            string key1 = reportModel.FiltersData[i].Key + ":";
                            string value1 = reportModel.FiltersData[i].Value;

                            string key2 = string.Empty;
                            string value2 = string.Empty;

                            // Check if there's another set of data available
                            if (i + 1 < reportModel.FiltersData.Count)
                            {
                                key2 = reportModel.FiltersData[i + 1].Key + ":";
                                value2 = reportModel.FiltersData[i + 1].Value;
                            }

                            worksheet.Range(headerRow, headerColumn, headerRow, headerMergeEndColumn + 1).Merge().SetValue(key1);
                            worksheet.Range(headerRow, headerMergeEndColumn +2 , headerRow, headerMergeEndColumn + 4).Merge().SetValue(value1);

                            // Move to the next pair of cells for the next data entry
                            headerColumn = headerMergeEndColumn + 5;
                            worksheet.Range(headerRow, headerColumn, headerRow, headerColumn).Merge().SetValue(key2);
                            worksheet.Range(headerRow, headerColumn + 1, headerRow, headerColumn + 4).Merge().SetValue(value2);

                            headerRow++;
                            // Reset headerColumn for the next row
                            headerColumn = 3;
                        }
                    }
                    // line
                    // Set borders for the inserted row
                    int targetRow = 7; // Row number where you want to add the line
                    IXLRow row = worksheet.Row(targetRow);
                    row.Style.Border.BottomBorder = XLBorderStyleValues.Thin;
                    //// Merge the cells for the header section and style it
                    worksheet.SheetView.FreezeRows(9);
                    int startRow = 1;
                    int endRow = 9;

                    for (int rowNumber = startRow; rowNumber <= endRow; rowNumber++)
                    {
                        IXLRow headerRange = worksheet.Row(rowNumber);
                        headerRange.Style.Border.LeftBorder = XLBorderStyleValues.None;
                        headerRange.Style.Border.RightBorder = XLBorderStyleValues.None;
                        headerRange.Style.Border.TopBorder = XLBorderStyleValues.None;
                        headerRange.Style.Border.BottomBorder = XLBorderStyleValues.None;
                        headerRange.Style.Fill.BackgroundColor = XLColor.White;
                    }
                    targetRow = 10; // Row number where you want to remove the borders
                    IXLRow tenRow = worksheet.Row(targetRow);
                    tenRow.Style.Border.LeftBorder = XLBorderStyleValues.None;
                    tenRow.Style.Border.RightBorder = XLBorderStyleValues.None;
                    tenRow.Style.Border.TopBorder = XLBorderStyleValues.None;
                    tenRow.Style.Border.BottomBorder = XLBorderStyleValues.None;
                    tenRow.Style.Fill.BackgroundColor = XLColor.White;

                    int targetColumn = 1; // Column number where you want to remove the borders
                    IXLColumn column = worksheet.Column(targetColumn);
                    column.Style.Border.LeftBorder = XLBorderStyleValues.None;
                    column.Style.Border.RightBorder = XLBorderStyleValues.None;
                    column.Style.Border.TopBorder = XLBorderStyleValues.None;
                    column.Style.Border.BottomBorder = XLBorderStyleValues.None;
                    column.Style.Fill.BackgroundColor = XLColor.White;

                    var currentRow = 11;
                    var currentColumn = 2;
                    int mergeEndRow = currentRow + 2; // Three rows to merge
                    int mergeEndColumn = currentColumn + 5; // Three rows to merge
                    worksheet.Range(currentRow, currentColumn, mergeEndRow, mergeEndColumn).Merge().SetValue("Customers");
                    worksheet.Range(currentRow, currentColumn, mergeEndRow, mergeEndColumn).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                    worksheet.Range(currentRow, currentColumn, mergeEndRow, mergeEndColumn).Style.Alignment.Vertical = XLAlignmentVerticalValues.Center;
                    worksheet.Range(currentRow, currentColumn, mergeEndRow, mergeEndColumn).Style.Alignment.WrapText = true;
                    worksheet.Range(currentRow, currentColumn, mergeEndRow, mergeEndColumn).Style.Font.FontSize = 14;
                    //worksheet.Cell(currentRow, currentColumn).SetValue("Customers");
                    currentColumn = mergeEndColumn;
                    // Add divisions as headers in the first row starting from the second column
                    var divisionCategoryItems = reportModel.ReportData
                                                .GroupBy(x => new { x.DivisionID, x.Division, x.ItemCategory, x.ItemID, x.ItemName })
                                                .Select(g => new
                                                {
                                                    DivisionID = g.Key.DivisionID,
                                                    Division = g.Key.Division,
                                                    ItemCategory = g.Key.ItemCategory,
                                                    ItemID = g.Key.ItemID,
                                                    ItemName = g.Key.ItemName
                                                })
                                                .ToList();

                    var distinctDivisions = divisionCategoryItems
                        .Select(x => new { x.DivisionID, x.Division })
                        .Distinct()
                        .ToList();

                    foreach (var division in distinctDivisions)
                    {
                        currentColumn++;
                        var categoriesForDivision = divisionCategoryItems
                            .Where(x => x.DivisionID == division.DivisionID)
                            .Select(x => x.ItemCategory)
                            .Distinct()
                            .ToList();
                        int totalCategories = categoriesForDivision.Count();
                        int totalDivision = 0;
                        int totalItems = 0;
                        // Iterate over categories for the current division
                        foreach (var category in categoriesForDivision)
                        {
                            var itemsInCategory = divisionCategoryItems
                                .Where(x => x.DivisionID == division.DivisionID && x.ItemCategory == category)
                                .Select(x => new { x.ItemID, x.ItemName })
                                .Distinct()
                                .ToList();
                            totalItems = itemsInCategory.Count();
                            totalDivision += totalItems;
                            //totalDivision = categoriesForDivision.Count() + itemsInCategory.Count();
                            // Now 'itemsInCategory' contains all items for the current category under the division
                            // You can use this data to perform further operations or display the information
                        }
                        int mergeEndColumnDivision = currentColumn + totalDivision -1;
                        //int mergeEndColumnDivision = currentColumn + totalDivision - 2;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnDivision).Merge().SetValue(division.Division);
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnDivision).Style.Fill.BackgroundColor = XLColor.Orange;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnDivision).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnDivision).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnDivision).Style.Alignment.WrapText = true;

                        currentColumn = mergeEndColumnDivision;
                    }
                    currentRow = 12; // Start on the second row for categories
                    currentColumn = mergeEndColumn; // Start on the first column for categories

                    foreach (var division in distinctDivisions)
                    {
                        var categoriesForDivision = divisionCategoryItems
                            .Where(x => x.DivisionID == division.DivisionID)
                            .Select(x =>  x.ItemCategory )
                            .Distinct()
                            .ToList();
                        foreach (var category in categoriesForDivision)
                        {
                            currentColumn++;
                            var itemsInCategory = divisionCategoryItems.Where(x => x.DivisionID == division.DivisionID && x.ItemCategory == category)
                                             .Select(x => new { x.ItemID, x.ItemName })
                                             .Distinct().ToList();

                            var totalItemsCount = itemsInCategory.Count();
                            int mergeEndColumnCategory = currentColumn + totalItemsCount - 1;
                            //if (totalItemsCount == 1)
                            //    mergeEndColumnCategory = currentColumn;
                            worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnCategory).Merge().SetValue(category);
                            worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnCategory).Style.Fill.BackgroundColor = XLColor.Yellow;
                            worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnCategory).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                            worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnCategory).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                            worksheet.Range(currentRow, currentColumn, currentRow, mergeEndColumnCategory).Style.Alignment.WrapText = true;
                            currentColumn = mergeEndColumnCategory;
                        }
                    }
                    currentRow = 13; // Start on the second row for categories
                    currentColumn = mergeEndColumn; // Start on the first column for categories 
                    foreach (var division in distinctDivisions)
                    {
                        var categoriesForDivision = divisionCategoryItems
                                                    .Where(x => x.DivisionID == division.DivisionID)
                                                    .Select(x =>  x.ItemCategory)
                                                    .Distinct()
                                                    .ToList();
                        foreach (var category in categoriesForDivision)
                        {
                            var itemsInCategory = divisionCategoryItems.Where(x => x.DivisionID == division.DivisionID && x.ItemCategory == category)
                                                                 .Select(x => new { x.ItemID, x.ItemName })
                                                                 .Distinct().ToList();

                            foreach (var item in itemsInCategory)
                            {
                                currentColumn++;
                                worksheet.Range(currentRow, currentColumn, currentRow, currentColumn).Merge().SetValue(item.ItemName);
                                worksheet.Cell(currentRow, currentColumn).Style.Fill.BackgroundColor = XLColor.LightGray;
                                worksheet.Cell(currentRow, currentColumn).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                                worksheet.Cell(currentRow, currentColumn).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                                worksheet.Cell(currentRow, currentColumn).Style.Alignment.WrapText = true;
                                worksheet.Column(currentColumn).Width = 25;
                            }
                        }
                    }
                    int lastColumn = currentColumn; // last column for the table
                    currentRow = 14;
                    currentColumn = 2;
                    int mergeEndCustomerColumn = currentColumn + 2;
                    int customerOutletColumns = mergeEndCustomerColumn + 3; // 2 columns for CustomerName and 3 columns for OutletName

                    foreach (var customer in reportModel.ReportData.Select(x => new { x.CustomerName, x.OutletName }).Distinct())
                    {
                        // Set values for CustomerName
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndCustomerColumn).Merge().SetValue(customer.CustomerName);
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndCustomerColumn).Style.Alignment.WrapText = true;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndCustomerColumn).Style.Fill.BackgroundColor = XLColor.LightGray;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndCustomerColumn).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                        worksheet.Range(currentRow, currentColumn, currentRow, mergeEndCustomerColumn).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;

                        // Set values for OutletName
                        worksheet.Range(currentRow, mergeEndCustomerColumn + 1, currentRow, mergeEndCustomerColumn + 3).Merge().SetValue(customer.OutletName);
                        worksheet.Range(currentRow, mergeEndCustomerColumn + 1, currentRow, mergeEndCustomerColumn + 3).Style.Alignment.WrapText = true;
                        worksheet.Range(currentRow, mergeEndCustomerColumn + 1, currentRow, mergeEndCustomerColumn + 3).Style.Fill.BackgroundColor = XLColor.LightGray;
                        worksheet.Range(currentRow, mergeEndCustomerColumn + 1, currentRow, mergeEndCustomerColumn + 3).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                        worksheet.Range(currentRow, mergeEndCustomerColumn + 1, currentRow, mergeEndCustomerColumn + 3).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;

                        int currentItemColumn = customerOutletColumns + 1;
                        // Iterate through divisions, categories, and items for the current customer
                        foreach (var division in distinctDivisions)
                        {
                            var categoriesForDivision = divisionCategoryItems
                                                        .Where(x => x.DivisionID == division.DivisionID)
                                                        .Select(x => x.ItemCategory)
                                                        .Distinct()
                                                        .ToList();

                            foreach (var category in categoriesForDivision)
                            {
                                var itemsInCategory = divisionCategoryItems
                                                        .Where(x => x.DivisionID == division.DivisionID && x.ItemCategory == category)
                                                        .Select(x => new { x.ItemID, x.ItemName })
                                                        .Distinct()
                                                        .ToList();

                                foreach (var item in itemsInCategory)
                                {
                                    var valueForItem = reportModel.ReportData
                                                        .Where(x => x.DivisionID == division.DivisionID && x.ItemCategory == category && x.ItemName == item.ItemName && x.CustomerName == customer.CustomerName)
                                                        .Select(x => x.value)
                                                        .FirstOrDefault();

                                    // Set the value in the worksheet
                                    if (currentColumn <= lastColumn)
                                    {
                                        if (valueForItem != null && (valueForItem == "Y" || valueForItem == "N"))
                                        {
                                            worksheet.Cell(currentRow, currentItemColumn).SetValue(valueForItem);
                                            worksheet.Cell(currentRow, currentItemColumn).Style.Font.SetFontColor(valueForItem == "Y" ? XLColor.Green : XLColor.Red);
                                            worksheet.Cell(currentRow, currentItemColumn).Style.Alignment.Vertical = XLAlignmentVerticalValues.Top;
                                            worksheet.Cell(currentRow, currentItemColumn).Style.Alignment.Horizontal = XLAlignmentHorizontalValues.Center;
                                            worksheet.Cell(currentRow, currentItemColumn).Style.Alignment.WrapText = true;
                                            currentItemColumn++;
                                        }
                                        else
                                        {
                                            worksheet.Cell(currentRow, currentItemColumn).SetValue(string.Empty);
                                            currentItemColumn++;
                                        }
                                    }
                                }
                            }
                        }

                        // Move to the next row for the next customer
                        currentRow++;
                        // Reset column for the next customer
                        currentColumn = 2;
                    }
                    int lastRow = currentRow - 1; //last row for table
                    currentRow = 14;
                    currentColumn = mergeEndColumn + 1;
                    startRow = 11; // Start row of your table
                    int startColumn = 2;
                    var firstCell = worksheet.Cell(startRow, startColumn);
                    var lastCell = worksheet.Cell(lastRow, lastColumn);

                    if (firstCell != null && lastCell != null)
                    {
                        var tableRange = worksheet.Range(firstCell.Address, lastCell.Address);

                        // Set outer borders for the table
                        tableRange.Style.Border.OutsideBorder = XLBorderStyleValues.Thin;

                        // Set inner borders for the table
                        tableRange.Style.Border.InsideBorder = XLBorderStyleValues.Thin;
                    }
                    using (var stream = new MemoryStream())
                    {
                        workbook.SaveAs(stream);
                        var content = stream.ToArray();
                        return File(content, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", CoreDataBaseConstants.SheetsConstants.PosStockCountingSheet);
                    }
                }
                else
                {
                    return View();
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return View();
            }
        }
        #region [Report Schedule]
        [HttpPost]
        [ActionName("GetReportSchedules")]
        public IActionResult GetReportSchedules()
        {
            try
            {
                List<ReportScheduleModel> data = new List<ReportScheduleModel>();
                GlobalErrors result = _dynamicReportService.GetReportSchedule(ref data);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("SaveReportScheduleData")]
        public IActionResult SaveReportScheduleData([FromBody] DynamicReportModel reportModel)
        {
            try
            {
                var result = _dynamicReportService.SaveReportScheduleData(reportModel);

                if (result == GlobalErrors.Success)
                {                 
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("PopulateScheduledReports")]
        public IActionResult PopulateScheduledReports([FromBody] int reportId)
        {
            try
            {
                List<ReportScheduleModel> reportSchedule = new List<ReportScheduleModel>();
                List<ReportFilterModel> reportFilters = new List<ReportFilterModel>();
                
                GlobalErrors result = _dynamicReportService.PopulateScheduledReports(ref reportSchedule, ref reportFilters, reportId);

                var data = new
                {
                    data = reportSchedule,
                    Filters = reportFilters
                };

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetReportScheduleFilter")]
        public IActionResult GetReportScheduleFilter([FromBody] ReportScheduleModel reportSchedule)
        {
            try
            {
                string data = string.Empty;
                GlobalErrors result = _dynamicReportService.GetReportScheduleFilter(ref data , reportSchedule.ScheduleId , reportSchedule.ReportScheduleAssignmentId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("DeleteReportSchedule")]
        public IActionResult DeleteReportSchedule([FromBody] ReportScheduleModel reportSchedule)
        {
            try
            {
                GlobalErrors result = _dynamicReportService.DeleteReportScheduleAssignment(reportSchedule.ScheduleId, reportSchedule.ReportScheduleAssignmentId);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Deleted_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Delete_Failed, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion
    }
}
