﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Sonic.Web.DAL;
using Sonic.Web.Licensing;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Reflection;
using Newtonsoft.Json;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class EchoController : BaseController
    {
        private readonly IRequestRepository _requestRepository;
        private readonly EchoService _echoService;
        private readonly FQAsService _faqAsService;
        private IHostingEnvironment _hostingEnvironment;
        private IActiveTokens _activeToken;

        public EchoController(IHostingEnvironment hostingEnvironment, IRequestRepository requestRepository, ILicensing licensing, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _hostingEnvironment = hostingEnvironment;
            _requestRepository = requestRepository;
            _echoService = new EchoService(requestRepository);
            _faqAsService = new FQAsService(requestRepository);

            _activeToken = activeToken;
        }

        [HttpPost]
        [ActionName("GetAllFQAs")]
        public JsonResult GetAllFQAs([FromBody] FQAsModel fQAs)
        {
            try
            {
                SharedTableResult<FQAsModel> FQAsList = new SharedTableResult<FQAsModel>();

                GlobalErrors result = _faqAsService.GetFQASList(ref FQAsList);

                if (result == GlobalErrors.Success)
                {
                    if (FQAsList != null && FQAsList.Data != null && FQAsList.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(FQAsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("InsertFQA")]
        public JsonResult InsertFQA([FromBody] List<FQAsModel> fQAs)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _faqAsService.InsertFQA(fQAs);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon( ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetCustomerProperties")]
        public JsonResult GetCustomerProperties()
        {
            CustAppPropertiesModel customerProperties = new CustAppPropertiesModel();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetCustomerProperties(ref customerProperties);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(customerProperties, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("InsertCustomerProperties")]
        public JsonResult InsertCustomerProperties([FromBody] CustAppPropertiesModel customerProperties)
        {
            try
            {
                GlobalErrors result = _echoService.SaveCustomerProperties(customerProperties);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertBanner")]
        public JsonResult InsertBanner([FromForm] Collection<IFormFile> image, [FromForm] string banner)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                BannerModel bannerModel = JsonConvert.DeserializeObject<BannerModel>(banner);
                string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                result = _echoService.SaveBanner(bannerModel, image, hostingEnvironment);
                    if (result == GlobalErrors.Success)
                    {
                        return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                    }
                
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetBanners")]
        public JsonResult GetBanners()
        { 
            try
            {
                List<BannerModel> banner = new List<BannerModel>();
                var result = _echoService.GetBanners(ref banner);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(banner, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("DeleteBanner")]
        public JsonResult DeleteBanner(string fileName)
        {
            try
            {
                string errorMessage = string.Empty;
                var result = _echoService.DeleteBanner(fileName,ref errorMessage);
                if (result == GlobalErrors.Success) 
                {
                    string rootPath = _hostingEnvironment.ContentRootPath;
                    string folderName = LocalUtilities.BannerImageFolderPath + "\\"+ fileName;
                    var _imageToBeDeleted = Path.Combine(rootPath, folderName);
                    if ((System.IO.File.Exists(_imageToBeDeleted)))
                    {
                        System.IO.File.Delete(_imageToBeDeleted);
                    }
                    folderName = LocalUtilities.CustomerAppBannerImageFolderPath + "\\" + fileName;
                    string path = string.Format(@"{0}", LocalUtilities.CustomerAppServicePath);
                    _imageToBeDeleted = Path.Combine(path, folderName);
                    if ((System.IO.File.Exists(_imageToBeDeleted)))
                    {
                        System.IO.File.Delete(_imageToBeDeleted);
                    }
                    return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetContactUsList")]
        public JsonResult GetContactUsList([FromBody] FeedbackFilterModel filter)
        {
            try
            {
                SharedTableResult<FeedbackModel> list = new SharedTableResult<FeedbackModel>();
                GlobalErrors result = _echoService.GetContactUsList(filter, ref list);
                if (result == GlobalErrors.Success)
                {
                    if (list != null && list.Data != null && list.Data.ToList().Count > 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(list, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(list, ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertFeedbackReply")]
        public JsonResult InsertFeedbackReply([FromBody] FeedbackModel feedBackReply)
        {
            try
            {
                GlobalErrors result = _echoService.SaveFeedbackReply(feedBackReply);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetEchoItems")]
        public JsonResult GetEchoItems()
        {
            SharedTableResult<ItemPackModel> itemPacklistSharedTable = new SharedTableResult<ItemPackModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetEchoItems(ref itemPacklistSharedTable);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(itemPacklistSharedTable, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("InsertEchoItems")]
        public JsonResult InsertEchoItems([FromBody] List<ItemPackModel> PacksList)
        {
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.SaveEchoItems(PacksList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetEchoDivisions")]
        public JsonResult GetEchoDivisions()
        {
            List<DivisionModel> divisionsList = new List<DivisionModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetEchoDivisions(ref divisionsList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(divisionsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetEchoCategories")]
        public JsonResult GetEchoCategories(int DivisionId)
        {
            List<CategoryModel> categoriesList = new List<CategoryModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetEchoCategories(DivisionId, ref categoriesList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(categoriesList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetEchoBrands")]
        public JsonResult GetEchoBrands(int divisionID, int itemCategoryId)
        {
            List<BrandModel> brandsList = new List<BrandModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetEchoBrands(divisionID, itemCategoryId, ref brandsList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(brandsList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetEchoFirstOrderItems")]
        public JsonResult GetEchoFirstOrderItems(string itemsFilter, bool getAddedItems)
        {
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                GeneralFilter filter = JsonConvert.DeserializeObject<GeneralFilter>(itemsFilter);
                List<ItemPackModel> firstOrderFreeItems = new List<ItemPackModel>();
                result = _echoService.GetEchoFirstOrderItems(filter, getAddedItems, ref firstOrderFreeItems);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(firstOrderFreeItems, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("InsertEchoFirstOrderItems")]
        public JsonResult InsertEchoFirstOrderItems([FromBody] List<ItemPackModel> addedFirstOrderFreeItems)
        {
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
                result = _echoService.SaveEchoFirstOrderItems(addedFirstOrderFreeItems);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetCustomerAppSettings")]
        public JsonResult GetCustomerAppSettings()
        {
            List<CustomerAppSettingsModel> customerAppSettingslistSharedTable = new List<CustomerAppSettingsModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetCustomerAppSettings(ref customerAppSettingslistSharedTable);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(customerAppSettingslistSharedTable, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateCustomerAppSettings")]
        public JsonResult UpdateCustomerAppSettings([FromBody] CustomerSettingsObject obj)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if((obj.SettingsList != null && obj.SettingsList.Count > 0) || (obj.CustomizeList != null && obj.CustomizeList.Count > 0))
                {
                    result = _echoService.UpdateCustomerAppSettings(obj.SettingsList, obj.CustomizeList);

                }
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetCustomizeHomePage")]
        public JsonResult GetCustomizeHomePage()
        {
            List<EchoCustomeHomePageModel> customizeHomePageList = new List<EchoCustomeHomePageModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetCustomizeHomePage(ref customizeHomePageList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(customizeHomePageList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetCompany")]
        public JsonResult GetCompany()
        {
            List<CompanyDetails> companyDetails = new List<CompanyDetails>();
            try{
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetCompany(ref companyDetails);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(companyDetails, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrganizationListCompanyDetails")]
        public JsonResult GetOrganizationListCompanyDetails(bool IsEditMode)
        {
            List<OrganizationModel> organizationList = new List<OrganizationModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetOrganizationListCompanyDetails(IsEditMode, ref organizationList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(organizationList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetCompanyDetails")]
        public JsonResult GetCompanyDetails(int organizationId)
        {
            CompanyDetails companyDetails = new CompanyDetails();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetCompanyDetails(organizationId, ref companyDetails);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(companyDetails, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertCompanyDetails")]
        public JsonResult InsertCompanyDetails([FromBody] CompanyDetails CompanyDetailsObject)
        {
            try
            {
                GlobalErrors result = _echoService.InsertCompanyDetails(CompanyDetailsObject);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateCompanyDetails")]
        public JsonResult UpdateCompanyDetails([FromBody] CompanyDetails CompanyDetailsObject)
        {
            try
            {
                GlobalErrors result = _echoService.UpdateCompanyDetails(CompanyDetailsObject);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("DeleteCompanyDetails")]
        public JsonResult DeleteCompanyDetails(int organizationId)
        {
            try
            {
                string errorMessage = string.Empty;
                GlobalErrors result = _echoService.DeleteCompanyDetails(organizationId, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrganizationListOrganizationDetails")]
        public JsonResult GetOrganizationListOrganizationDetails(bool IsEditMode)
        {
            List<OrganizationModel> organizationList = new List<OrganizationModel>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetOrganizationListOrganizationDetails(IsEditMode, ref organizationList);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(organizationList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrganization")]
        public JsonResult GetOrganization()
        {
            List<OrganizationDetails> organizationDetails = new List<OrganizationDetails>();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetOrganization(ref organizationDetails);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(organizationDetails, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrganizationDetails")]
        public JsonResult GetOrganizationDetails(int organizationId)
        {
            OrganizationDetails organizationDetails = new OrganizationDetails();
            try
            {
                GlobalErrors result = GlobalErrors.Success;
                result = _echoService.GetOrganizationDetails(organizationId, ref organizationDetails);
                if (result == GlobalErrors.Success)
                {

                    return JSonResultResponse.GetSuccessrJSon(organizationDetails, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("InsertOrganizationDetails")]
        public JsonResult InsertOrganizationDetails([FromBody] OrganizationDetails OrganizationDetailsObject)
        {
            try
            {
                GlobalErrors result = _echoService.InsertOrganizationDetails(OrganizationDetailsObject);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateOrganizationDetails")]
        public JsonResult UpdateOrganizationDetails([FromBody] OrganizationDetails OrganizationDetailsObject)
        {
            try
            {
                GlobalErrors result = _echoService.UpdateOrganizationDetails(OrganizationDetailsObject);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("DeleteOrganizationDetails")]
        public JsonResult DeleteOrganizationDetails(int organizationId)
        {
            try
            {
                string errorMessage = string.Empty;
                GlobalErrors result = _echoService.DeleteOrganizationDetails(organizationId, ref errorMessage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}