﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.Model.Customer;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;

namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class ItemCategoryController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly CustomerService _customerService;
        private readonly MasterDataService _masterDataService;
        private readonly IHostingEnvironment _hostingEnvironment;
        private readonly IWebHostEnvironment _webHostEnvironment;
        private readonly ItemService _itemService;

        public ItemCategoryController(IRequestRepository requestRepository, IWebHostEnvironment webHostEnvironment, ILicensing licensing, IHostingEnvironment hostingEnvironment, IActiveTokens activeTokens) : base(requestRepository, licensing, activeTokens)
        {
            _requestRepository = requestRepository;
            _customerService = new CustomerService(_requestRepository);
            _masterDataService = new MasterDataService(requestRepository);
            _hostingEnvironment = hostingEnvironment;
            _webHostEnvironment = webHostEnvironment;
            _itemService = new ItemService(_requestRepository);
        }

        #region GET
        [HttpGet]
        [ActionName("GetItemCategoryList")]
        public JsonResult GetItemCategoryList()
        {
            try
            {
                List<ItemCategoryModel> categoryList = new List<ItemCategoryModel>();

                var result = _itemService.GetItemCategoryList(ref categoryList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(categoryList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetCategoryData")]
        public JsonResult GetCategoryData([FromBody] int categoryId)
        {
            try
            {
                var data = new CategoryModel();
                var result = _itemService.GetCategoryData(categoryId, ref data);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("CheckCategoryCodeExist")]
        public JsonResult CheckCategoryCodeExist([FromBody] CategoryModel itemCategory)
        {
            try
            {
                bool exist = false;
                var result = _itemService.CheckCodeIsExist("ItemCategory", "ItemCategoryCode", itemCategory.CategoryNameCode, "ItemCategoryId", itemCategory.ItemCategoryId, ref exist);
                var data = new
                {
                    isExist = exist,
                };

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, "");
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon();
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetAllCategories")]
        public JsonResult GetAlllCategories([FromBody] ItemCategoryFilter filter)
        {
            try
            {
                var data = new List<CategoryModel>();
                GlobalErrors result = _itemService.GetAlllCategories(filter, ref data);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        #endregion

        #region UPDATE
        [HttpPost]
        [ActionName("UpdateCategory")]
        public JsonResult UpdateCategory([FromForm] Collection<IFormFile> image, [FromForm] string category)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                CategoryModel categoryModel = JsonConvert.DeserializeObject<CategoryModel>(category);
                if (categoryModel != null)
                {
                    string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                    var result = _itemService.EditCategory(categoryModel, image, hostingEnvironment);
                    if (result == GlobalErrors.Success)
                    {
                        return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                    }
                    else if (result == GlobalErrors.CustomerAppPath)
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorAppPath, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region INSERT
        [HttpPost]
        [ActionName("InsertCategory")]
        public JsonResult InsertCategory([FromForm] Collection<IFormFile> image, [FromForm] string category)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Provide_UserName_Password, _requestRepository.LanguageId));
                }
                CategoryModel categoryModel = JsonConvert.DeserializeObject<CategoryModel>(category);
                if (categoryModel != null)
                {
                    string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                    var result = _itemService.SaveCategory(categoryModel, image, hostingEnvironment);
                    if (result == GlobalErrors.Success)
                    {
                        return JSonResultResponse.GetSuccessrJSon(true, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                    }
                    else if (result == GlobalErrors.CustomerAppPath)
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorAppPath, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region DELETE
        [HttpPost]
        [ActionName("DeleteCategory")]
        public JsonResult DeleteCategory([FromBody] int categoryId)
        {
            try
            {
                string hostingEnvironment = _hostingEnvironment.ContentRootPath;
                bool isCategoryLinkedToItem = false;
                GlobalErrors result = _itemService.DeleteCategory(categoryId, hostingEnvironment, ref isCategoryLinkedToItem);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    string errorMessage = isCategoryLinkedToItem ? MessagesConstants.Desc_Category_Linked_To_Item : MessagesConstants.Desc_Error_While_Saving;
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion
    }
}