﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.PL.Helper;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class OrderApprovedController : BaseController
    {
        private readonly IRequestRepository _requestRepository;
        private readonly OrderService _orderService;

        public OrderApprovedController(IRequestRepository requestRepository, ILicensing licensing, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _requestRepository = requestRepository;
            _orderService = new OrderService(_requestRepository);
        }
        #region GET
        [HttpPost]
        [ActionName("GetOrders")]
        public JsonResult GetOrders([FromBody] OrderListFilter filter)
        {
            try
            {
                var orderList = new SharedTableResult<OrderModel>();
                var result = _orderService.GetOrderList(filter, ref orderList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(orderList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetOrder")]
        public JsonResult GetOrder([FromBody] OrderListFilter filter)
        {
            try
            {
                var order = new OrderModel();
                var result = _orderService.GetOrder(filter, ref order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        [HttpPost]
        [ActionName("UpdateOrder")]
        public JsonResult UpdateOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                string message = string.Empty;
                var result = _orderService.UpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);

                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("IsOrderValidToBill")]
        public JsonResult IsOrderValidToBill([FromBody] OrderModel order)
        {
            string errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId);
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                var result = _orderService.IsOrderValidToInvoice(ref order, ref errorMesage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(errorMesage);
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("DeliveryOrder")]
        public JsonResult DeliveryOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                List<SerialCurrentStatusModel> invalidSerials = new List<SerialCurrentStatusModel>();
                string errorMessage = string.Empty;
                bool invalidZATCATransaction = false;
                var result = _orderService.InvoiceOrder(order, "", ref invalidSerials, null, ref errorMessage, ref invalidZATCATransaction);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    if (result == GlobalErrors.SomeInvalidSerials)
                    {
                        return JSonResultResponse.GetErrorJSon(invalidSerials, ResourcesManager.TranslateKey(MessagesConstants.Desc_Some_Serials_Not_Valid, _requestRepository.LanguageId));
                    }
                    else if (result == GlobalErrors.AllSerialsInvalid)
                        return JSonResultResponse.GetErrorJSon(invalidSerials, ResourcesManager.TranslateKey(MessagesConstants.Desc_All_Serials_Not_Valid, _requestRepository.LanguageId));
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_While_Saving : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }



        [HttpPost]
        [ActionName("DisapproveOrder")]
        public JsonResult DisapproveOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                string message = string.Empty;
                var result = _orderService.UpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);

                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }



        [HttpPost]
        [ActionName("AssignOrder")]
        public JsonResult AssignOrder([FromBody] List<OrderModel> orders)
        {
            string errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId);
            try
            {
                var result = _orderService.CheckAndAssignOrders(orders, ref errorMesage);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(orders, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(errorMesage);
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #region VALIDATION
        [HttpPost]
        [ActionName("IsValidOperation")]
        public JsonResult IsValidOperation([FromBody] OrderModel order)
        {
            try
            {
                var operation = new OrderOperationModel();
                var result = _orderService.IsValidOperation(order, ref operation);
                if (result == GlobalErrors.Success || operation.OnlyShowWarning)
                {
                    return JSonResultResponse.GetSuccessrJSon(operation, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

 

        [HttpPost]
        [ActionName("CheckForPostedCustomerTransactions")]
        public JsonResult CheckForPostedCustomerTransactions([FromBody] OrderModel order)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
                bool showWaitForUpdateWarning = false;
                result = _orderService.CheckForPostedCustomerTransactions(order, ref showWaitForUpdateWarning);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(showWaitForUpdateWarning, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("CheckOrderStatusBeforeAnyAction")]
        public JsonResult CheckOrderStatusBeforeAnyAction([FromBody] OrderModel order)
        {
            try
            {
                string message = string.Empty;
                bool isValid = _orderService.CheckOrderStatusBeforeAnyAction(order);
                if (isValid)
                    isValid = _orderService.IsThereRelatedToTargetPromDependsOnThisOrder(order, ref message);
                if (isValid)
                {
                    return JSonResultResponse.GetSuccessrJSon(isValid, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    if (message != "")
                        return JSonResultResponse.GetErrorJSon(message);
                    else
                        return JSonResultResponse.GetErrorJSon(isValid, ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

    }
}
