﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.PL.Helper;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Sonic.Web.PL.Controllers
{
    
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class OrderController : BaseController
    {

        private readonly IRequestRepository _requestRepository;
        private readonly OrderService _orderService;

        public OrderController(IRequestRepository requestRepository, ILicensing licensing, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _requestRepository = requestRepository;
            _orderService = new OrderService(_requestRepository);
        }

        [HttpPost]
        [ActionName("VoidOrderInvoice")]
        public JsonResult VoidOrderInvoice([FromBody] OrderModel order)
        {
            try
            {
                string message = string.Empty;
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                var result = _orderService.VoidOrderInvoiceAndUpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (message == string.Empty)
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                    else
                        return JSonResultResponse.GetErrorJSon(message);
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }



        [HttpPost]
        [ActionName("GetDashboardsData")]
        public async Task<JsonResult> GetDashboardsData([FromBody] TimePeriodModel timePeriod)
        {
            try
            {
                var dashboardsData = await _orderService.GetDashboardsData(timePeriod.IsToday, timePeriod.IsMonthly, timePeriod.IsAnnual);

                if (dashboardsData != null)
                {
                    return await Task.Run(() =>
                     {
                         return JSonResultResponse.GetSuccessrJSon(dashboardsData, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                     });
                }
                else
                {
                    return await Task.Run(() =>
                     {
                         return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                     });
                }
            }
            catch (Exception ex)
            {
                return await Task.Run(() =>
                 {
                     return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                 });
            }
        }

        [HttpPost]
        [ActionName("GetOrder")]
        public JsonResult GetOrder([FromBody]OrderListFilter filter)
        {
            try
            {
                var order = new OrderModel();
                var result = _orderService.GetOrder(filter, ref order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetInvoicedOrderForView")]
        public JsonResult GetInvoicedOrderForView([FromBody]OrderListFilter filter)
        {
            try
            {
                var order = new OrderModel();
                var result = _orderService.GetInvoicedOrderForView( filter,ref order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }


        [HttpPost]
        [ActionName("InvoiceOrder")]
        public JsonResult InvoiceOrder([FromForm] string orderModel, Collection<IFormFile> attachedFiles)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                OrderModel order = JsonConvert.DeserializeObject<OrderModel>(orderModel);
                if (order != null)
                {
                    List<SerialCurrentStatusModel> invalidSerials = new List<SerialCurrentStatusModel>();
                    string errorMessage = string.Empty;
                    bool invalidZATCATransaction = false;
                    var result = _orderService.InvoiceOrder(order, "", ref invalidSerials, null, ref errorMessage, ref invalidZATCATransaction, false, attachedFiles);

                    if (result == GlobalErrors.Success)
                    {
                        var data = new
                        {
                            order = order,
                            invalidZATCATransaction = invalidZATCATransaction
                        };
                        return JSonResultResponse.GetSuccessrJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                    }
                    else
                    {
                        if (result == GlobalErrors.SomeInvalidSerials)
                        {
                            return JSonResultResponse.GetErrorJSon(invalidSerials, ResourcesManager.TranslateKey(MessagesConstants.Desc_Some_Serials_Not_Valid, _requestRepository.LanguageId));
                        }
                        else if (result == GlobalErrors.DataAlreadyExists)
                        {
                            return JSonResultResponse.GetErrorJSon(invalidSerials, ResourcesManager.TranslateKey(MessagesConstants.MsgRefreshData, _requestRepository.LanguageId));
                        }
                        else if (result == GlobalErrors.AllSerialsInvalid)
                        {
                            return JSonResultResponse.GetErrorJSon(invalidSerials, ResourcesManager.TranslateKey(MessagesConstants.Desc_All_Serials_Not_Valid, _requestRepository.LanguageId));
                        }
                        else
                        {
                            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_While_Saving : errorMessage, _requestRepository.LanguageId));
                        }
                    }
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
            return JSonResultResponse.GetErrorJSon("Desc_Saving_Faild");
        }

        [HttpPost]
        [ActionName("GetOrders")]
        public JsonResult GetOrders([FromBody] OrderListFilter filter)
        {
            try
            {
                var orderList = new SharedTableResult<OrderModel>(); 
                var result = _orderService.GetOrderList(filter, ref orderList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(orderList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }


        [HttpPost]
        [ActionName("IsOrderValidToAppoint")]
        public JsonResult IsOrderValidToAppoint([FromBody] OrderModel order)
        {
            string errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId);
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                var result = _orderService.IsOrderValidToAssign(ref order, ref errorMesage);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(errorMesage);
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }


 

        [HttpPost]
        [ActionName("InsertOrder")]
        public JsonResult InsertOrder([FromBody] OrderModel order)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                string message = string.Empty;
                result = _orderService.SaveOrderSonic(order, ref message);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);
                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("UpdateOrder")]
        public JsonResult UpdateOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                string message = string.Empty;
                var result = _orderService.UpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);

                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("ApproveSelectedOrders")]
        public JsonResult ApproveSelectedOrders([FromBody] List<OrderModel> orders)
        {
            try
            {
                int countOfFaild = 0;
                string message = _orderService.ApproveSelectedOrders(ref countOfFaild, orders);
                if(countOfFaild == 0)
                {
                    return JSonResultResponse.GetSuccessrJSon(countOfFaild, ResourcesManager.TranslateKey(MessagesConstants.Desc_Approved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    StringBuilder responseMessage = new StringBuilder();
                    responseMessage.AppendLine(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Approve_Orders, _requestRepository.LanguageId));
                    responseMessage.AppendLine(message);
                    return JSonResultResponse.GetErrorJSon(countOfFaild, responseMessage.ToString());
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateRecurringOrder")]
        public JsonResult UpdateRecurringOrder([FromBody] OrderCRMModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                var result = _orderService.EditRecurringOrder(order);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("IsValidOperation")]
        public JsonResult IsValidOperation([FromBody] OrderModel order)
        {
            try
            {
                var operation = new OrderOperationModel();
                var result = _orderService.IsValidOperation(order, ref operation);
                if (result == GlobalErrors.Success || operation.OnlyShowWarning)
                {
                    return JSonResultResponse.GetSuccessrJSon(operation, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CheckOrderStatusBeforeAnyAction")]
        public JsonResult CheckOrderStatusBeforeAnyAction([FromBody] OrderModel order)
        {
            try
            {
                string message = string.Empty;
                bool isValid = _orderService.CheckOrderStatusBeforeAnyAction(order);
                if (isValid)
                    isValid = _orderService.IsThereRelatedToTargetPromDependsOnThisOrder(order, ref message);
                if (isValid)
                {
                    return JSonResultResponse.GetSuccessrJSon(isValid, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    if (message != "") 
                       return JSonResultResponse.GetErrorJSon(message);
                    else
                       return JSonResultResponse.GetErrorJSon(isValid, ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetMaxOrderId")]
        public JsonResult GetMaxOrderId()
        {
            try
            {
                // 1. Get Max Order Id
                string orderId = _orderService.GetMaxOrderId();
                if (string.IsNullOrEmpty(orderId.Trim()))
                {
                    return JSonResultResponse.GetErrorJSon(MessagesConstants.Desc_Error_While_Getting_Data);
                }
                return JSonResultResponse.GetSuccessrJSon(orderId, MessagesConstants.Desc_Success_Getting_Data);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        //[HttpPost]
        //[ActionName("GetRecurringChangedReason")]
        //public JsonResult GetRecurringChangedReason()
        //{
        //    try
        //    {
        //        List<RecurringChangedReasoneModel> reasonList = new List<RecurringChangedReasoneModel>();
        //        GlobalErrors err = _orderService.GetOrderRecurringChangeReason(ref reasonList);
        //        if (err != GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetErrorJSon(MessagesConstants.Desc_Error_While_Getting_Data);
        //        }
        //        return JSonResultResponse.GetSuccessrJSon(reasonList, MessagesConstants.Desc_Success_Getting_Data);
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}


        //[HttpPost]
        //[ActionName("GetRecurringOrdersList")]
        //public JsonResult GetRecurringOrdersList()
        //{
        //    try
        //    {
        //        var recurringList = new List<RecurringModel>();
        //        var result = _orderService.GetRecurringHistory(ref recurringList);
        //        if (result == GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(recurringList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //        }
        //        else if (result == GlobalErrors.NoMoreRows)
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}
        //[HttpPost]
        //[ActionName("GetRecurringOrderDetails")]
        //public JsonResult GetRecurringOrderDetails([FromBody]RecurringModel recurringOrder)
        //{
        //    try
        //    {
        //        OrderCRMModel order = new OrderCRMModel();
        //        var result = _orderService.GetRecurringDetails(recurringOrder, ref order);
        //        int todayWeekNo = LocalUtilities.GetWeekNumber(DateTime.Today);
        //        bool isEven = todayWeekNo % 2 == 0;
        //        order.IsOddWeekNo = !isEven;
        //        if (result == GlobalErrors.Success)
        //        {
        //            return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
        //        }
        //        else if (result == GlobalErrors.NoMoreRows)
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Data_Found, _requestRepository.LanguageId));
        //        }
        //        else
        //        {
        //            return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return JSonResultResponse.GetErrorJSon();
        //    }
        //}


        [HttpPost]
        [ActionName("DeleteMultiOrders")]
        public JsonResult DeleteMultiOrders([FromBody] List<OrderModel> ordersList)
        {
            try
            {
                string errorMessage = string.Empty;
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                int countOfFaild = 0;
                var result = _orderService.DeleteMultiOrders(ordersList,ref errorMessage, ref countOfFaild);
                if(result == GlobalErrors.Success)
                {
                    if (countOfFaild == 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Delete_Successfully, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(errorMessage, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_Update_Data : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetOrderTypes")]
        public JsonResult GetOrderTypes([FromBody]List<int>orderTypesFilter)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<OrderTypeModel> orderTypes = new List<OrderTypeModel>();
                result = _orderService.GetOrderTypes(orderTypesFilter, ref orderTypes);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(orderTypes, ResourcesManager.TranslateKey( MessagesConstants.Desc_Success_Getting_Data , _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey( MessagesConstants.Desc_Error_Happend , _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("GetOrderSources")]
        public JsonResult GetOrderSources()
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<OrderSourceModel> sourceList = new List<OrderSourceModel>();
                result = _orderService.GetOrderSources(ref sourceList);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(sourceList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("CheckResellableItems")]
        public JsonResult CheckResellableItems([FromBody] List<int> orderItemsPackStatuses)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
                bool isContainResellable = false;
                if(orderItemsPackStatuses.Count > 0)
                {
                    result = _orderService.CheckResellableItems(orderItemsPackStatuses, ref isContainResellable);
                }

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(isContainResellable, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }


        [HttpPost]
        [ActionName("IsReturnOrderValidToOperate")]
        public JsonResult IsReturnOrderValidToOperate([FromBody] OrderModel order)
        {
            string errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId);
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                var result = _orderService.IsReturnOrderValidToProcess(ref order, ref errorMesage);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(errorMesage);
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CheckForPostedCustomerTransactions")]
        public JsonResult CheckForPostedCustomerTransactions([FromBody] OrderModel order)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
                bool showWaitForUpdateWarning = false;
                result = _orderService.CheckForPostedCustomerTransactions(order, ref showWaitForUpdateWarning);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(showWaitForUpdateWarning, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

    }
}