﻿using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Sonic.Web.DAL;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.PL.Helper;
using Sonic.Web.PL.SystemModels;
using Sonic.Web.Resources;
using Sonic.Web.Service;
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
namespace Sonic.Web.PL.Controllers
{
    [Route("api/[controller]/[action]")]
    [Authorize]
    public class OrderPendingController : BaseController
    {
        private readonly IRequestRepository _requestRepository;
        private readonly OrderService _orderService;

        public OrderPendingController(IRequestRepository requestRepository, ILicensing licensing, IActiveTokens activeToken) : base(requestRepository, licensing, activeToken)
        {
            _requestRepository = requestRepository;
            _orderService = new OrderService(_requestRepository);
        }
        #region GET
        [HttpPost]
        [ActionName("GetOrders")]
        public JsonResult GetOrders([FromBody] OrderListFilter filter)
        {
            try
            {
                var orderList = new SharedTableResult<OrderModel>();
                var result = _orderService.GetOrderList(filter, ref orderList);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(orderList, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("GetOrder")]
        public JsonResult GetOrder([FromBody] OrderListFilter filter)
        {
            try
            {
                var order = new OrderModel();
                var result = _orderService.GetOrder(filter, ref order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region INSERT
        [HttpPost]
        [ActionName("InsertOrder")]
        public JsonResult InsertOrder([FromBody] OrderModel order)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                string message = string.Empty;
                result = _orderService.SaveOrderSonic(order, ref message);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Saved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);
                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Saving, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region UPDATE

        [HttpPost]
        [ActionName("UpdateOrder")]
        public JsonResult UpdateOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                string message = string.Empty;
                var result = _orderService.UpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);

                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("SelectWarehouseOrder")]
        public JsonResult SelectWarehouseOrder([FromBody] OrderModel order)
        {

            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));

                var result = _orderService.UpdateWarehouseOrder(order);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("ApproveSelectedOrders")]
        public JsonResult ApproveSelectedOrders([FromBody] List<OrderModel> orders)
        {
            try
            {
                int countOfFaild = 0;
                string message = _orderService.ApproveSelectedOrders(ref countOfFaild, orders);
                if (countOfFaild == 0)
                {
                    return JSonResultResponse.GetSuccessrJSon(countOfFaild, ResourcesManager.TranslateKey(MessagesConstants.Desc_Approved_Successfully, _requestRepository.LanguageId));
                }
                else
                {
                    StringBuilder responseMessage = new StringBuilder();
                    responseMessage.AppendLine(ResourcesManager.TranslateKey(MessagesConstants.Desc_Cant_Approve_Orders, _requestRepository.LanguageId));
                    responseMessage.AppendLine(message);
                    return JSonResultResponse.GetErrorJSon(countOfFaild, responseMessage.ToString());
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion

        #region DELETE
        [HttpPost]
        [ActionName("DeleteOrder")]
        public JsonResult DeleteOrder([FromBody] OrderModel order)
        {
            try
            {
                if (!ModelState.IsValid)
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                string message = string.Empty;
                var result = _orderService.UpdateOrder(order, ref message);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(order, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Update_Data, _requestRepository.LanguageId));
                }
                else if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    if (!string.IsNullOrEmpty(message))
                    {
                        return JSonResultResponse.GetErrorJSon(message);

                    }
                    else
                        return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("DeleteMultiOrders")]
        public JsonResult DeleteMultiOrders([FromBody] List<OrderModel> ordersList)
        {
            try
            {
                string errorMessage = string.Empty;
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                int countOfFaild = 0;
                var result = _orderService.DeleteMultiOrders(ordersList, ref errorMessage, ref countOfFaild);
                if (result == GlobalErrors.Success)
                {
                    if (countOfFaild == 0)
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(MessagesConstants.Desc_Delete_Successfully, _requestRepository.LanguageId));
                    }
                    else
                    {
                        return JSonResultResponse.GetSuccessrJSon(result, ResourcesManager.TranslateKey(errorMessage, _requestRepository.LanguageId));
                    }
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(string.IsNullOrEmpty(errorMessage) ? MessagesConstants.Desc_Error_Update_Data : errorMessage, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion
        #region VALIDATION
        [HttpPost]
        [ActionName("IsValidOperation")]
        public JsonResult IsValidOperation([FromBody] OrderModel order)
        {
            try
            {
                var operation = new OrderOperationModel();
                var result = _orderService.IsValidOperation(order, ref operation);
                if (result == GlobalErrors.Success || operation.OnlyShowWarning)
                {
                    return JSonResultResponse.GetSuccessrJSon(operation, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                if (result == GlobalErrors.DataAlreadyExists)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CheckForPostedCustomerTransactions")]
        public JsonResult CheckForPostedCustomerTransactions([FromBody] OrderModel order)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Happend, _requestRepository.LanguageId));
                }
                bool showWaitForUpdateWarning = false;
                result = _orderService.CheckForPostedCustomerTransactions(order, ref showWaitForUpdateWarning);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessrJSon(showWaitForUpdateWarning, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("IsValidToConfirm")]
        public JsonResult IsValidToConfirm([FromBody] List<OrderModel> orders)
        {
            try
            {
                List<OrderModel> validOrdersToApprove = new List<OrderModel>();
                List<OrderModel> invalidOrdersToApprove = new List<OrderModel>();
                string message = string.Empty;
                GlobalErrors result = _orderService.IsValidToApprove(orders, ref validOrdersToApprove, ref invalidOrdersToApprove, ref message);
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        validOrdersToApprove = validOrdersToApprove,
                        invalidOrdersToApprove = invalidOrdersToApprove,
                        message = message,
                    };
                    return JSonResultResponse.GetSuccessrJSon(data, MessagesConstants.Desc_Success_Getting_Data);
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }
        #endregion
    }
}
