
using DinkToPdf;
using DinkToPdf.Contracts;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.SpaServices.AngularCli;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.FileProviders;
using Microsoft.Extensions.Hosting;
using Microsoft.IdentityModel.Tokens;
using Sonic.Web.DAL;
using Sonic.Web.Licensing;
using Sonic.Web.Licensing.Licensing;
using Sonic.Web.Model;
using Sonic.Web.Service;
using System;
using System.IO;
using System.Text;
using Sonic.Web.PL.PDFUtility;
using static Sonic.Web.Model.CoreDataBaseConstants;
using Microsoft.AspNetCore.Mvc.Razor.RuntimeCompilation;
using Sonic.Web.PL.Controllers;
using InCube.Security;
using System.Linq;
using System.Threading.Tasks;

namespace Sonic.Web.PL
{
    public class Startup
    {
        private IWebHostEnvironment _env;
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
        {

            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }
            else
            {
                app.UseExceptionHandler("/API/ErrorController");
            }
            app.UseRouting();
            app.UseCors("CorsPolicy");

            app.UseEndpoints(endpoints =>
            {
                endpoints.MapHub<NotifyHub>("/notify");
                endpoints.MapHub<MessageHub>("/MessageHub");
            });
            app.UseAuthentication();
            app.UseStaticFiles();
            app.UseStaticFiles(new StaticFileOptions()
            {
                FileProvider = new PhysicalFileProvider(Path.Combine(Directory.GetCurrentDirectory(), "Resources")),
                RequestPath = new PathString("/Resources")
            });
            app.UseStaticFiles(new StaticFileOptions()
            {
                FileProvider = new PhysicalFileProvider(Path.Combine(Directory.GetCurrentDirectory(), "wwwroot")),
                RequestPath = new PathString("/wwwroot")
            });
            app.UseSpaStaticFiles();

            app.UseSession();
            app.UseMvc(routes =>
            {
                routes.MapRoute(
                    name: "default",
                    template: "{controller}/{action=Index}/{id?}");
            });
            app.Use((context, next) =>
            {
                if (context.Request.Path.StartsWithSegments(new PathString("/api")))
                {
                    context.Response.StatusCode = StatusCodes.Status404NotFound;
                    var bytes = Encoding.UTF8.GetBytes("Invalid request, please check your request and try again.");

                    context.Response.Body.WriteAsync(bytes, 0, bytes.Length);
                    return Task.CompletedTask;
                }

                return next();
            });
            app.UseSpa(spa =>
            {
                spa.Options.SourcePath = "ClientApp";

                if (env.IsDevelopment())
                {
                    spa.UseProxyToSpaDevelopmentServer("http://localhost:4200");
                    //  spa.UseAngularCliServer(npmScript: "start --app=app1 --base-href=/app1/ --serve-path=/");
                }
            });
            //Create Banners Folder
            string rootPath = env.ContentRootPath;
            string echoFolderName = "wwwroot\\Echo";
            string echoPath = Path.Combine(rootPath, echoFolderName);
            if (!Directory.Exists(echoPath))
                Directory.CreateDirectory(echoPath);

            string bannerFoldername = "wwwroot\\Echo\\banners";
            string bannersPath = Path.Combine(rootPath, bannerFoldername);
            if (!Directory.Exists(bannersPath))
                Directory.CreateDirectory(bannersPath);
            //// Read Configuration
            //SystemConfigService.ReadConfiguration(rootPath);
            //// Read Languages
            //SettingService.WriteJSONs();
        }
        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddControllersWithViews();
            services.AddRazorTemplating();
            services.AddSignalR();
            string securityKey = SessionsConstants.SecurityKeyWeb;
            var symmetricSecurityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(securityKey));
            services.AddSession(options =>
            {
                options.IdleTimeout = TimeSpan.FromHours(SessionsConstants.TokenExpiredAfterHoursWeb);//You can set Time
            });

            services.AddSpaStaticFiles(configuration => configuration.RootPath = $"ClientApp/dist");

            LocalUtilities.AllowedOriginsString = Configuration["AllowedOrigins"] != null ? Configuration["AllowedOrigins"] : "";
            if (!string.IsNullOrEmpty(LocalUtilities.AllowedOriginsString))
            {
                string [] arr = LocalUtilities.AllowedOriginsString.Split(',');
                if(arr != null && arr.Length > 0)
                {
                    LocalUtilities.AllowedOriginsList = arr.ToList();
                }
            }

            var AllowedMethodsList = new string[] {
                     "GET",
                     "POST",
                     "DELETE",
                     "OPTIONS" };
            var AllowedHeadersList = new string[] {
                     "X-Request-Token",
                     "Accept",
                     "Content-Type",
                     "Authorization" };
            services.AddCors(options =>
            {
                if (LocalUtilities.AllowedOriginsString.Equals("*"))
                {
                    options.AddPolicy("CorsPolicy",
                   builder => builder.AllowAnyOrigin());
                }
                else
                {
                    options.AddPolicy("CorsPolicy",
                   builder => builder.WithHeaders(AllowedHeadersList).WithMethods(AllowedMethodsList)
                   .WithOrigins(LocalUtilities.AllowedOriginsString)
                  // .SetIsOriginAllowed((host) => true)
                  .AllowCredentials());
                }
            });
            services.AddMvc(option => option.EnableEndpointRouting = false);
            services.AddAuthentication(JwtBearerDefaults.AuthenticationScheme)
                .AddJwtBearer(options =>
                {
                    options.TokenValidationParameters = new TokenValidationParameters
                    {
                        //what to validate
                        ValidateIssuer = true,
                        ValidateAudience = true,
                        ValidateIssuerSigningKey = true,
                        //setup validate data
                        ValidIssuer = SessionsConstants.ValidIssuerWeb,
                        ValidAudience = SessionsConstants.ValidAudienceWeb,
                        IssuerSigningKey = symmetricSecurityKey
                    };
                });
            services.AddHttpContextAccessor();
            var context = new CustomAssemblyLoadContext();
            context.LoadUnmanagedLibrary(Path.Combine(Directory.GetCurrentDirectory(), "libwkhtmltox.dll"));
            services.AddSingleton(typeof(IConverter), new SynchronizedConverter(new PdfTools()));
            // In production, the Angular files will be served from this directory
            services.AddSpaStaticFiles(configuration =>
            {
                configuration.RootPath = "ClientApp/dist";
            });
            services.AddRazorPages().AddRazorRuntimeCompilation();
            services.Configure<MvcRazorRuntimeCompilationOptions>(options =>
            {
                var libraryPath = Path.GetFullPath(
                    Path.Combine(Directory.GetCurrentDirectory(), "PDFUtility", "Reports"));
                options.FileProviders.Add(new PhysicalFileProvider(libraryPath));
            });
            LocalUtilities.AppVersion = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
            LocalUtilities.WebApplication = WebApplication.Sonic;
            Sonic.Web.Licensing.Core.LicensingLocalUtilities.AppVersion = LocalUtilities.AppVersion;

#if Echo
            LocalUtilities.WebApplication = WebApplication.Echo;
#endif
            SystemConfigService.ReadConfiguration(Directory.GetCurrentDirectory());
            // Read Languages
            SettingService.WriteJSONs();
            // Read LicensingService
            services.AddSingleton<ILicensing, LicensingService>();
            services.AddSingleton<IActiveTokens, ActiveTokens>();
            services.AddScoped<IRequestRepository, RequestRepository>();
            services.AddHostedService<ServiceWorker>();
        }
    }
}